//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "PerspectiveProjection.h"
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

PerspectiveProjection::PerspectiveProjection():
    Projection(),
    m_isMatrixUpToDate(false),
    m_nearZ(1.0F),
    m_farZ(1000.0F),
    m_fovYDegrees(45.0F),
    m_aspectRatio(1.0F)
{
}

PerspectiveProjection::SharedPointer PerspectiveProjection::Create()
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

    PerspectiveProjection* ptr = FEATSTD_NEW(PerspectiveProjection)();
    if (ptr == 0) {
        FEATSTD_LOG_ERROR("Perspective projection create failed, out of memory.");
    }
    PerspectiveProjection::SharedPointer sharedPointer(ptr);
    return sharedPointer;
}

PerspectiveProjection::~PerspectiveProjection()
{
}

Projection::SharedPointer PerspectiveProjection::Clone() const
{
    return SharedPointer(FEATSTD_NEW(PerspectiveProjection)(*this));
}

PerspectiveProjection::PerspectiveProjection(const PerspectiveProjection& other):
    Projection(other),
    m_isMatrixUpToDate(other.m_isMatrixUpToDate),
    m_nearZ(other.m_nearZ),
    m_farZ(other.m_farZ),
    m_fovYDegrees(other.m_fovYDegrees),
    m_aspectRatio(other.m_aspectRatio)
{
}

FEATSTD_RTTI_DEFINITION(PerspectiveProjection, Projection)

void PerspectiveProjection::SetNearZ(Float nearZ)
{
    m_nearZ = nearZ;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float PerspectiveProjection::GetNearZ() const
{
    return m_nearZ;
}

void PerspectiveProjection::SetFarZ(Float farZ)
{
    m_farZ = farZ;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float PerspectiveProjection::GetFarZ() const
{
    return m_farZ;
}

void PerspectiveProjection::SetFovYDegrees(Float fovYDegrees)
{
    m_fovYDegrees = fovYDegrees;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float PerspectiveProjection::GetFovYDegrees() const
{
    return m_fovYDegrees;
}

void PerspectiveProjection::SetAspectRatio(Float aspectRatio)
{
    m_aspectRatio = aspectRatio;
    m_isMatrixUpToDate = false;
    NotifyListenersOnProjectionChanged();
}

Float PerspectiveProjection::GetAspectRatio() const
{
    return m_aspectRatio;
}

void PerspectiveProjection::UpdateProjectionMatrix() const
{
    if (m_isMatrixUpToDate) {
        return;
    }
    Matrix4 perspectiveProjection;
    perspectiveProjection = Math3D::SetPerspectiveProjection(perspectiveProjection, m_nearZ, m_farZ, m_fovYDegrees, m_aspectRatio);
    SetProjectionMatrix(perspectiveProjection);
    m_isMatrixUpToDate = true;

}
}

