//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(OrthographicProjection_h)
#define OrthographicProjection_h

#include <Candera/Engine3D/Core/Projection.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Macros.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
class Projection;

/**
 * @brief   The class OrthographicProjection represents a form of parallel projection where the view direction is
 *          orthogonal to the projection plane.
 */
class OrthographicProjection : public Projection
{
    FEATSTD_TYPEDEF_BASE(Projection);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(OrthographicProjection);
        /**
         *  Creates an instance of this class.
         *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructs an OrthographicProjection object.
         */
        virtual ~OrthographicProjection() override;

        // Overrides Projection::Clone.
        virtual Projection::SharedPointer Clone() const override;


        FEATSTD_RTTI_DECLARATION();

        /**
         *  Set the near plane of the depth range.
         *  @param nearZ Near plane of depth range to be set.
         */
        void SetNearZ(Float nearZ);

        /**
         *  Retrieve the near plane of the depth range.
         *  @return The near plane of the depth range.
         */
        Float GetNearZ() const;

        /**
         *  Set the far plane of the depth range.
         *  @param farZ Far plane of depth range to be set.
         */
        void SetFarZ(Float farZ);

        /**
         *  Retrieve the far plane of the depth range.
         *  @return The far plane of the depth range.
         */
        Float GetFarZ() const;

        /**
         *  Set width of this orthographic projection object.
         *  @param width Width of this orthographic projection object.
         */
        void SetWidth(Float width);

        /**
         *  Retrieve the width of this orthographic projection object.
         *  @return The width of this orthographic projection object.
         */
        Float GetWidth() const;

        /**
         *  Set height of this orthographic projection object.
         *  @param height Height of this orthographic projection object.
         */
        void SetHeight(Float height);

        /**
         *  Retrieve the height of this orthographic projection object.
         *  @return The height of this orthographic projection object.
         */
        Float GetHeight() const;

    protected:
        /**
         *  Override of class Projection.
         */
        virtual void UpdateProjectionMatrix() const override;

    private:
        mutable bool m_isMatrixUpToDate;    // internaly used matrix cache. $CACHE
        Float m_nearZ;    // near plane of the depth range.
        Float m_farZ;     // far plane of the depth range.
        Float m_width;    // width of this projection.
        Float m_height;   // height of this projection.

        // private because only ::Create() should be used to create an instance of this class
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::OrthographicProjection::OrthographicProjection, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        OrthographicProjection();
        OrthographicProjection(const OrthographicProjection& other);
        OrthographicProjection& operator=(const OrthographicProjection& other);
};
 
/** @} */ // end of Core3D
 
} // namespace Candera

#endif

