//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_OcclusionCullingCameraRenderStrategy_h
#define Candera_OcclusionCullingCameraRenderStrategy_h

#include <Candera/Engine3D/Core/CameraRenderStrategy.h>
#include <Candera/Engine3D/Core/QueryProperty.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

class Node;

/**
 * @brief The class OcclusionCullingCameraRenderStrategy is an implementation of the abstract class CameraRenderStrategy.
 * This render strategy issues occlusion queries when rendering the nodes that have been initialized by the strategy. If a query result from
 * the previous frame is available, the node is either deemed visible and rendered, or deemed occluded and its bounding box is rendered
 * invisibly instead. Either case issues a new query to determine visibility for the next frame. Nodes that do no have a QueryProperty
 * are rendered as usual.
 * This camera render strategy only reaches its full potential when nodes were sorted front to back first.
 * A performance increase by using occlusion culling can only be expected when objects occlude each other, and occluded objects consist
 * of lots of vertices, have expensive fragment shaders, or both.
 */
class OcclusionCullingCameraRenderStrategy : public CameraRenderStrategy
{
    FEATSTD_TYPEDEF_BASE(CameraRenderStrategy);

    public:

        /**
         *  Constructor
         */
        OcclusionCullingCameraRenderStrategy();

        /**
         *  Destructor
         */
        virtual ~OcclusionCullingCameraRenderStrategy() override;

        /**
         *  Initialize the rootNode's subtree for use by this render strategy.
         *  @param rootNode   The root node of the subtree that will be initialized for use by this render strategy.
         *  @param queryType  The type of queries that are issued by this render strategy. This provides a hint to the
         *                    driver to either use an accurate method (QueryAnySamplesPassed), or a faster
         *                    approximation (QueryAnySamplesPassedConservative) for determining occlusion.
         */
        void Initialize(Node* rootNode, Query::Type queryType = Query::QueryAnySamplesPassedConservative) const;

        /**
         *  Reset the rootNode's subtree. This removes any association of this strategy with nodes, and must be executed before
         *  destroying the strategy. Nodes that have been initialized will be automatically reset if they are removed from their
         *  parent.
         *  @param rootNode  The root node of the subtree that will be reset.
         */
        void Reset(Node* rootNode) const;

        /**
         *  Gets the number of occluded nodes since the last render pass began.
         *  @return The number of occluded nodes since the last render pass.
         */
        UInt GetOccludedNodesCount() const { return m_occludedNodesCount; }

        /**
         *  Gets the number of non-occluded (i.e. visible) nodes since the last render pass began.
         *  @return The number of visible nodes since the last render pass.
         */
        UInt GetVisibleNodesCount() const { return m_visibleNodesCount; }

        FEATSTD_RTTI_DECLARATION();

    protected:

        /**
         *  Called by Renderer at the beginning of each Camera's render pass, resetting the statistics counters
         */
        virtual void OnRenderPassBegins() override { m_occludedNodesCount = 0; m_visibleNodesCount = 0; }

        /**
         *  Tells the Renderer if the given node should be rendered or not.
         *  If the node is deemed to be occluded, the node's bounding box is rendered as an invisible cube to predict
         *  visibility for the next frame, and RenderPassAction::SkipNode is returned.
         *  @param  nextNode  Next node to be rendered. Based on this node the returned RenderPassAction is chosen.
         *  @return           RenderPassAction::ProceedRenderPass when the node shall be rendered,
         *                    RenderPassAction::SkipNode when the node shall not be rendered.
         */
        virtual RenderPassAction GetRenderPassAction(const Node* nextNode) override;

        /**
         *  Called by Renderer after a Camera's render pass has finished. This ends the last query.
         */
        virtual void SetRenderPassCompleted(bool complete) override;

    private:
        // Ends the previous query. Called by GetRenderPassAction() and SetRenderPassCompleted().
        void EndPreviousQuery();

        const Internal::QueryProperty::QueryInfo* m_previousQuerySubmitted;

        UInt m_occludedNodesCount;
        UInt m_visibleNodesCount;
};

/** @} */ // end of Core3D

} // namespace Candera

#endif
