//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "OcclusionCullingCameraRenderStrategy.h"

#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>

namespace Candera
{

using namespace Internal;

FEATSTD_RTTI_DEFINITION(OcclusionCullingCameraRenderStrategy, CameraRenderStrategy)

/**
 *  QueryPropertyTraverser invokes QueryProperty::Set at the Node given and all its descendants.
 */
class QueryPropertyTraverser : public TreeTraverser {
    typedef TreeTraverser Base;

public:
    QueryPropertyTraverser(const OcclusionCullingCameraRenderStrategy* occlusionCullingCameraRenderStrategy, Query::Type queryType, bool cleanup = false)
        :
        Base(),
        m_occlusionCullingCameraRenderStrategy(occlusionCullingCameraRenderStrategy),
        m_queryType(queryType),
        m_cleanup(cleanup) {}
    ~QueryPropertyTraverser() {}

protected:
    virtual TraverserAction ProcessNode(Node& node) override
    {
        if (m_cleanup) {
            static_cast<void>(QueryProperty::Remove(node, m_occlusionCullingCameraRenderStrategy));
        }
        else {
            if (!QueryProperty::Set(node, m_occlusionCullingCameraRenderStrategy, m_queryType)) {
                return StopTraversing;
            }
        }

        return ProceedTraversing;
    }

private:
    const OcclusionCullingCameraRenderStrategy* m_occlusionCullingCameraRenderStrategy;
    Query::Type m_queryType;
    bool m_cleanup;

    FEATSTD_MAKE_CLASS_UNCOPYABLE(QueryPropertyTraverser);
};

OcclusionCullingCameraRenderStrategy::OcclusionCullingCameraRenderStrategy()
    :
    m_previousQuerySubmitted(0),
    m_occludedNodesCount(0),
    m_visibleNodesCount(0)
{
}

OcclusionCullingCameraRenderStrategy::~OcclusionCullingCameraRenderStrategy()
{
    FEATSTD_DEBUG_ASSERT(0 == m_previousQuerySubmitted);
}


void OcclusionCullingCameraRenderStrategy::Initialize(Node* rootNode, Query::Type queryType) const
{
    if (0 == rootNode) {
        return;
    }

    QueryPropertyTraverser queryPropertyTraverser(this, queryType);
    queryPropertyTraverser.Traverse(*rootNode);
}

void OcclusionCullingCameraRenderStrategy::Reset(Node* rootNode) const
{
    if (0 == rootNode) {
        return;
    }

    QueryPropertyTraverser queryPropertyTraverser(this, Query::QueryAnySamplesPassedConservative, true);
    queryPropertyTraverser.Traverse(*rootNode);
}

CameraRenderStrategy::RenderPassAction OcclusionCullingCameraRenderStrategy::GetRenderPassAction(const Node* nextNode)
{
    EndPreviousQuery();

    QueryProperty* queryProperty = QueryProperty::Get(*nextNode);
    if (0 == queryProperty) {
        return CameraRenderStrategy::ProceedRenderPass;
    }

    QueryProperty::QueryInfo* queryInfo = queryProperty->GetQueryInfo(this);
    if (0 == queryInfo)
    {
        // The queryProperty is not assigned to this OcclusionCullingCameraRenderStrategy, but an other one.
        return CameraRenderStrategy::ProceedRenderPass;
    }

    // if there is no result available yet for whatever reason, assume the object is visible
    UInt32 queryResult = 0xFFffFFff;
    const bool isResultAvailable = queryInfo->IsResultAvailable();
    if (isResultAvailable) {
        queryResult = queryInfo->GetResult();
    }

    if (queryInfo->Begin()) {
        m_previousQuerySubmitted = queryInfo;
    }

    if (queryResult > 0) {
        m_visibleNodesCount++;
        return CameraRenderStrategy::ProceedRenderPass;
    }

    queryProperty->RenderBoundingBox(nextNode);
    m_occludedNodesCount++;

    return CameraRenderStrategy::SkipNode;
}

void OcclusionCullingCameraRenderStrategy::SetRenderPassCompleted(bool complete)
{
    CameraRenderStrategy::SetRenderPassCompleted(complete);

    EndPreviousQuery();

    const Renderer::Statistics& constStatistics = Renderer::GetStatistics();
    Renderer::Statistics& statistics = const_cast<Renderer::Statistics&>(constStatistics);
    statistics.DrawCallsVisible += GetVisibleNodesCount();
    statistics.DrawCallsInvisible += GetOccludedNodesCount();
}

void OcclusionCullingCameraRenderStrategy::EndPreviousQuery()
{
    if (0 != m_previousQuerySubmitted) {
        static_cast<void>(m_previousQuerySubmitted->End());
        m_previousQuerySubmitted = 0;
    }
}

} // namespace Candera
