//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "NodeRenderSequence.h"
#include <Candera/Engine3D/ShaderParamSetters/AbstractShaderParamSetter.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace MemoryManagement;
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

void NodeRenderSequence::Render(Node& node, const SharedPointer<VertexBuffer>& vertexBuffer)
{
    for (SharedPointer<Appearance> appearance = node.GetAppearance(); appearance != 0; appearance = appearance->GetNextPass()) {
        if (Activate(node, vertexBuffer, appearance)) {
            vertexBuffer->Render();
        }
    }
}

bool NodeRenderSequence::Activate(Node& node, const SharedPointer<VertexBuffer>& vertexBuffer, const SharedPointer<Appearance>& appearance)
{
    if (appearance == 0) {
        FEATSTD_LOG_ERROR("NodeRenderSequence failed as appearance is null for Node:\"%s\".", (node.GetName() == 0) ? "" : node.GetName());
        return false;
    }

    SharedPointer<Shader> shader = appearance->GetShader();
    if (shader == 0) {
        FEATSTD_LOG_ERROR("NodeRenderSequence failed as shader is null for Node:\"%s\".", (node.GetName() == 0) ? "" : node.GetName());

        return false;
    }

    if (!appearance->Activate(&node)) {
        FEATSTD_LOG_ERROR("Activation of Appearance failed for Node:\"%s\".", (node.GetName() == 0) ? "" : node.GetName());
        return false;
    }

    if (!BindAttributes(shader, vertexBuffer)) {
        FEATSTD_LOG_ERROR("Binding Attributes failed for Node:\"%s\".", (node.GetName() == 0) ? "" : node.GetName());
        return false;
    }

    SharedPointer<AbstractShaderParamSetter> shaderParams = appearance->GetShaderParamSetter();
    // Activate Node's ShaderParamSetter object if set, otherwise the default ShaderParamSetter object.
    if (shaderParams != 0) {
        if (!shaderParams->Activate(node, appearance)) {
            FEATSTD_LOG_ERROR("Activation of ShaderParamSetter failed for Node:\"%s\".", (node.GetName() == 0) ? "" : node.GetName());
            return false;
        }
    }
    else {
        if ((m_defaultShaderParams == 0) || (!m_defaultShaderParams->Activate(node, appearance))) {
            FEATSTD_LOG_ERROR("Activation of default ShaderParamSetter failed for Node:\"%s\".", (node.GetName() == 0) ? "" : node.GetName());
            return false;
        }
    }

    return true;
}

CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::NodeRenderSequence::BindAttributes, CANDERA_LINT_REASON_NONCONSTMETHOD)
bool NodeRenderSequence::BindAttributes(const SharedPointer<Shader>& shader, const SharedPointer<VertexBuffer>& vertexBuffer)
{
    if (vertexBuffer == 0) {
        FEATSTD_LOG_ERROR("Bind attributes failed, VertexBuffer is null.");
        return false;
    }

    if (!vertexBuffer->Activate()) {
        FEATSTD_LOG_ERROR("Activation of VertexBuffer failed.");
        return false;
    }

    if (!shader->BindAttributes(vertexBuffer)) {
        FEATSTD_LOG_ERROR("Shader::BindAttributes failed.");
        return false;
    }

    return true;
}
} // namespace Candera
