//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_NODE_H)
#define CANDERA_NODE_H

#include <Candera/Engine3D/Core/3DStringBufferAppenders.h>
#include <Candera/Engine3D/Core/Transformable.h>
#include <Candera/Engine3D/RenderOrder/OrderCriterionValue.h>
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/EngineBase/Common/ScopeMask.h>
#include <Candera/System/Container/LinkedList.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

    //forward declarations
    class Group;
    class Renderer;
    class Camera;
    class Scene;
    class OnAncestorAddedTraverser;
    class OrderCriterion;
    class NodeListener;
    class PickTraverser;
    class Layouter;

    namespace Monitor {
        class RenderBenchmarkMeasurement;
    }

    /**
     * @brief   The class Node is an abstract base class for all scene graph nodes.
     *          Each Node defines a local coordinate system relative to the coordinate system of the parent Node.
     *          The functionality to define the local coordinate system is derived from Transformable and consists of
     *          following components: translation, rotation, scale and a generic matrix called transform matrix.
     *          If the Node is transformed from the local coordinate system to the world's coordinate system, then
     *          the Node's local transformation is multiplied with all its parents transformations.
     *          A Node can also store a set of Nodes as its children but a Node can at most have one parent at a time.
     *          Cycles are prohibited. Changing the Nodes alpha value affects also the children Nodes.
     */
    class Node: public Transformable
    {
        FEATSTD_TYPEDEF_BASE(Transformable);

        friend class CameraRenderStrategy;
        friend class Renderer;
        friend class RenderOrder;
        friend class OrderCriterion;
        friend class BatchOrderCriterion;
        friend class OnAncestorAddedTraverser;
        friend class OnAncestorRemovedTraverser;
        friend class InvalidateWorldTransformCacheTraverser;
        friend class InvalidateEffectiveAlphaDownwardsTraverser;
        friend class Monitor::RenderBenchmarkMeasurement;
        friend class PickTraverser;


    public:

        /**
         *  Traversing specifies the method used to traverse attached descendants starting from this Node.
         */
        enum Traversing {
            Flat,       ///< Only this Node.
            Deep        ///< This Node and all its descendants.
        };

        /**
         *  Provides a shallow clone of this node. The new node is not attached
         *  to any node graph, and any other naked pointers, such as strategies
         *  or listeners are not liked to the clone. Strings (objects or type
         *  const Char*) are the only exception from this rule.
         *  @return               The cloned Node if successful, null otherwise.
         */
        virtual Node* Clone() const = 0;

        /**
         *  Disposes this node and all descendant children Nodes if any.
         */
        void Dispose();

        /**
         *  Retrieves the scene graph parent of this Node
         *  @return  Pointer to parent Node, or null if there is no parent available
         */
        Node* GetParent() const { return m_parent; }

        /**
         *  Retrieve the Scene Node that is ancestor to this Node.
         *  @return Scene Node if available, 0 otherwise.
         */
        Scene* GetScene() const;

        /**
         *  Defines the local setting if rendering shall be enabled or not.
         *  This boolean value influences this Node's descendants. See function "IsEffectiveRenderingEnabled" for further details.
         *  @param enable Enables(true) / Disables(false) rendering of this node.
         */
        void SetRenderingEnabled(bool enable) {m_isRenderingEnabled = enable; }

        /**
         *  Retrieve this Node's local setting, if rendering is enabled or disabled.
         *  To compute if effective rendering is enabled in dependency to this Node's ancestors, see
         *  function "IsEffectiveRenderingEnabled" for further details.
         *  @return Whether rendering of this node is enabled (true) or disabled (false).
         */
        bool IsRenderingEnabled() const { return m_isRenderingEnabled; }

        /**
         *  Retrieve whether rendering is enabled or not in dependency to this Node's ancestors settings.
         *  @return true if this Node AND all ancestor Nodes have set rendering enabled.
         *  @return false if either this Node OR any ancestor Node have set rendering disabled.
         */
        bool IsEffectiveRenderingEnabled() const;

        /**
         *  Set Appearance associated to this Node.
         *  @param appearance The appearance to be set to this Node.
         */
        void SetAppearance(MemoryManagement::SharedPointer<Appearance> appearance) { m_appearance = appearance; }

        /**
         *  Retrieves the appearance associated to this Node.
         *  @return The appearance associated to this Node.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Node::GetAppearance, CANDERA_LINT_REASON_NONCONSTMETHOD)
        MemoryManagement::SharedPointer<Appearance> GetAppearance() const { return m_appearance; }

        /**
         *  Sets the local alpha value for this Node.
         *  @param    alphaValue  Values allowed between [0.0F, 1.0F] at which 0.0F means fully transparent (invisible), 1.0F means fully opaque.
         *  This can be used to define translucency of mesh groups.
         *  It is the responsibility of Node specializations to actually use the alpha value; also, see GetEffectiveAlphaValue().
         */
        void SetAlphaValue(Float alphaValue);

        /**
         *  Gets the local alpha value for this Node.
         *  @return   The alpha value of this Node: [0.0F, 1.0F]; 0.0F means fully transparent (invisible), 1.0F means fully opaque.
         *  The alpha value returned is the local alpha value and therefore not multiplied with the alpha values of any ancestor to this Node.
         */
        Float GetAlphaValue() const { return m_alphaValue; }

        /**
         *  Gets the product of the alpha values of this Node and all ancestors.
         *  @return Product of the alpha values of this Node and all ancestors.
         */
        Float GetEffectiveAlphaValue() const;

        /**
         *  Retrieves the world transformation matrix of this Node
         *  Multiplies local transformation with transformations of all ancestors to this Node.
         *  As internal cache is used, recalculation of world transform occurs only, if local transform or any transform of an ancestor to this Node changes.
         *  @return World transformation matrix of this Node.
         */
        virtual const Matrix4& GetWorldTransform() const;

        /**
         *  Set the world transformation matrix of this Node.
         *  @param worldTransform  The world transformation matrix to set.
         */
         void SetWorldTransform(const Matrix4& worldTransform);

        /**
         *  Retrieves the world rotation matrix of this Node
         *  Multiplies local rotation with rotations of all ancestors to this Node.
         *  @return World rotation matrix of this Node.
         */
        Matrix4 GetWorldRotation() const;

        /**
         *  Retrieves the world scale matrix of this Node
         *  Multiplies local scale with scales of all ancestors to this Node.
         *  @return World scale matrix of this Node.
         */
        Matrix4 GetWorldScale() const;

        /**
         *  Retrieves the world transform position
         *  Multiplies the origin of the cartesian coordinate system (x=0, y=0, z=0) with the world transformation matrix of this Node
         *  @return Transformed position of this Node in world space.
         */
        Vector3 GetWorldPosition() const;

        /**
         *  Set this objects center in object space.
         *  @param center in object space.
         */
        void SetCenter(const Vector3& center) { m_center = center; InvalidateWorldCenterCache(); }

        /**
         *  Retrieves this object's center in object space.
         *  @return center in object space.
         */
        const Vector3& GetCenter() const { return m_center; }

        /**
         *  Retrieves the transformed center of this object's bounding sphere in world space.
         *  @return center in world space.
         */
        Vector3 GetWorldCenter() const;

        /**
         *  Set the radius of the object bounding sphere in object space.
         *  @param radius in object space.
         */
        void SetRadius(const Float radius) { m_radius = radius; InvalidateWorldRadiusCache(); }

        /**
         *  Retrieves the radius of this object's bounding sphere in object space.
         *  @return radius in object space.
         */
        Float GetRadius() const { return m_radius; }

        /**
         *  Retrieves the radius of the object's bounding sphere in world space, thus is multiplied with the world scale.
         *  @return Radius of the object's bounding sphere in world space, thus is multiplied with the world scale.
         */
        Float GetWorldRadius() const;

        /**
         *  Compute a bounding sphere represented by this Node�s center and radius in object coordinate space.
         *  @return true if bounding sphere was computed successfully and false if not because of missing geometry.
         */
        virtual bool ComputeBoundingSphere();

        /**
         *  Returns if bounding sphere of this node object is valid or not. A bounding sphere can be invalid if it has
         *  not been set or computed (see ComputeBoundingSphere).
         *  @return true if bounding sphere is valid, false otherwise.
         */
        bool IsBoundingSphereValid() const {  return m_radius > 0.0F; }

        /**
         *  Set axis-aligned bounding box (AABB) in object coordinate space explicitly.
         *  @param minBounds describing the returned lower-left-back vertex of the bounding box.
         *  @param maxBounds describing the returned upper-right-front vertex of the bounding box.
         */
        void SetBoundingBox(const Vector3& minBounds, const Vector3& maxBounds);

        /**
         *  Get axis-aligned bounding box (AABB) in object coordinate space.
         *  @param minBounds is an out parameter describing the returned lower-left-back vertex of the bounding box.
         *  @param maxBounds is an out parameter describing the returned upper-right-front vertex of the bounding box.
         */
        void GetBoundingBox(Vector3& minBounds, Vector3& maxBounds) const;

        /**
         *  Computes an axis-aligned bounding box in object coordinate space.
         *  If AABB computation is possible, the values will be set to the node and
         *  can be retrieved with GetBoundingBox().
         *  Computation will only be done for the current node and will not include child nodes.
         *  @return true if bounding box was computed successfully and false if not because of missing geometry.
         */
        bool ComputeBoundingBox();

        /**
         *  Returns if bounding box of this node object is valid or not. A bounding box can be invalid if it has
         *  not been set or computed (see ComputeBoundingSphere).
         *  @return true if bounding box is valid, false otherwise.
         */
        virtual bool IsBoundingBoxValid() const;

        /**
         *  Get oriented bounding box (OBB) in world coordinate space.
         *  The OBB will only be returned for the current node and will not include child nodes.
         *  @param wobbBounds is an out parameter containing the 8 vertices of the bounding box in world space.
         */
        void GetWorldOrientedBoundingBox(Vector3 wobbBounds[8]) const;

        /**
         *  Get axis-aligned bounding box (AABB) in world coordinate space.
         *  @param minBounds is an out parameter describing the returned lower-left-back vertex of the bounding box.
         *  @param maxBounds is an out parameter describing the returned upper-right-front vertex of the bounding box.
         *  @param traverse Whether only this Node (Flat, default) or all child nodes (Deep) shall be considered.
         */
        void GetWorldAxisAlignedBoundingBox(Vector3& minBounds, Vector3& maxBounds, Traversing traverse = Flat) const;

        /**
         *  Set the name of the render order bin to which this Node is assigned to.
         *  If a Node is not explicitly assigned to a particular render order bin (name is null), then Candera will assign
         *  this Node into either the "opaque" or the "transparent" bin, according to whether it has transparency enabled or not.
         *  (For details how to enable or disable blending, see RenderMode::SetBlendingEnabled).
         *  If a Node is assigned to the "opaque" or "transparent" bin explicitly, the automatic bin assignment is disabled.
         *
         *  @param binName    pointer to a null terminated character array that must not be deleted during runtime.
         *                    Default value for bin name is null, that means Node is assigned to render order bins by Candera, automatically.
         */
        void SetRenderOrderBinAssignment(const Char* binName);

        /**
         *  Retrieves the name of the render order bin to which this Node is assigned to.
         *  If a Node is not explicitly assigned to a particular render order bin (name is null), then Candera will assign
         *  this Node into either the "opaque" or the "transparent" bin, according to whether it has transparency enabled or not.
         *  (For details how to enable or disable blending, see RenderMode::SetBlendingEnabled).
         *  If a Node is assigned to the "opaque" or "transparent" bin explicitly, the automatic bin assignment is disabled.
         *  @return The name of the render order bin to which this Node is assigned to.
         */
        const Char* GetRenderOrderBinAssignment() const { return m_binName; }

        /**
         *  Retrieves the hash of the name of the render order bin to which this Node is assigned to.
         *  @return The hash of the name of the render order bin to which this Node is assigned to.
         */
        UInt32 GetRenderOrderBinAssignmentHash() const { return m_binNameHash; }

        /**
         *  Set render order rank to define an explicit render order for this Node within the assigned render order bin
         *  (For details see Node::SetRenderOrderBinAssignment).
         *  The rank specifies a comparison parameter that is evaluated, if the Node is attached to a render order bin that is sorted by rank.
         *  Note: The Node's rank is meaningless, if the Node is attached to a render order bin that is not sorted
         *  by rank but by other parameters, like for instance the "opaque" and "transparent" render order bins per default.
         *  Default value for render order rank is 0.
         *  @param rank Render oder rank to be set. Used for defining an explicit render order for this Node within the assigned render order bin.
         */
        void SetRenderOrderRank(Int32 rank) { m_renderOrderRank = rank; }

        /**
         *  Retrieves the render order rank that defines an explicit render order for this Node within the assigned render order bin
         *  (For details see Node::SetRenderOrderBinAssignment).
         *  The rank specifies a comparison parameter that is evaluated, if the Node is attached to a render order bin that is sorted by rank.
         *  Note: The Node's rank is meaningless, if the Node is attached to a render order bin that is not sorted
         *  by rank but by other parameters, like for instance the "opaque" and "transparent" render order bins per default.
         *  Default value for render order rank is 0.
         *  @return The render order rank.
         */
        Int32 GetRenderOrderRank() const { return m_renderOrderRank; }

        /**
         *  Set the render measure value of this Node, which is a benchmark related to rendering time, typically calculated by
         *  external tool and utilized by a camera render strategy (see class CameraRenderStrategy).
         *  @param benchmark Render measure value of this Node to be set.
         */
        void SetRenderBenchmark(Float benchmark) { m_renderBenchmark = benchmark; }

        /**
         *  Retrieves the render measure value of this Node, which is a benchmark related to rendering time, typically calculated by
         *  external tool and utilized by a camera render strategy (see class CameraRenderStrategy).
         *  @return The render measure value of this Node.
         */
        Float GetRenderBenchmark() const { return m_renderBenchmark; }

        /**
         *  Retrieve this Node's local setting, if an intersection test can be done or not.
         *  To compute if effective rendering is enabled in dependency to this Node's ancestors, see
         *  function "IsEffectiveIntersectionTestEnabled" for further details.
         *  @return If intersection tests are enabled (true) or not (false);
         */
        bool IsIntersectionTestEnabled() const { return m_isIntersectionTestEnabled; }

        /**
         *  Retrieve whether intersection test is enabled or not in dependency to this Node's ancestors settings.
         *  @return true if this Node AND all ancestor Nodes have set intersection test enabled.
         *  @return false if either this Node OR any ancestor Node have set intersection test disabled.
         */
        bool IsEffectiveIntersectionTestEnabled() const;

        /**
         *  Defines the local setting if this node should be intersectable or not.
         *  This boolean value influences this Node's descendants.
         *  See function "IsEffectiveIntersectionTestEnabled" for further details.
         *  @param enable Sets if this node should be intersectable(true) or not(false).
         */
        void SetIntersectionTestEnabled(bool enable) { m_isIntersectionTestEnabled = enable; }

        /**
         *  Tests if the bounding sphere of this node intersects with the given Line.
         *  @param line               The intersection Line.
         *  @param distance [out]     The distance from the bounding sphere to the starting point of the Line.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this node, otherwise false.
         */
        virtual bool IsLineIntersectingBoundingSphere(const Line& line, Float& distance) const;

        /**
         *  Tests if geometry of this node intersects with the given Line.
         *  @param line               The intersection Line.
         *  @param distance [out]     The distance from the nearest geometry to the starting point of the Line.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this node, otherwise false.
         */
        virtual bool IsLineIntersectingGeometry(const Line& line, Float& distance) const;

        /**
         *  Tests if geometry of this node intersects with a pick in screen coordinates.
         *  @remark                   This interface is usually used for object selection in scene editors.
         *  @param camera             The view camera. (i.e. the current camera)
         *  @param x                  The x value as a screen coordinate in pixels starting from top-left.
         *  @param y                  The y value as a screen coordinate in pixels starting from top-left.
         *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
         *                            If no intersection is found then this param stays unchanged.
         *  @param traverse           Whether picking test affects only this Node (Flat, default) or all child nodes (Deep).
         *                            If traverse is Deep then distance is the smallest distance to the cameras near plane
         *                            from this node and all child nodes.
         *  @return                   True if an intersection is found with this node, otherwise false.
         */
        bool IsPickIntersectingGeometry(const Camera& camera, Int x, Int y, Float& distance /*out*/, Traversing traverse = Flat) const;

        /**
         *  Adds a Node as child to this Node. The position at which the Node is inserted is undefined.
         *  @param   child: The Node to add to this Node must not form a loop.
         *  @return  true if the adding has been successful. false is
         *           returned if child is null, the child is this Node, or the child already has a parent,
         *           or if the Node is created without a NodeContainer
         *           and therefore cannot add children.
         */
        bool AddChild(Node* child) { return InsertBeforeChild(child, 0); }

        /**
        *  Adds a Node as child to this node. The position will be before the child 'beforeChild'.
        *  @param  child       The Node2D to add to this node must not form a loop.
        *  @param  beforeChild newChild will be added before beforeChild in the list of child nodes.
        *  @return             true if the adding has been successful. false is returned if child is null,
        *                      the child is this Node, or the child already has a parent,
        *                      or if the Node is created without a NodeContainer and therefore cannot add children.
        */
        bool InsertBeforeChild(Node* child, Node* beforeChild);

        /**
         *  Removes the child.
         *  @param child Child node to be removed.
         *  @return true if successfully removed, false if node given is 0 or no child of this Node object.
         */
        bool RemoveChild(Node* child);

        /**
         *  Removes all children.
         *  Just like RemoveChild, any child node is only removed, but not discarded.
         */
        void RemoveAllChildren();

        /**
         *  Retrieves a descendant Node by name.
         *  A depth search explores all Nodes and descendants to this Node.
         *  In other words, a Node can be found that is either a child of this Node or any other Node that is descendant to this Node.
         *  @param  name   Identifier of Node.
         *  @return        The first Node found with the name given. Returns null if no attached Node matches the name given.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Node::GetDescendant, CANDERA_LINT_REASON_NONCONSTMETHOD)
        Node* GetDescendant(const Char* name);

        /**
         *  Retrieves a descendant Node by name.
         *  A depth search explores all Nodes and descendants to this Node.
         *  In other words, a Node can be found that is either a child of this Node or any other Node that is descendant to this Node.
         *  @param  name   Identifier of Node.
         *  @return        The first Node found with the name given. Returns null if no attached Node matches the name given.
         */
        const Node* GetDescendant(const Char* name) const;

        /**
         *  Gets a child Node by name.
         *  @param   name     Name of the the child Node to get.
         *  @return           The child node with the given name, or null if there is no child node
         *                    with this name or there is no child attached to this Node.
         */
        Node* GetChild(const Char* name);

        /**
         *  Gets a child Node by name.
         *  @param   name     Name of the the child Node to get.
         *  @return           The child node with the given name, or null if there is no child node
         *                    with this name or there is no child attached to this Node.
         */
        const Node* GetChild(const Char* name) const;

        /**
         *  Gets a child Node by id.
         *  @param   id       Id of the the child Node to get.
         *  @return           The child node with the given id, or null if there is no child node
         *                    with this id or there is no child attached to this Node.
         */
        Node* GetChild(Id id);

        /**
         *  Gets a child Node by id.
         *  @param   id       Id of the the child Node to get.
         *  @return           The child node with the given id, or null if there is no child node
         *                    with this id or there is no child attached to this Node.
         */
        const Node* GetChild(Id id) const;

        /**
         *  Retrieves the number of children attached to this Node.
         *  @return  The number of children that are directly attached to this Node.
         */
        SizeType GetChildCount() const;

        /** 
         *  Gets the first child node (0 if no children). Use together with GetNextSibling to iterate like this:
         *  for (Node* child = m_firstChild; child != 0; child = child->GetNextSibling()) {
         *      child->Render();
         *  }
         *  @remark The order of the returned child is undefined.
         *  @return Gets the first child node of this Node, 0 of no children.
         */
        Node* GetFirstChild();

        /** 
         *  Gets the first child node (0 if no children). Use together with GetNextSibling to iterate like this:
         *  for (Node* child = m_firstChild; child != 0; child = child->GetNextSibling()) {
         *      child->Render();
         *  }
         *  @remark The order of the returned child is undefined.
         *  @return Gets the first child node of this Node, 0 of no children.
         */
        const Node* GetFirstChild() const;

        /**
         *  Gets the next sibling (0 if no more siblings).
         *  @remark The order of the returned sibling is undefined.
         *  @return The next sibling (0 if no more siblings).
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Node::GetNextSibling, CANDERA_LINT_REASON_NONCONSTMETHOD)
        Node* GetNextSibling();

        /**
         *  Gets the next sibling (0 if no more siblings).
         *  @remark The order of the returned sibling is undefined.
         *  @return The next sibling (0 if no more siblings).
         */
        const Node* GetNextSibling() const;

        /**
         *  Upload asset data attached to this Node to video memory.
         *  Note: The nodes data is only uploaded into the currently active context.
         *  @return true if upload succeeded and false if upload failed.
         */
        virtual bool Upload() { return true; }

        /**
         *  Upload asset data attached to this Node AND all its descendants.
         *  Can upload to all contexts affected by the current scene.
         *  @param scopeMask  Only Nodes that are in scope of the ScopeMask given will be uploaded.
         *  @param multiContext If multiContext is true then the node and all its descendants are 
         *                      uploaded to all ContextResourcePools belonging to the cameras
         *                      the nodes are in scope of.
         *                      Otherwise the subtree is uploaded to the currently active context only.
         *  @return true if upload succeeded for all Nodes and false if upload failed for any Node.
         */
        bool UploadAll(const ScopeMask& scopeMask = ScopeMask(), bool multiContext = false);

        /**
         *  Unload asset data attached to this Node from video memory.
         *  Note: The nodes data is only unloaded from the currently active context.
         *  @return true if unload succeeded and false if unload failed.
         */
        virtual bool Unload() { return true; }

        /**
         *  Unload asset data attached to this Node AND all its descendants from video memory.
         *  Can unload from all contexts affected by the current scene.
         *  @param scopeMask  Only Nodes that are in scope of the ScopeMask given will be unloaded.
         *  @param multiContext If multiContext is true then the node and all its descendants are 
         *                      unloaded from all render target contexts belonging to the cameras
         *                      the nodes are in scope of.
         *                      Otherwise the subtree is uploaded to the currently active context only.
         *  @return true if unload succeeded for all Nodes and false if unload failed for any Node.
         */
        bool UnloadAll(const ScopeMask& scopeMask = ScopeMask(), bool multiContext = false);

        /**
         *  Sets ScopeMask that defines all scope memberships of this Node independent from the scene graph hierarchy.
         *  If deep traversing is specified the scope mask is set also of all descendants of this Node.
         *  @param scopeMask specifies the entire scope mask of a Node, thus all scopes a Node belongs to.
         *  @param traverse tells if the scope mask shall be set for this Node only or for all its descendants also.
         */
        void SetScopeMask(const ScopeMask& scopeMask, Traversing traverse = Flat);

        /**
         *  Retrieves ScopeMask of this Node.
         *  @return the ScopeMask defining all scope memberships of this Node.
         */
        const ScopeMask& GetScopeMask() const { return m_scopeMask; }

        /**
         *  Sets dedicated scope of this Node and - if deep traversing is specified - also of all descendants of this Node.
         *  @param scopeIndex specifies the scope to set.
         *  @param enable defines if scope is enabled or not. Thus, if Node belongs to the scope specified or not.
         *  @param traverse tells if the scope shall be set for this Node only or for all its descendants also.
         *  @return true if scope index is within allowed indices (for details, see class ScopeMask), false otherwise.
         */
        bool SetScopeEnabled(UInt scopeIndex, bool enable, Traversing traverse = Flat);

        /**
         *  Retrieves if this Node belongs to a certain scope or not.
         *  @param scopeIndex specifies to scope to interrogate.
         *  @return true if Node belongs to the scope specified and false otherwise.
         */
        bool IsScopeEnabled(UInt scopeIndex) const { return m_scopeMask.IsScopeEnabled(scopeIndex); }

        /**
         *  Retrieves if this Node is in Scope of the scope mask given. Thus, at least one scope of this Node is enabled that is
         *  also enabled in the scope mask passed.
         *  @param scopeMask defines all scopes to test this Node's scopes against.
         *  @return true if this Node is in scope of the scopes defined by the scope mask given, false otherwise.
         */
        bool IsInScopeOf(const ScopeMask& scopeMask) const { return m_scopeMask.OverlapsWith(scopeMask); }

         /**
          *  Determines if this Node fulfills all requirements for rendering. This can be false even if rendering is enabled (See IsRenderingEnabled).
          *  The render precondition is not fulfilled if at least one of following statements is true for any render pass:
          *  - Appearance is either null or no Shader is associated with Appearance.
          *  - The Shader is not uploaded.
          *  - Shader has to be associated and Uploaded also for every attached additional Appearance (in case
          *    of MultiPassAppearances).
          *  @return True if render prerequisites are fulfilled, false otherwise. Virtual, implemented in derived subclasses.
          */
        virtual bool IsRenderPrerequisiteFulfilled() const;


        /**
         *  Retrieves if node given is parented by this Node object.
         *  @param node is tested if it is direct child of this Node object.
         *  @return true if node given is immediate child of this Node object, false otherwise.
         */
        bool IsParentOf(const Node * node) const;

        /**
         *  Adds a NodeListener object.
         *  @param listener The lifetime of the listener is managed by the calling code. Node does not take ownership.
         *  @return True if successful, false otherwise.
         */
        bool AddNodeListener(NodeListener* listener);

        /**
         *  Removes a NodeListener object.
         *  @param listener Specifies the NodeListener object to remove from this Node.
         *  @return True if successful, false otherwise.
         */
        bool RemoveNodeListener(NodeListener* listener);

        /**
         *  Provides the parent for dynamic property scene graph inheritance.
         *  @param node The node, whose parent is sought.
         *  @return The parent of node if available, null otherwise.
         */
        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost *node) {
            return static_cast<const Node*>(node)->GetParent();
        }

#if defined(CANDERA_LAYOUT_ENABLED)
        /**
        *  Sets a layout strategy for the child nodes, e.g. a grid or stack layout.
        *  The attached layouter is Disposed when the node is disposed.
        *  @param layouter Layout strategy for child nodes.
        */
        void SetLayouter(Layouter* layouter);

        /**
        *  Gets the layout strategy for the child nodes.
        *  @return Layouter layout the child nodes.
        */
        Layouter* GetLayouter() const;

        /**
        *  Starts the layout for this node, if a layouter has been assigned with SetLayouter().
        */
        void Layout();
#endif

        FEATSTD_RTTI_DECLARATION();

    protected:
        /**
         *  Constructor
         */
        Node();

        /**
         *  Copies the given Node. The parent is set to zero! A Node must be added to a Group, explicitly!
         *  @param node Node to be copied.
         *  @return The copied node.
         */
        Node(const Node& node);

        /**
         *  Destructor. Don't call delete on Node directly. Use Dispose.
         */
        virtual ~Node() override;

        /**
         *  Renders this Node. Pure virtual function that must be overridden in each object derived from Node.
         */
        virtual void Render() = 0;

        /**
         *  Disposes this Node only - not descendant children nodes.
         *  Pure virtual function that must be overridden in each object derived from Node.
         */
        virtual void DisposeSelf() = 0;

        /**
         *  Convenience function to implement deprecated Clone(Traversing).
         */
        Node* CloneTraversing(Traversing traverse) const;

        /**
         *  Sets the scene graph parent of this Node.
         *  @param parent Node to be set as parent of this Node.
         */
        void SetParent(Node* parent);

        /**
         *  Notification that an ancestor to this Node has been added.
         *  An ancestor is each Node up the tree hierarchy that has a parent relationship to this Node.
         *  An ancestor can therefore be reached via GetParent(), recursively.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnAncestorAdded is invoked, the parent of this Node is already set and the
         *  Node's WorldTransformCache is invalidated.
         *  @param scene The Scene node which became root of this node's scene graph.
         */
        virtual void OnAncestorAdded(Scene* scene) {
            FEATSTD_UNUSED(scene);
        }

        /**
         *  Notification that an ancestor to this Node has been removed.
         *  An ancestor is each Node up the tree hierarchy that has a parent relationship to this Node.
         *  An ancestor can therefore be reached via GetParent(), recursively.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnAncestorRemoved is invoked, the ancestor of this node is already
         *  disconnected from the scene.
         *  @param scene The Scene node which previously had been the root of the scene graph this node was
         *  connected with.
         */
        virtual void OnAncestorRemoved(Scene* scene) {
            FEATSTD_UNUSED(scene);
        }

        /**
         *  Notification that a descendant of this Node has been removed.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnDescendantRemoved is invoked, the descendant of this node is already
         *  disconnected from the node.
         *  @param descendant The descendant Node that was removed.
         */
        virtual void OnDescendantRemoved(Node* descendant) {
            FEATSTD_UNUSED(descendant);
        }

        /**
         *  Override from Transformable. Invalidates the world transform cache of all descendants. Do not override.
         */
        virtual void OnCompositeTransformChanged() override;

        /** 
         *  Reacts to a transformation change further up in the tree. The reaction should be local/flat, i.e. limited
         *  to this Node (not propagated recursively to descendants).
         */
        virtual void OnAncestorTransformChanged();

        /**
         *  Computes axis-aligned bounding box (AABB) in object coordinate space.
         *  Concrete implementation provided by the derived classes.
         *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
         *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
         *  @return true if bounding box was computed successfully and false if not because of missing geometry.
         */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const {
            FEATSTD_UNUSED(minBounds);
            FEATSTD_UNUSED(maxBounds);
            return false;
        };

        /**
         *  Invoked by class OrderCriterion derivations. Sets a prepared render order criterion value for more efficient sorting.
         *  @param criterionValue specifies the precalculated sort criterion value.
         */
        void SetRenderOrderCriterionValue(const OrderCriterionValue& criterionValue) { m_renderOrderCriterionValue = criterionValue; }
        OrderCriterionValue GetRenderOrderCriterionValue() const { return m_renderOrderCriterionValue; }


        /**
         *  Tests if geometry of this node intersects with a pick in screen coordinates.
         *  In this method the "flat" picking algorithm of a node shall be implemented.
         *  Returns false in case of pure Node, has to be implemented by any derived class that shall support picking.
         *  @remark                   This interface is usually used for object selection in scene editors.
         *  @param camera             The view camera. (i.e. the current camera)
         *  @param x                  The x value as a screen coordinate in pixels starting from top-left.
         *  @param y                  The y value as a screen coordinate in pixels starting from top-left.
         *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   False in case of Node.
         */
        virtual bool IsPickIntersectingGeometryInternal(const Camera& /*camera*/, Int /*x*/, Int /*y*/, Float& /*distance*/ /*out*/) const { return false; }

#if defined(CANDERA_LAYOUT_ENABLED)


        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector3& basePoint) const;

        /**
        *  Returns the "offset" coordinate which is required to position 3D objects correctly when using layout.
        *  Normally, 3D objects are centered but the layouter thinks its position is the "upper-left" corner.
        *  This will be replaced as soon as there is a more appropriate handling (e.g. own pivot point)
        *  @return  layout start position  
        */
        virtual const Vector3 GetLayoutStartPosition() const;
#endif
    private:
        // Notifies all listeners that this node has been removed from its parent.
        void NotifyListenersOnNodeRemoved(Node* parent);

        // Notifies all listeners that the parent of this node has been changed.
        void NotifyListenersOnNodeAdded(Node* parent);

        // Notifies all listeners that a new child has been added to this node.
        void NotifyListenersOnChildAdded(Node* child);

        // Notifies all listeners that a child has been removed from this node.
        void NotifyListenersOnChildRemoved(Node* child);

        // Assigning to a node is prohibited, since it was proved error prone.
        // Clone instead.
        Node& operator=(const Node& node);

        bool m_isRenderingEnabled;  // This Node's local setting, whether rendering is enabled or not.
        bool m_isIntersectionTestEnabled; // Indicates if this Node should react on intersection interfaces.
        // Composite transform cache.
        mutable bool    m_isWorldTransformCacheValid;
        // Effective alpha value cache.
        mutable bool  m_isEffectiveAlphaCacheValid;
        // Center cache.
        mutable bool m_isWorldCenterCacheValid;
        // Radius cache.
        mutable bool m_isWorldRadiusCacheValid;
        Int32 m_renderOrderRank;    // Node's rank within render order bin assigned to.
        SizeType m_childCount;
        Float m_alphaValue;         // This Node's local alpha value.
        Float m_radius;             // The radius of this Node's bounding sphere.
        OrderCriterionValue m_renderOrderCriterionValue; // Node's prepared render order criterion value used by OrderCrition for sorting more efficiently.
        Float m_renderBenchmark;      // Benchmark value related to render time, used BenchmarkCameraRenderStrategy. Default value is 0.0F.
        // Effective alpha value cache.
        mutable Float m_effectiveAlphaValueCache;
        // Radius cache.
        mutable Float m_maxWorldRadiusCache;
        const Char* m_binName;      // Node's assignment to render order bin.
        UInt32 m_binNameHash;
        Node* m_parent;             // parent to this Node.
        Node* m_firstChild;
        mutable Scene* m_scene;
        MemoryManagement::SharedPointer<Appearance> m_appearance;   // Appearance associated to this Node.
        Vector3 m_center;           // Center in object space.
        Vector3 m_minBounds;        // Minimum bounds vector of axis-aligned bounding box (lower left vertex)
        Vector3 m_maxBounds;        // Maximum bounds vector of axis-aligned bounding box (upper right vertex)
        // Center cache.
        mutable Vector3 m_worldCenterCache;
        ScopeMask m_scopeMask;      // Scope mask defines the scopes this Node is related to separate from the scene hierarchy.
        // Composite transform cache.
        mutable Matrix4 m_worldTransformCache;
        // intrusive linked list node for membership in the list of children of the parent
        Candera::Internal::LinkedListNode<Node> m_childListNode;

        typedef Candera::Internal::SingleLinkedList<NodeListener*> NodeListenerContainer;
        NodeListenerContainer m_nodeListeners;

        friend class NodeDynamicProperties;

#if defined(CANDERA_LAYOUT_ENABLED)
        friend class AbstractNodePointer;
        void SetPreferredSize(const Vector2& preferredSize);
        const Vector2& GetPreferredSize() const;
        bool ClearLayouter();
        virtual void OnBeforeLayouterSet(Layouter*) {}
#endif
        /**
         *  Invalidates current world transform cache.
         *  Forces this Node to recalculate its world transform and to update its internal world transform cache the next time the function
         *  GetWorldTransform() is invoked.
         */
        void InvalidateWorldTransformCache() { m_isWorldTransformCacheValid = false; }
        bool IsWorldTransformCacheValid() const { return m_isWorldTransformCacheValid; }

        /**
         *  Invalidates world bounding sphere cache, presented by center and radius.
         */
        void InvalidateWorldCenterCache() { m_isWorldCenterCacheValid = false; }
        bool IsWorldCenterCacheValid() const { return m_isWorldCenterCacheValid;}
        void InvalidateWorldRadiusCache() { m_isWorldRadiusCacheValid = false; }
        bool IsWorldRadiusCacheValid() const { return m_isWorldRadiusCacheValid; }

        /**
         *  Invalidates effective alpha cache, which is the product of this Node's alpha value combined with all ancestors alpha value.
         */
        void InvalidateEffectiveAlphaCache() { m_isEffectiveAlphaCacheValid = false; }
        void InvalidateEffectiveAlphaCache() const { m_isEffectiveAlphaCacheValid = false; }
        bool IsEffectiveAlphaCacheValid() const { return m_isEffectiveAlphaCacheValid; }

        // Updates effective alpha value cache.
        void UpdateEffectiveAlphaValueCache() const;

        // Generates deep clone.
        Node* CloneDeep() const;

        CdaDynamicProperties(Candera::Node, Candera::Transformable);
        CdaDynamicPropertiesEnd();
    };

/** @} */ // end of Core3D

} // namespace Candera


#endif  // CANDERA_NODE_H
