//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Node.h"

#if defined(CANDERA_3D_CANVAS_ENABLED)
#include <Candera/Engine3D/Canvas/CanvasInclude.h>
#endif
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Cloning/TreeCloner.h>
#include <Candera/Engine3D/Core/NodeListener.h>
#include <Candera/Engine3D/ShaderParamSetters/ShaderParamSetter.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Line.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <FeatStd/Util/Hash.h>
#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/Layouter.h>
#include <Candera/EngineBase/Layout/DefaultLayouter.h>
#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>
#endif

namespace Candera {
    using namespace Diagnostics;
    using namespace Candera::Internal;
    using namespace MemoryManagement;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

static const Vector3 c_invalidMinBounds(Math::MaxFloat(), Math::MaxFloat(), Math::MaxFloat());
static const Vector3 c_invalidMaxBounds(-Math::MaxFloat(), -Math::MaxFloat(), -Math::MaxFloat());

// Traverser *******************************************************

/**
 *  OnAncestorAddedTraverser informs descendants that an ancestor has been added.
 */
class OnAncestorAddedTraverser: public TreeTraverser {
    typedef TreeTraverser Base;

    public:
        OnAncestorAddedTraverser(Scene* scene) : Base(), m_scene(scene) {}
        ~OnAncestorAddedTraverser() { m_scene = 0; }

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            node.m_scene = m_scene;
            node.OnAncestorAdded(m_scene);
            return ProceedTraversing;
        }

    private:
        Scene* m_scene;
};

/**
 *  OnAncestorRemovedTraverser informs descendants that an ancestor has been removed.
 */
class OnAncestorRemovedTraverser: public TreeTraverser {
    typedef TreeTraverser Base;

    public:
        OnAncestorRemovedTraverser(Scene* scene) : Base(), m_scene(scene) {}
        ~OnAncestorRemovedTraverser() { m_scene = 0; }

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            node.m_scene = 0;
            node.OnAncestorRemoved(m_scene);
            return ProceedTraversing;
        }

    private:
        Scene* m_scene;
};

/**
 *  InvalidateWorldTransformCacheTraverser informs descendants that world transform has changed.
 */
class InvalidateWorldTransformCacheTraverser: public TreeTraverser
{
    public:
        InvalidateWorldTransformCacheTraverser() {}
        ~InvalidateWorldTransformCacheTraverser() {}

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            if (node.IsWorldTransformCacheValid()) {
                node.InvalidateWorldTransformCache();
                node.InvalidateWorldCenterCache();
                node.InvalidateWorldRadiusCache();
                node.OnAncestorTransformChanged();
                return ProceedTraversing;
            }

            return StopTraversingForDescendants;
        }
};

/**
 *  UploadTraverser invokes Upload at the Node given and all its descendants.
 */
class UploadTraverser: public TreeTraverser {
    typedef TreeTraverser Base;

    public:
        UploadTraverser(const ScopeMask& scopeMask) :
            Base(),
            m_isEffectiveResultSuccessful(true),
            m_scopeMask(scopeMask)
        {
        }

        ~UploadTraverser() {}


        bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            if (node.IsInScopeOf(m_scopeMask)) {
                if(! node.Upload() ) {
                    m_isEffectiveResultSuccessful = false;
                }
            }
            return ProceedTraversing;
        }

    private:
        bool m_isEffectiveResultSuccessful;
        const ScopeMask m_scopeMask;

    FEATSTD_MAKE_CLASS_UNCOPYABLE(UploadTraverser);
};

/**
 * MultiContextUploadTraverser looks for all cameras in the scene tree,
 * activates them and uploads the given node and its descendants to the bound context.
 */
class MultiContextUploadTraverser : public TreeTraverser {
    typedef TreeTraverser Base;

public:
    MultiContextUploadTraverser(const ScopeMask& scopeMask, Node* node) :
        Base(),
        m_isEffectiveResultSuccessful(true),
        m_scopeMask(scopeMask),
        m_node(node)
    {
    }

    ~MultiContextUploadTraverser()
    {
        m_node = 0;
    }

    bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

protected:
    virtual TraverserAction ProcessNode(Node& node) override
    {
        if (m_node != 0) {
            Camera* camera = Dynamic_Cast<Camera*>(&node);
            if((camera != 0) && (node.IsInScopeOf(m_scopeMask))) {
                RenderTarget3D* rt = camera->GetRenderTarget();
                if (rt != 0) {
                    if (rt->Activate()) {
                        //ScopeMask intersection.
                        ScopeMask scopeMaskUnion = (node.GetScopeMask() & m_scopeMask);
                        UploadTraverser traverser(scopeMaskUnion);
                        traverser.Traverse(*m_node);
                        m_isEffectiveResultSuccessful =
                            m_isEffectiveResultSuccessful &&
                            traverser.IsEffectiveResultSuccessful();
                    }
                }
            }
        }

        return ProceedTraversing;
    }

private:
    bool m_isEffectiveResultSuccessful;
    const ScopeMask m_scopeMask;
    Node* m_node;

    FEATSTD_MAKE_CLASS_UNCOPYABLE(MultiContextUploadTraverser);
};

/**
 *  UnloadTraverser invokes Unload at the Node given and all its descendants.
 */
class UnloadTraverser: public TreeTraverser {
    typedef TreeTraverser Base;

    public:
        UnloadTraverser(const ScopeMask& scopeMask) :
            Base(),
            m_isEffectiveResultSuccessful(true),
            m_scopeMask(scopeMask)
        {
        }

        ~UnloadTraverser() {}

        bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            if (node.IsInScopeOf(m_scopeMask)) {
                if(! node.Unload() ) {
                    m_isEffectiveResultSuccessful = false;
                }
            }
            return ProceedTraversing;
        }

    private:
        bool m_isEffectiveResultSuccessful;
        const ScopeMask m_scopeMask;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(UnloadTraverser);
};

/**
 * MultiContextUnloadTraverser looks for all cameras in the scene tree,
 * activates them and unloads the given node and its descendants from the bound context.
 */
class MultiContextUnloadTraverser : public TreeTraverser {
    typedef TreeTraverser Base;

public:
    MultiContextUnloadTraverser(const ScopeMask& scopeMask, Node* node) :
        Base(),
        m_isEffectiveResultSuccessful(true),
        m_scopeMask(scopeMask),
        m_node(node)
    {
    }

    ~MultiContextUnloadTraverser()
    {
        m_node = 0;
    }

    bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

protected:
    virtual TraverserAction ProcessNode(Node& node) override
    {
        if (m_node != 0) {
            Camera* camera = Dynamic_Cast<Camera*>(&node);
            if((camera != 0) && (node.IsInScopeOf(m_scopeMask))) {
                RenderTarget3D* rt = camera->GetRenderTarget();
                if (rt != 0) {
                    if (rt->Activate()) {
                        //ScopeMask intersection.
                        ScopeMask scopeMaskUnion = (node.GetScopeMask() & m_scopeMask);
                        UnloadTraverser traverser(scopeMaskUnion);
                        traverser.Traverse(*m_node);
                        m_isEffectiveResultSuccessful =
                            m_isEffectiveResultSuccessful &&
                            traverser.IsEffectiveResultSuccessful();
                    }
                }
            }
        }

        return ProceedTraversing;
    }

private:
    bool m_isEffectiveResultSuccessful;
    const ScopeMask m_scopeMask;
    Node* m_node;

    FEATSTD_MAKE_CLASS_UNCOPYABLE(MultiContextUnloadTraverser);
};

/**
 *  SetScopeMaskTraverser invokes SetScopeMask at the Node given and all its descendants.
 */
class SetScopeMaskTraverser: public TreeTraverser {
    typedef TreeTraverser Base;

    public:
        SetScopeMaskTraverser(const ScopeMask& scopeMask) : Base(), m_scopeMask(scopeMask) {}
        ~SetScopeMaskTraverser() {}

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            node.SetScopeMask(m_scopeMask, Node::Flat);
            return ProceedTraversing;
        }

    private:
        const ScopeMask m_scopeMask;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(SetScopeMaskTraverser);
};

/**
 *  SetScopeEnabledTraverser invokes SetScopeEnabled at the Node given and all its descendants.
 */
class SetScopeEnabledTraverser: public TreeTraverser {
    typedef TreeTraverser Base;

    public:
        SetScopeEnabledTraverser(UInt scopeIndex, bool enable) : Base(), m_scopeIndex(scopeIndex), m_enable(enable), m_isEffectiveResultSuccessful(true) {}
        ~SetScopeEnabledTraverser() {}

        bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            if (! node.SetScopeEnabled(m_scopeIndex, m_enable, Node::Flat)) {
                m_isEffectiveResultSuccessful = false;
            }
            return ProceedTraversing;
        }

    private:
        const UInt m_scopeIndex;  // Scope index to update.
        const bool m_enable;       // Enable or disable scope at scope index given.
        bool m_isEffectiveResultSuccessful;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(SetScopeEnabledTraverser);
};

/**
 *  FindByNameTraverser traverses to find a Node by the name given.
 */
class FindByNameTraverser: public ConstTreeTraverser
{
    public:

        FindByNameTraverser() :
            m_findName(0),
            m_matchingNode(0),
            m_rootNode(0)
        {
        }
        ~FindByNameTraverser()
        {
            m_findName = 0;
            m_matchingNode = 0;
            m_rootNode = 0;
        }

        void SetNameToFind(const Char* findName)
        {
            m_findName = findName;
            m_matchingNode = 0;
        }

        void SetRootNode(const Node* rootNode) { m_rootNode = rootNode; }

        const Node* GetMatchingNode() const { return m_matchingNode; }

    protected:
        virtual TraverserAction ProcessNode(const Node& node) override
        {
            if (&node == m_rootNode) {
                return ProceedTraversing;
            }
            const Char* const nodeName = node.GetName();
            const bool found = ((nodeName != 0) && (StringPlatform::CompareStrings(nodeName, m_findName) == 0));
            if (found) {
                m_matchingNode = &node;
            }
            return (found ? StopTraversing: ProceedTraversing);
        }

    private:
        const Char* m_findName;
        const Node* m_matchingNode;
        const Node* m_rootNode;
};

/**
 *  InvalidateEffectiveAlphaDownwardsTraverser informs descendants that the their effective alpha value cache becomes invalidated.
 */
class InvalidateEffectiveAlphaDownwardsTraverser: public TreeTraverser
{
    public:
        InvalidateEffectiveAlphaDownwardsTraverser() {}
        ~InvalidateEffectiveAlphaDownwardsTraverser() {}

    protected:
        virtual TraverserAction ProcessNode(Node& node) override
        {
            node.InvalidateEffectiveAlphaCache();
            return ProceedTraversing;
        }
};

/**
*  WorldAxisAlignedBoundingBoxTraverser checks position of all nodes .
*/
class WorldAxisAlignedBoundingBoxTraverser: public ConstTreeTraverser {
    typedef ConstTreeTraverser Base;

    public:
        WorldAxisAlignedBoundingBoxTraverser() :
            Base(),
            m_minBounds(c_invalidMinBounds),
            m_maxBounds(c_invalidMaxBounds)
        {
        }

        ~WorldAxisAlignedBoundingBoxTraverser() {};

        Vector3 GetMinBounds() const { return m_minBounds; }
        Vector3 GetMaxBounds() const { return m_maxBounds; }

    protected:
        virtual TraverserAction ProcessNode(const Node& node) override
        {
            Vector3 minBounds;
            Vector3 maxBounds; // Node-oriented bounding box.
            node.GetBoundingBox(minBounds, maxBounds);

            if (node.IsBoundingBoxValid()) {
                Vector3 worldMinBounds;
                Vector3 worldMaxBounds;  // World-oriented bounding box.

                // Transform node's bounding box into world space.
                Math3D::TransformBoundingBox(minBounds, maxBounds, node.GetWorldTransform(), worldMinBounds, worldMaxBounds);

                // Accumulate boundaries if they exceed current world axis aligned bounding box.
                if (worldMinBounds.GetX() < m_minBounds.GetX()) { m_minBounds.SetX(worldMinBounds.GetX()); }
                if (worldMinBounds.GetY() < m_minBounds.GetY()) { m_minBounds.SetY(worldMinBounds.GetY()); }
                if (worldMinBounds.GetZ() < m_minBounds.GetZ()) { m_minBounds.SetZ(worldMinBounds.GetZ()); }

                if (worldMaxBounds.GetX() > m_maxBounds.GetX()) { m_maxBounds.SetX(worldMaxBounds.GetX()); }
                if (worldMaxBounds.GetY() > m_maxBounds.GetY()) { m_maxBounds.SetY(worldMaxBounds.GetY()); }
                if (worldMaxBounds.GetZ() > m_maxBounds.GetZ()) { m_maxBounds.SetZ(worldMaxBounds.GetZ()); }
            }
            return ProceedTraversing;
        }

    private:
        Vector3 m_minBounds;
        Vector3 m_maxBounds;
};


/**
 *  PickTraverser traverses to pick test all child nodes.
 */
class PickTraverser: public ConstTreeTraverser {
    typedef ConstTreeTraverser Base;

    public:
        PickTraverser(const Camera& camera, Int x, Int y):
            Base(),
            m_isPickIntersectingSubTree(false),
            m_x(x),
            m_y(y),
            m_nearestDistance(Math::MaxFloat()),
            m_camera(&camera)
          {

          }

        ~PickTraverser()
        {
            m_camera = 0;
        };

        /**
         * Returns whether the subtree was hit by the picking line or not.
         * @return Whether the subtree was hit by the picking line or not.
         */
        bool IsPickIntersectingSubTree() const { return m_isPickIntersectingSubTree; }

        /**
         * Returns the smallest distance of all hit nodes of the subtree.
         * @return The smallest distance of all hit nodes of the subtree.
         */
        Float GetDistance() const { return m_nearestDistance; }

    protected:
        virtual TraverserAction ProcessNode(const Node& node) override
        {
            if (!node.IsIntersectionTestEnabled()) {
                return StopTraversingForDescendants;
            }

            Float distance = 0.0F;
            bool intersects = node.IsPickIntersectingGeometryInternal(*m_camera, m_x, m_y, distance);

            if (intersects) {
                m_isPickIntersectingSubTree = true;
                if (m_nearestDistance > distance ) {
                    m_nearestDistance = distance;
                }
            }

            return ProceedTraversing;
        }

    private:
        bool m_isPickIntersectingSubTree;
        Int m_x;
        Int m_y;
        Float m_nearestDistance;
        const Camera* m_camera;

        FEATSTD_MAKE_CLASS_UNCOPYABLE(PickTraverser);
};

#if defined(CANDERA_LAYOUT_ENABLED)
/**
*  LayoutTraverser traverses through all children until if finds a node having a Layouter. Therefore Layouter::Layout() is called,
*  all its children are skipped and the traverser continues with the next sibling.
*/
class Layout3DTraverser : public TreeTraverserBase<Node> {
protected:
    /**
    *  If the node has a layouter, Layout() is called but its children are ignored - they are handled by the layouter itself.
    */
    virtual TraverserAction ProcessNode(Node& node) override
    {
        Layouter* layouter = node.GetLayouter();
        if ((layouter != 0) && (layouter != DefaultLayouter::GetInstance())) {
            AbstractNodePointer abstractNode(&node);
            layouter->Layout(abstractNode);
            return StopTraversingForDescendants;
        }
        return ProceedTraversing;
    }
};
#endif

// Dynamic Node properties implementation ****************************

class NodeDynamicProperties {
private:
    friend class Node;


#if defined(CANDERA_LAYOUT_ENABLED)
    static void SetLayouter(Node& node, Layouter* layouter)
    {
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(Layouter), layouter));
    }

    static Layouter* GetLayouter(const Node& node)
    {
        return node.GetValue(CdaDynamicPropertyInstance(Layouter));
    }

    static Layouter* GetDefaultLayouter()
    {
        return DefaultLayouter::GetInstance();
    }

    static void SetPreferredSize(Node& node, const Vector2& preferredSize)
    {
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(PreferredSize), preferredSize));
    }

    static const Vector2& GetPreferredSize(const Node& node)
    {
        return node.GetValue(CdaDynamicPropertyInstance(PreferredSize));
    }
    static Vector2& GetDefaultPreferredSize()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(Vector2, s_preferredSizeDefault);
        return s_preferredSizeDefault;
    }

#endif
    CdaDynamicPropertiesDefinition(Candera::Node, Candera::Node::Base);

#if defined(CANDERA_LAYOUT_ENABLED)

    CdaDynamicProperty(Layouter, Layouter*);
        CdaDynamicPropertyDefaultValue(GetDefaultLayouter());
        CdaDynamicPropertyDescription("Defines a layout strategy for the child nodes, e.g. a grid or stack layout.")
        CdaDynamicPropertyCategory("Layout")
    CdaDynamicPropertyEnd();

    CdaDynamicProperty(PreferredSize, Vector2);
        CdaDynamicPropertyDefaultValue(GetDefaultPreferredSize());
        CdaDynamicPropertyCategory("Layout")
    CdaDynamicPropertyEnd();

#endif
    CdaDynamicPropertiesEnd();
};

// Node implementation ***********************************************

FEATSTD_RTTI_DEFINITION(Node, Transformable)

Node::Node():
    Base(),
    m_isRenderingEnabled(true),
    m_isIntersectionTestEnabled(true),
    m_isWorldTransformCacheValid(false),
    m_isEffectiveAlphaCacheValid(false),
    m_isWorldCenterCacheValid(false),
    m_isWorldRadiusCacheValid(false),
    m_renderOrderRank(0),
    m_childCount(0),
    m_alphaValue(1.0F),
    m_radius(0.0F),
    m_renderOrderCriterionValue(0.0F),
    m_renderBenchmark(0.0F),
    m_effectiveAlphaValueCache(1.0F),
    m_maxWorldRadiusCache(0.0F),
    m_binName(0),
    m_binNameHash(0),
    m_parent(0),
    m_firstChild(0),
    m_scene(0),
    m_appearance(0),
    m_center(0.0F, 0.0F, 0.0F),
    m_minBounds(c_invalidMinBounds),
    m_maxBounds(c_invalidMaxBounds),
    m_worldCenterCache(0.0F, 0.0F, 0.0F),
    m_childListNode()
{
}

Node::Node(const Node &node):
    Base(node),
    m_isRenderingEnabled(node.m_isRenderingEnabled),
    m_isIntersectionTestEnabled(node.m_isIntersectionTestEnabled),
    m_isWorldTransformCacheValid(false),
    m_isEffectiveAlphaCacheValid(false),
    m_isWorldCenterCacheValid(false),
    m_isWorldRadiusCacheValid(false),
    m_renderOrderRank(node.m_renderOrderRank),
    m_childCount(0),
    m_alphaValue(node.m_alphaValue),
    m_radius(node.m_radius),
    m_renderOrderCriterionValue(node.m_renderOrderCriterionValue),
    m_renderBenchmark(node.m_renderBenchmark),
    m_effectiveAlphaValueCache(1.0F),
    m_maxWorldRadiusCache(0.0F),
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1554, Candera::Node::m_binName, CANDERA_LINT_REASON_ASSOCIATION)
    m_binName(node.m_binName),
    m_binNameHash(node.m_binNameHash),
    m_parent(0),
    m_firstChild(0),
    m_scene(0),
    m_appearance(node.m_appearance),
    m_center(node.m_center),
    m_minBounds(node.m_minBounds),
    m_maxBounds(node.m_maxBounds),
    m_worldCenterCache(0.0F, 0.0F, 0.0F),
    m_scopeMask(node.m_scopeMask),
    m_childListNode()
{
}

Node::~Node()
{
#if defined(CANDERA_LAYOUT_ENABLED)
    Layouter* layouter = GetLayouter();
    if (layouter != 0) {
        layouter->Dispose();
    }
#endif
    m_parent = 0;
    m_scene = 0;
    m_firstChild = 0;
    m_binName = 0;
}

void Node::Dispose()
{
    // Dispose this Node and descendants.
    Node* const root = this;
    Node* currentNode = this;
    bool done = false;
    while (!done) {
        Node* const firstChild = currentNode->GetFirstChild();
        const bool currentNodeIsLeaf = (firstChild == 0);
        if (currentNodeIsLeaf) {
            Node* const parentNode = currentNode->GetParent();
            if (parentNode != 0) {
                static_cast<void>(parentNode->RemoveChild(currentNode));
            }
            currentNode->DisposeSelf();
            if (currentNode == root) {
                done = true;
            }
            else {
                // up to parent
                currentNode = parentNode;
            }
        }
        else {
            // down to first child
            currentNode = firstChild;
        }
    }
}

Node* Node::CloneTraversing(Traversing traverse) const
{
    if (IsTypeOf(Scene::GetTypeId())) {
        return 0; // it is not possible to clone Scenes
    }

    if (traverse == Flat) {
        return Clone();
    }
    else if (traverse == Deep) {
        return CloneDeep();
    }
    else {
        FEATSTD_LOG_ERROR("Node clone failed, reached unexpected default case.");
        return 0;
    }
}

Node* Node::CloneDeep() const
{
    return TreeCloner().CreateClone(*this);
}

Scene* Node::GetScene() const
{
    if (m_scene == 0) {
         m_scene = Dynamic_Cast<Scene*>(const_cast<Node*>(this));
    }
    return m_scene;
}

bool Node::IsEffectiveRenderingEnabled() const
{
    const Node* currentNode = this;
    while (currentNode != 0) {
        if (!currentNode->IsRenderingEnabled()) {
            return false;
        }
        currentNode = currentNode->GetParent();
    }
    return true;
}

void Node::SetAlphaValue(Float alphaValue)
{
    m_alphaValue = alphaValue;
    InvalidateEffectiveAlphaDownwardsTraverser traverser;
    traverser.Traverse(*this);
}

Float Node::GetEffectiveAlphaValue() const
{
    UpdateEffectiveAlphaValueCache();
    return m_effectiveAlphaValueCache;
}

void Node::UpdateEffectiveAlphaValueCache() const
{
    if (IsEffectiveAlphaCacheValid()) {
        return;
    }

    Float effectiveAlpha = GetAlphaValue();
    const Node* node = GetParent();

    while (node != 0) {
        effectiveAlpha *= node->GetAlphaValue();
        node = node->GetParent();
    }

    m_effectiveAlphaValueCache = effectiveAlpha;
    m_isEffectiveAlphaCacheValid = true;
}

Matrix4 Node::GetWorldRotation() const
{
    Vector3 s;
    Matrix4 r;
    Vector3 t;
    // When this fails, r is already identity.
    static_cast<void>(GetWorldTransform().Decompose(s, r, t));
    return r;
}

Matrix4 Node::GetWorldScale() const
{
    Vector3 worldScaleVec;
    Matrix4 r;
    Vector3 t;
    if (!GetWorldTransform().Decompose(worldScaleVec, r, t)) {
        worldScaleVec.SetZero();
    }

    // Construct scale matrix.
    Matrix4 worldScale;
    worldScale.SetScaling(worldScaleVec.GetX(), worldScaleVec.GetY(), worldScaleVec.GetZ());
    return worldScale;
}

Vector3 Node::GetWorldPosition() const
{
    const Matrix4& wt = GetWorldTransform();

    Vector3 position(wt.Get(3, 0),
                     wt.Get(3, 1),
                     wt.Get(3, 2));

    return position;
}

Vector3 Node::GetWorldCenter() const
{
    if (!IsWorldCenterCacheValid()) {
        m_worldCenterCache = m_center;
        m_worldCenterCache.TransformCoordinate(GetWorldTransform());
        m_isWorldCenterCacheValid = true;
    }
    return m_worldCenterCache;
}

Float Node::GetWorldRadius() const
{
    if (!IsWorldRadiusCacheValid()) {
        const Matrix4& wt = GetWorldTransform();

        // Decompose world scale of world transform matrix.
        const Float radX = Math::SquareRoot((wt(0, 0) * wt(0, 0)) + (wt(0, 1) * wt(0, 1)) + (wt(0, 2) * wt(0, 2)));
        const Float radY = Math::SquareRoot((wt(1, 0) * wt(1, 0)) + (wt(1, 1) * wt(1, 1)) + (wt(1, 2) * wt(1, 2)));
        const Float radZ = Math::SquareRoot((wt(2, 0) * wt(2, 0)) + (wt(2, 1) * wt(2, 1)) + (wt(2, 2) * wt(2, 2)));

        // As non-uniform scale is possible, find the biggest scale value.
        m_maxWorldRadiusCache = Math::Maximum(radX, radY);
        m_maxWorldRadiusCache = Math::Maximum(m_maxWorldRadiusCache, radZ);

        // Finally, multiply biggest scale factor with radius to retrieve world radius.
        m_maxWorldRadiusCache *= m_radius;
        m_isWorldRadiusCacheValid = true;
    }

    return m_maxWorldRadiusCache;
}

bool Node::ComputeBoundingSphere()
{
    Vector3 minBounds;
    Vector3 maxBounds;
    if (ComputeBoundingBoxImpl(minBounds, maxBounds)) {
        SetCenter((minBounds + maxBounds) / 2.0F);
        SetRadius(((maxBounds - minBounds) / 2.0F).GetLength());
        return true;
    }
    return false;
}

void Node::SetBoundingBox(const Vector3& minBounds, const Vector3& maxBounds)
{
    m_minBounds = minBounds;
    m_maxBounds = maxBounds;
}

void Node::GetBoundingBox(Vector3& minBounds, Vector3& maxBounds) const
{
    minBounds = m_minBounds;
    maxBounds = m_maxBounds;
}

bool Node::ComputeBoundingBox()
{
    return ComputeBoundingBoxImpl(m_minBounds, m_maxBounds);
}

bool Node::IsBoundingBoxValid() const
{
    return ((m_minBounds != c_invalidMinBounds) && (m_maxBounds != c_invalidMaxBounds));
}

void Node::GetWorldOrientedBoundingBox(Vector3 wobbBounds[8]) const
{
    //get all 8 vertices based on previously set or calculated minBounds/maxBounds
    wobbBounds[0] = Vector3(m_minBounds.GetX(), m_minBounds.GetY(), m_minBounds.GetZ()); // xyz
    wobbBounds[1] = Vector3(m_maxBounds.GetX(), m_minBounds.GetY(), m_minBounds.GetZ()); // Xyz
    wobbBounds[2] = Vector3(m_minBounds.GetX(), m_maxBounds.GetY(), m_minBounds.GetZ()); // xYz
    wobbBounds[3] = Vector3(m_maxBounds.GetX(), m_maxBounds.GetY(), m_minBounds.GetZ()); // XYz
    wobbBounds[4] = Vector3(m_minBounds.GetX(), m_minBounds.GetY(), m_maxBounds.GetZ()); // xyZ
    wobbBounds[5] = Vector3(m_maxBounds.GetX(), m_minBounds.GetY(), m_maxBounds.GetZ()); // XyZ
    wobbBounds[6] = Vector3(m_minBounds.GetX(), m_maxBounds.GetY(), m_maxBounds.GetZ()); // xYZ
    wobbBounds[7] = Vector3(m_maxBounds.GetX(), m_maxBounds.GetY(), m_maxBounds.GetZ()); // XYZ

    //transform coordinates to world space object space
    Matrix4 worldMatrix = this->GetWorldTransform();
    for (Int8 i=0;i<8;i++) {
        wobbBounds[i].TransformCoordinate(worldMatrix);
    }
}

void Node::GetWorldAxisAlignedBoundingBox(Vector3& minBounds, Vector3& maxBounds, Traversing traverse) const
{
    if (traverse == Flat) {
        if (IsBoundingBoxValid()) {
            // Transform node's bounding box into world space.
            Math3D::TransformBoundingBox(m_minBounds, m_maxBounds, GetWorldTransform(), minBounds, maxBounds);
        }
        else {
            minBounds = c_invalidMinBounds;
            maxBounds = c_invalidMaxBounds;
        }
    }
    else {
        WorldAxisAlignedBoundingBoxTraverser traverser;
        traverser.Traverse(*this);
        minBounds = traverser.GetMinBounds();
        maxBounds = traverser.GetMaxBounds();
    }
}

void Node::SetRenderOrderBinAssignment(const Char* binName)
{
    m_binName = binName;
    if (binName != 0) {
        m_binNameHash = FeatStd::Hash::CalcHash(binName);
    }
    else {
        m_binNameHash = 0;
    }
}

void Node::SetParent(Node* parent)
{
    m_parent = parent;
    NotifyListenersOnNodeAdded(parent);
}

bool Node::IsEffectiveIntersectionTestEnabled() const
{
    const Node* currentNode = this;
    while (currentNode != 0) {
        if (!currentNode->IsIntersectionTestEnabled()) {
            return false;
        }
        currentNode = currentNode->GetParent();
    }
    return true;
}

bool Node::IsLineIntersectingBoundingSphere(const Line& line, Float& distance) const
{
    bool result = false;
    if ( IsIntersectionTestEnabled() ) {
        Vector3 hitPosition;
        if (Math3D::SphereLineIntersection(line, GetWorldCenter(), GetWorldRadius(), hitPosition)) {
            distance = (hitPosition - line.GetStart()).GetLength();
            result = true;
        }
    }
    return result;
}

bool Node::IsLineIntersectingGeometry(const Line& /*line*/, Float& /*distance*/) const
{
    return false;
}

bool Node::IsPickIntersectingGeometry(const Camera& camera, Int x, Int y, Float& distance /*out*/, Traversing traverse) const
{
    bool result = false;

    if (IsEffectiveIntersectionTestEnabled()) {
        //Only check node or sub tree if intersection test is enabled effectively.
        if (traverse == Flat) {
            result = IsPickIntersectingGeometryInternal(camera, x, y, distance);
        }
        else if (traverse == Deep) {
            PickTraverser traverser(camera, x, y);
            traverser.Traverse(*this);

            if(traverser.IsPickIntersectingSubTree()) {
                distance = traverser.GetDistance();
                result = true;
            }
        }
        else {
            // result = false;
        }
    }

    return result;
}

bool Node::InsertBeforeChild(Node* child, Node* beforeChild)
{
    // Return error if child is null, child is this group, child has already a parent set, or child is a scene.
    // XXX TODO Return error if child is ancestor to this group
    if ((child == 0) || (child == this) || (child->GetParent() != 0) || child->IsTypeOf(Scene::GetTypeId()) ) {
        if (child == 0) {
            FEATSTD_LOG_ERROR("Add child failed, child == 0.");
        }
        else {
            if (child == this) {
                FEATSTD_LOG_ERROR("Add child failed, child == this.");
            }
            if (child->GetParent() != 0) {
                FEATSTD_LOG_ERROR("Add child failed, child already has a parent.");
            }
            if (child->IsTypeOf(Scene::GetTypeId())) {
                FEATSTD_LOG_ERROR("Add child failed, child is a scene.");
            }
        }
        return false;
    }

#if defined(CANDERA_3D_CANVAS_ENABLED)

    bool isCanvasNode = CanvasTransformable::IsCanvasNode(*this);
    //bool isCanvas = this->IsTypeOf(Canvas::GetTypeId());
    bool isChildCanvasNode = CanvasTransformable::IsCanvasNode(*child);
    bool isChildCanvas = child->IsTypeOf(Canvas::GetTypeId());

    if (isCanvasNode) {
        if (isChildCanvas) {
            FEATSTD_LOG_ERROR("Add child failed, nested canvases are not possible.");
            return false;
        }

        if (!isChildCanvasNode) {
            FEATSTD_LOG_ERROR("Add child failed, only CanvasNodes can be children of CanvasNodes.");
            return false;
        }
    }
    else{
        if (isChildCanvasNode && (!isChildCanvas)) {
            FEATSTD_LOG_ERROR("Add child failed, CanvasNodes cannot be children of 3D nodes.");
            return false;
        }
    }

 #endif

    child->SetParent(this);

    InvalidateWorldTransformCacheTraverser iwtcTraverser;
    iwtcTraverser.Traverse(*child);

    InvalidateEffectiveAlphaDownwardsTraverser alphaTraverser;
    alphaTraverser.Traverse(*this);

    Candera::Internal::LinkedList<Node, &Node::m_childListNode> childList(m_childCount, m_firstChild);
    childList.Insert(child, beforeChild);
    m_firstChild = &*childList.Begin();
    m_childCount = childList.GetSize();

#if defined(CANDERA_LAYOUT_ENABLED)
    Scene* scene = GetScene();
    if (scene != 0) {
        AbstractNodePointer abstractNode(this);
        abstractNode.InvalidateLayout();
    }
#endif
    OnAncestorAddedTraverser oaaTraverser(GetScene());
    oaaTraverser.Traverse(*child);

    NotifyListenersOnChildAdded(child);

    return true;
}

bool Node::RemoveChild(Node *child)
{
    if (child == 0) {
        FEATSTD_LOG_ERROR("Remove child failed, child == 0.");
        return false;
    }

    Candera::Internal::LinkedList<Node, &Node::m_childListNode> childList(m_childCount, m_firstChild);
    const bool removeSucceeded = childList.Remove(child);

    if (removeSucceeded) {
        child->SetParent(0);
    }

    m_firstChild = &*childList.Begin();
    m_childCount = childList.GetSize();

    if (removeSucceeded) {
#if defined(CANDERA_LAYOUT_ENABLED)
        Scene *scene = GetScene();
        if (scene != 0) {
            AbstractNodePointer abstractNode(this);
            abstractNode.InvalidateLayout();
        }
#endif
        OnAncestorRemovedTraverser oarTraverser(GetScene());
        oarTraverser.Traverse(*child);
        child->NotifyListenersOnNodeRemoved(this);
        for (Node* parent = this; parent != 0; parent = parent->GetParent()) {
            parent->OnDescendantRemoved(child); 
        }
    }

    if (removeSucceeded) {
        NotifyListenersOnChildRemoved(child);
    }

    return removeSucceeded;
}

void Node::RemoveAllChildren()
{
    Node* const root = this;
    Node* currentNode = this;
    bool done = false;
    while (!done) {
        Node* const firstChild = currentNode->GetFirstChild();
        const bool currentNodeIsLeaf = (firstChild == 0);
        if (currentNodeIsLeaf) {
            Node* const parentNode = currentNode->GetParent();
            if (parentNode != 0) {
                static_cast<void>(parentNode->RemoveChild(currentNode));
            }
            if (currentNode == root) {
                done = true;
            }
            else {
                // up to parent
                currentNode = parentNode;
                if (0 == currentNode){
                    FEATSTD_LOG_FATAL("Node or one of it's children is in an invalid state.");
                    done = true;
                }
            }
        }
        else {
            // down to first child
            currentNode = firstChild;
        }
    }
}

void Node::NotifyListenersOnNodeRemoved(Node* parent)
{
    for (NodeListenerContainer::Iterator it = m_nodeListeners.Begin(); it != m_nodeListeners.End(); ++it) {
        (*it)->OnNodeRemoved(parent, this);
    }
}

void Node::NotifyListenersOnNodeAdded(Node* parent)
{
    for (NodeListenerContainer::Iterator it = m_nodeListeners.Begin(); it != m_nodeListeners.End(); ++it) {
        (*it)->OnNodeAdded(parent, this);
    }
}

void Node::NotifyListenersOnChildAdded(Node* child)
{
    for (NodeListenerContainer::Iterator it = m_nodeListeners.Begin(); it != m_nodeListeners.End(); ++it) {
        (*it)->OnChildAdded(child, this);
    }
}

void Node::NotifyListenersOnChildRemoved(Node* child)
{
    for (NodeListenerContainer::Iterator it = m_nodeListeners.Begin(); it != m_nodeListeners.End(); ++it) {
        (*it)->OnChildRemoved(child, this);
    }
}

Node* Node::GetDescendant(const Char* name)
{
    FindByNameTraverser finder;
    finder.SetNameToFind(name);
    finder.SetRootNode(this);
    finder.Traverse(*this);
    return const_cast<Node*>(finder.GetMatchingNode());
}

const Node* Node::GetDescendant(const Char* name) const
{
    FindByNameTraverser finder;
    finder.SetNameToFind(name);
    finder.SetRootNode(this);
    finder.Traverse(*this);
    return finder.GetMatchingNode();
}

Node* Node::GetChild(const Char* name)
{
    if (0 != name) {
        for (Node* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
            const Char* currentNodeName = currentNode->GetName();
            if (0 != currentNodeName) {
                if (StringPlatform::CompareStrings(currentNodeName, name) == 0) {
                    return currentNode;
                }
            }
        }
    }

    return 0;
}

const Node* Node::GetChild(const Char* name) const
{
    if (0 != name) {
        for (Node* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
            const Char* currentNodeName = currentNode->GetName();
            if (0 != currentNodeName) {
                if (StringPlatform::CompareStrings(currentNodeName, name) == 0) {
                    return currentNode;
                }
            }
        }
    }

    return 0;
}

Node* Node::GetChild(Id id)
{
    for (Node* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
        if (currentNode->GetId() == id) {
            return currentNode;
        }
    }
    return 0;
}

const Node* Node::GetChild(Id id) const
{
    for (Node* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
        if (currentNode->GetId() == id) {
            return currentNode;
        }
    }
    return 0;
}

void Node::OnCompositeTransformChanged()
{
    m_isWorldCenterCacheValid = false;
    m_isWorldRadiusCacheValid = false;
    InvalidateWorldTransformCacheTraverser iwtcTraverser;
    iwtcTraverser.Traverse(*this);

    // notify listeners
    for (NodeListenerContainer::Iterator it = m_nodeListeners.Begin(); it != m_nodeListeners.End(); ++it) {
        NodeListener* listener = *it;
        listener->OnTransformChange(this);
    }
}

void Node::OnAncestorTransformChanged()
{
    // no default action
}

SizeType Node::GetChildCount() const
{
    return m_childCount;
}

Node* Node::GetFirstChild()
{
    return m_firstChild;
}

const Node* Node::GetFirstChild() const
{
    return m_firstChild;
}

Node* Node::GetNextSibling()
{
    return m_childListNode.GetNext();
}

const Node* Node::GetNextSibling() const
{
    return m_childListNode.GetNext();
}

bool Node::UploadAll(const ScopeMask& scopeMask, bool multiContext)
{
    bool result = false;

    if (multiContext) {
        MultiContextUploadTraverser uploadAllContextsTraverser(scopeMask, this);
        Scene* scene = this->GetScene();
        if (scene != 0) {
            uploadAllContextsTraverser.Traverse(*scene);
            result = uploadAllContextsTraverser.IsEffectiveResultSuccessful();
        }
    }
    else {
        UploadTraverser uploadTraverser(scopeMask);
        uploadTraverser.Traverse(*this);
        result = uploadTraverser.IsEffectiveResultSuccessful();
    }

    return result;
}

bool Node::UnloadAll(const ScopeMask& scopeMask, bool multiContext)
{
    bool result = false;

    if (multiContext) {
        MultiContextUnloadTraverser unloadAllContextsTraverser(scopeMask, this);
        Scene* scene = this->GetScene();
        if (scene != 0) {
            unloadAllContextsTraverser.Traverse(*scene);
            result = unloadAllContextsTraverser.IsEffectiveResultSuccessful();
        }
    }
    else {
        UnloadTraverser unloadTraverser(scopeMask);
        unloadTraverser.Traverse(*this);
        result = unloadTraverser.IsEffectiveResultSuccessful();
    }

    return result;
}

void Node::SetScopeMask(const ScopeMask& scopeMask, Traversing traverse)
{
    if (traverse == Flat) {
        m_scopeMask = scopeMask;
    }
    else { //traverse == Deep
        SetScopeMaskTraverser setScopeMaskTraverser(scopeMask);
        setScopeMaskTraverser.Traverse(*this);
    }
}

bool Node::SetScopeEnabled(UInt scopeIndex, bool enable, Traversing traverse)
{
    bool result;
    if (traverse == Flat) {
        result = m_scopeMask.SetScopeEnabled(scopeIndex, enable);
    }
    else { //traverse == Deep
        SetScopeEnabledTraverser setScopeEnabledTraverser(scopeIndex, enable);
        setScopeEnabledTraverser.Traverse(*this);
        result =  setScopeEnabledTraverser.IsEffectiveResultSuccessful();
    }
    return result;
}

bool Node::IsRenderPrerequisiteFulfilled() const
{
    for (Appearance* appearance = m_appearance.GetPointerToSharedInstance(); 0 != appearance; appearance = appearance->GetNextPass().GetPointerToSharedInstance())
    {
        Shader* shader = appearance->m_shader.GetPointerToSharedInstance();
        if ((0 == shader) || (!shader->IsUploaded())) {
            return false;
        }
    }

    return true;
}

bool Node::IsParentOf(const Node * node) const
{
    return (node != 0) ? ((node->GetParent()) == this) : false;
}

bool Node::AddNodeListener(NodeListener* listener)
{
    bool isSuccessful = false;
    if (listener != 0) {
        isSuccessful = m_nodeListeners.Append(listener);
    }
    return isSuccessful;
}

bool Node::RemoveNodeListener(NodeListener* listener)
{
    bool isSuccessful = false;
    if (listener != 0) {
        isSuccessful = m_nodeListeners.Remove(listener);
    }
    return isSuccessful;
}


const Matrix4& Node::GetWorldTransform() const
{
    //compute world transform and update world transformation cache only, if not up-to-date
    if ( (!IsWorldTransformCacheValid()) || (!IsCompositeTransformCacheValid()) ) {
        // Get local composite transformation of this Node first
        m_worldTransformCache = GetCompositeTransform();
        // Multiply local composite transform with ancestors world transform
        if (GetParent() != 0) {
            m_worldTransformCache *= GetParent()->GetWorldTransform();
        }
        m_isWorldTransformCacheValid = true;
    }
    return m_worldTransformCache;
}

void Node::SetWorldTransform(const Matrix4& worldTransform)
{
    m_worldTransformCache = worldTransform;
    m_isWorldTransformCacheValid = true;
    m_isCompositeTransformCacheValid = true;
}


/******************************************************************************
*  Layout
******************************************************************************/
#if defined(CANDERA_LAYOUT_ENABLED)


void Node::SetLayouter(Layouter* layouter)
{
    NodeDynamicProperties::SetLayouter(*this, layouter);
    AbstractNodePointer abstractNode(this);
    Candera::Layouter::InvalidateLayout(abstractNode);
}
Layouter* Node::GetLayouter() const
{
    return NodeDynamicProperties::GetLayouter(*this);
}

void Node::SetPreferredSize(const Vector2& preferredSize)
{
    NodeDynamicProperties::SetPreferredSize(*this, preferredSize);
}

const Vector2& Node::GetPreferredSize() const
{
    return NodeDynamicProperties::GetPreferredSize(*this);
}

void Node::Layout()
{
    Candera::Internal::LayoutEvents::LayoutEvent layoutBeginEvent(Candera::Internal::LayoutEvents::LayoutEvent::LayoutValidationBegin, AbstractNodePointer(this)); // put to variable due to MULTI compile warning
    Layouter::GetEventSource().DispatchEvent(layoutBeginEvent);
    Layout3DTraverser layoutTraverser;
    layoutTraverser.Traverse(*this);
    Candera::Internal::LayoutEvents::LayoutEvent layoutEndEvent(Candera::Internal::LayoutEvents::LayoutEvent::LayoutValidationEnd, AbstractNodePointer(this)); // put to variable due to MULTI compile warning
    Layouter::GetEventSource().DispatchEvent(layoutEndEvent);
}


void Node::GetBasePoint(Vector3& basePoint) const
{
    basePoint = GetCenter();
}


const Candera::Vector3 Node::GetLayoutStartPosition() const
{
    Vector3 result;
    Vector3 min;
    Vector3 max;
    Vector3 currPos = GetPosition();
    Vector3 center = GetCenter();
    
    GetBoundingBox(min, max);
    
    result.SetX(min.GetX());
    result.SetY(min.GetY());
    return result;
}

#endif


} // namespace Candera
