//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MULTI_PASS_APPEARANCE_H)
#define CANDERA_MULTI_PASS_APPEARANCE_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/Engine3D/Core/Material.h>
#include <Candera/Engine3D/Core/RenderMode.h>
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/ShaderParamSetters/AbstractShaderParamSetter.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/Macros.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

//forward declarations
class Texture;
class Shader;

/**
 * @brief A MultiPassAppearance enables a node to be rendered multiple times with different appearance settings.
 *        A MultiPassAppearance is a dedicated Appearance with a pointer to the next MultiPassAppearance.
 *        With function SetNextPass a MultiPassAppearance chain can be linked that defines a sequence of
 *        render passes with different Appearance settings. Multi Pass Rendering is helpful, if several
 *        node-based render passes with different render states shall be blended to achieve a certain visual
 *        result like e.g. fur or blurred rendering.
 */
class MultiPassAppearance: public Appearance
{
    FEATSTD_TYPEDEF_BASE(Appearance);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(MultiPassAppearance);
        /**
         *  Creates an instance of this class.
         *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
         */
        static MultiPassAppearance::SharedPointer Create();

        /**
         *  Destructor
         */
        virtual ~MultiPassAppearance() override {}

        // Overrides Appearance::Clone. 
        virtual Appearance::SharedPointer Clone() const override;

        /**
         *  Sets the next MultiPassAppearance in list, which defines the next render pass.
         *  @param multiPassAppearance defines the MultiPassAppearance object used to render subsequent render pass.
         */
        void SetNextPass(MultiPassAppearance::SharedPointer multiPassAppearance) { m_next = multiPassAppearance; }

        /**
         *  Gets the next MultiPassAppearance in list, which defines the next render pass.
         *  @return The next multiPassAppearance, if there is any, otherwise null.
         */
        virtual MemoryManagement::SharedPointer<Appearance> GetNextPass() const override { return m_next; }


        /**
         *  Upload Textures and Shader of this and any attached MultiPassAppearance to render-device memory (VRAM).
         *  @return False if shader of this or any attached MultiPassAppearance is 0 or any associated texture or shader couldn't be uploaded to VRAM.
         *          True otherwise.
         */
        virtual bool Upload() override;

        /**
         *  Unload Textures and Shader for this and any attached MultiPassAppearance from render-device memory (VRAM).
         *  @return False if shader of this or any attached MultiPassAppearance is 0 or any associated texture or shader couldn't be unloaded from VRAM.
         *  True otherwise.
         */
        virtual bool Unload() override;

        FEATSTD_RTTI_DECLARATION();

    protected:
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::MultiPassAppearance::MultiPassAppearance, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        MultiPassAppearance();
        MultiPassAppearance(const MultiPassAppearance& appearance);

    private:
        MultiPassAppearance::SharedPointer m_next;

        // Private because only ::Create() should be used to create an instance of this class.
        MultiPassAppearance& operator = (const MultiPassAppearance& appearance);

};

/** @} */ // end of Core3D

} // namespace Candera

#endif    // CANDERA_MULTI_PASS_APPEARANCE_H

