//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MESH_H)
#define CANDERA_MESH_H

#include <Candera/Engine3D/Core/3DStringBufferAppenders.h>
#include <Candera/Engine3D/Core/Renderable.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

    namespace Internal { class QueryProperty; }

/**
 * @brief   The class Mesh represents a three-dimensional rigid body object defined by polygons. The Mesh's polygonal surface is defined by its VertexBuffer.
 *          Each Mesh must have exactly one Appearance set, in order to specify how the Mesh geometry is rendered.
 */
class Mesh:  public Renderable
{
    FEATSTD_TYPEDEF_BASE(Renderable);

    friend class SkyBox;
    friend class Internal::QueryProperty;

    public:
        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to the created Mesh object.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::Mesh::Create, CANDERA_LINT_REASON_EXPLICITHIDING)
        static Mesh* Create();

        /**
         *  Destructor
         */
        virtual ~Mesh() override;

        /**
         *  Clones this Node only.
         *  Attached Node resources like Appearance or VertexBuffer are not deep-copied but referenced.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual Mesh* Clone() const override;

        /**
         *  Sets the VertexBuffer that will be assigned to this Mesh, which defines the geometry attributes.
         *  @param vertexBuffer The VertexBuffer that is set.
         */
        void SetVertexBuffer(MemoryManagement::SharedPointer<VertexBuffer> vertexBuffer) { m_vertexBuffer = vertexBuffer; }

        /**
         *  Retrieves the VertexBuffer assigned to this Mesh, which defines the geometry attributes.
         *  @return Vertex buffer assigned to this mesh.
         */
        MemoryManagement::SharedPointer<VertexBuffer> GetVertexBuffer() const { return m_vertexBuffer; }

        /**
         *  Tests if geometry of this mesh intersects with the given Line.
         *  @param line               The intersection Line.
         *  @param distance [out]     The distance from the nearest geometry to the starting point of the Line.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this mesh, otherwise false.
         */
        virtual bool IsLineIntersectingGeometry(const Line& line, Float& distance) const override;

        /**
         *  Upload asset data attached to this Mesh to video memory.
         *  @return True if Upload of vertex buffer and appearance was successful, false if upload of any
         *          of those failed.
         */
        virtual bool Upload() override;

        /**
         *  Unload asset data attached to this Node from video memory.
         *  @return True if Unload of vertex buffer and appearance was successful, false if upload of any
         *          of those failed.
         */
        virtual bool Unload() override;

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Mesh();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Mesh);

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

        /**
         *  Renders this Mesh. This method is overridden from Node.
         */
        virtual void Render() override;

        /**
         *  Computes axis-aligned bounding box (AABB) in object coordinate space.
         *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
         *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
         *  @return true if bounding box was computed successfully and false if not because of missing geometry.
         */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

        /**
         *  Tests if geometry of this mesh intersects with a pick in absolute Viewport coordinate in pixels.
         *  @remark                   This interface is usually used for object selection in scene editors.
         *  @param camera             The view camera. (i.e. the current camera)
         *  @param x                  The x value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param y                  The y value as an absolute Viewport coordinate in pixels starting from top-left.
         *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
         *                            If no intersection is found then this param stays unchanged.
         *  @return                   True if an intersection is found with this mesh, otherwise false.
         */
        virtual bool IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const override;

    private:
        CdaDynamicProperties(Candera::Mesh, Candera::Node);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_MESH_H
