//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATERIAL_H)
#define CANDERA_MATERIAL_H

#include <Candera/Engine3D/Core/UniformBuffer.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Environment.h>
#include <Candera/Macros.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */

class Node;
class Shader;

/**
 * @brief The class Material describes the color attributes of an object's surface and is primarily used for lighting computations.
 *        The color attributes are defined as follows:
 *        Ambient: RGB color that interacts with the ambient attribute of light.
 *        Diffuse: RGBA color that interacts with the diffuse attribute of light. The alpha value of the diffuse color, defines
 *                 the alpha factor of the entire Material, supposed that alpha blending is enabled (For details see RenderMode).
 *        Emissive: RGB color that defines the self-lighting of the material. This color is visible, even when Material is unlighted.
 *                  The emissive color attribute does neither interact with any type of light source nor with other 3D objects.
 *        Specular: RGB color that interacts with the specular attribute of light.
 *        Specular Power: Defines the sharpness of a specular highlight, if lit by specular light.
 */
class Material: public CanderaObject
{
    FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(Material);

        /**
         *  Data encapsulates romable material data that can also be used as a shader uniform block.
         *  ATTENTION: Do not add, remove, or change the order of members in this struct
         *             without adapting the 'ub_Material' uniform block in the shaders.
         */
        struct UniformBlock
        {
            Color::Data ambient;  // Ambient RGB color
            Color::Data diffuse;  // Diffuse RGBA color
            Color::Data emissive; // Emissive RGB color 
            Color::Data specular; // Specular RGB color
            Float power;          // Specular power (shininess)
            Float padding[3];
        };

        /**
         *  Creates an instance of this class.
         *  @return   A MemoryManagement::SharedPointer which manages the lifetime of the instance.
         */
        static Material::SharedPointer Create();

        /**
         * Destructor
         */
        virtual ~Material() override;

        /**
        *  Create a copy of this material.
        *  @return   A Shared pointer to the copy.
        */
        Material::SharedPointer Clone() const;

        /**
         *  Set diffuse color
         *  The alpha value of the diffuse color represents the alpha value of the material and
         *  can be set with SetAlphaValue(float), separately.
         *  @param color Diffuse color of this material to be set.
         */
        void SetDiffuse(const Color& color);

        /**
         *  Retrieves the diffuse color of this material.
         *  @return The diffuse color of this material.
         */
        Color GetDiffuse() const { return Color(m_uniformBuffer.m_data.diffuse); }

        /**
         *  Set ambient color.
         *  @param color Ambient color of this material to be set.
         */
        void SetAmbient(const Color& color);

        /**
         *  Retrieves the ambient color of this material.
         *  @return The ambient color of this material.
         */
        Color GetAmbient() const { return Color(m_uniformBuffer.m_data.ambient); }

        /**
         *  Set emissive color.
         *  @param color Emissive color of this material to be set.
         */
        void SetEmissive(const Color& color);

        /**
         *  Retrieves the emissive color of this material.
         *  @return The emissive color of this material.
         */
        Color GetEmissive() const { return Color(m_uniformBuffer.m_data.emissive); }

        /**
         *  Set specular color
         *  @param color Specular color of this material to be set.
         */
        void SetSpecular(const Color& color);

        /**
         *  Retrieves the specular color of this material.
         *  @return The specular color of this material.
         */
        Color GetSpecular() const { return Color(m_uniformBuffer.m_data.specular); }

        /**
         *  Set specular power. Defines the shininess of the Material for specular highlights.
         *  @param power Specular power of this material to be set. Defines the shininess of the Material for
         *              specular highlights.
         */
        void SetSpecularPower(Float power);

        /**
         *  Retrieves the specular power. This defines the shininess of the Material for specular highlights.
         *  @return The specular power of this material.
         */
        Float GetSpecularPower() const { return m_uniformBuffer.m_data.power; }

        /**
         *  Set alpha component of diffuse color, which defines the alpha value related to the material.
         *  @param    alphaValue: [0.0F, 1.0F] 0.0F means fully transparent (invisible), 1.0F means fully opaque.
         */
        void SetAlphaValue(Float alphaValue);

        /**
         *  Retrieves the alpha component of the diffuse color, which defines the alpha value related to the material.
         *  @return The alpha component of the diffuse color.
         */
        Float GetAlphaValue() const { return m_uniformBuffer.m_data.diffuse.alpha; }

        /**
         *  Activation of this Material sets its uniform in the shader given.
         *  @param shader         receives the Material's uniform settings.
         *  @param alphaFactor    specifies an external alpha factor with which the the diffuse color's alpha value is multiplied.
         *  If the scene graph is used, this is the effective alpha value of the associated Node. Thus, the final alpha value of this
         *  Material is a product of the diffuse color's alpha component and the Node's effective alpha value (see Node::GetEffectiveAlphaValue()).
         *  @param instanceIndex  if this value is negative, the Material's shader uniforms are applied immediately.
         *                        Otherwise this Material is treated as being part of an instance array.
         *  @return True if all shader uniforms are set, false if setting any shader parameter failed.
         */
        bool Activate(const Shader& shader, Float alphaFactor = 1.0F, Int instanceIndex = -1) const;

        /**
         *  Retrieve the material data from Material, only. Helpful for setting uniform values.
         *  @return Material Data struct, data only.
         */
        const Material::UniformBlock& GetMaterialData() const { return m_uniformBuffer.m_data; }

        FEATSTD_RTTI_DECLARATION();

    private:
        struct MaterialUniformBuffer : public UniformBuffer
        {
            virtual const void* GetData() const override { return &m_data; }
            virtual SizeType GetSize() const override;

            Material::UniformBlock m_data;
        };

        mutable MaterialUniformBuffer m_uniformBuffer;

        // private because only ::Create() should be used to create an instance of this class
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Material::Material, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
        Material();
        Material(const Material& material);
        Material& operator=(const Material& material);

        // Make this class manageable by MemoryManagement::SharedPointer
        CANDERA_SHARED_POINTER_DECLARATION();

};

/** @} */ // end of Core3D

} // namespace Candera

template<> struct FeatStdTypeTrait<Candera::Material::UniformBlock> { enum { IsPodType = true }; };

#endif    // CANDERA_MATERIAL_H
