//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_LodRenderStrategy_h
#define Candera_LodRenderStrategy_h

#include <Candera/Environment.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
// Forward declarations.
class LodCriterion;
class LodNode;

/**
 *  @brief The abstract class LodRenderStrategy defines an interface to configure the LodNode's visual appearance according to the LodCriterion retrieved.
 *         The value returned by function LodCrition::GetCriterionValue is mapped onto each LOD level's lower and upper bound to identify the LOD level
 *         to be shown.
 */
class LodRenderStrategy
{
    public:

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Constructor
         */
        LodRenderStrategy() : m_lodCriterion(0) {}

        /**
         *  Destructor
         */
        virtual ~LodRenderStrategy() { m_lodCriterion = 0; }

        /**
         *  Sets LodCrition, which is used to retrieve the appropriate LOD level of LodNode object.   
         *  @param lodCriterion LodCriterion to be set.
         */
        void SetLodCriterion(LodCriterion* lodCriterion) { m_lodCriterion = lodCriterion; }
        
        /**
         *  Retrieves current LodCriterion set for this LodRenderStrategy object.
         *  @return Current LodCriterion for this LodRenderStrategy object.
         */
        const LodCriterion* GetLodCriterion() const { return m_lodCriterion; }

        /**
         *  Updates the level-of-detail levels in given LodNode object, according to the derived render strategy.
         *  This function is called by LodNode each render pass after associated scene has been activated. A concrete LodRenderStrategy 
         *  might retrieve the criterion value from its LodCriterion attached and map it onto the LOD levels boundaries in order to choose
         *  corresponding detail levels to show.   
         *  @param lodNode LOD node to be updated.
         */
        virtual void Update(LodNode& lodNode) = 0;

    private:
        LodCriterion* m_lodCriterion;   // The LOD criterion defines a use case dependent value to identify the level of detail to render.
};
 
/** @} */ // end of Core3D
 
}

#endif // Candera_LodRenderStrategy_h
