//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_LodNode_h
#define Candera_LodNode_h

#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/SceneListener.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Candera/Macros.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

// Forward declaration
class LodRenderStrategy;

/**
 * @brief LodNode represents a Node which accommodates multiple nodes, each of them representing a certain, predefined level of detail (LOD).
 *        A detail level node can be any derivation of class Node, like Mesh, Billboard, Light, Group, etc.
 *        Which LOD level node is shown is either automatically determined by the LodRenderStrategy (see LodNode::SetLodRenderStrategy) or set
 *        directly (Example: lodNodeObject->GetLodLevel(0).node->SetRenderingEnabled(true)).
 *        Each LOD level defines a visibility range with a lower and an upper bound. LOD level ranges are expected to be in monotonically increasing order.
 *        The LodRenderStrategy decides, which LOD level shall be shown in which manner. Thus, non-adjacent boundaries between contiguous LOD levels define
 *        a blend transition range, if a BlendLodRenderStrategy is attached. Analogous, non-adjacent boundaries betweencontiguous LOD levels define the hysteresis,
 *        if a DiscreteLodRenderStrategy is set.
 *        Note: A Node becomes a child of this LodNode object, if it is part of a LodLevel configuration and not already added (See SetLodLevel).
 *        Analogous, if a certain LOD level Node is overwritten, it gets removed from this LodNode object. Thus, it is recommended not to call AddChild
 *        and RemoveChild explicitly; if done deliberately, then the LOD level configuration must be synchronized manually as well.
 */
class LodNode:  public Node
{
    FEATSTD_TYPEDEF_BASE(Node);

    /**
     *  @brief The inner class LodSceneListener notifies the LodNode when the associated scene is activated
     *  and the LOD level shall be updated according to LodNode's LodRenderStrategy.
     */
    class LodSceneListener : public SceneListener
    {
        public:
            /**
             *  Constructor
             */
            LodSceneListener(): m_lodNode(0) {}

            /**
             *  Destructor
             */
            virtual ~LodSceneListener() override { m_lodNode = 0; }

            /**
             *  Sets the LodNode for the LodSceneListener.
             *  @param lodNode The LodNode that is set.
             */
            void SetLodNode(LodNode* lodNode) { m_lodNode = lodNode; }

            /**
             *  Retrieves the LodNode from the LodSceneListener.
             *  @return The LodNode from the LodSceneListener.
             */
            const LodNode* GetLodNode() const { return m_lodNode; }

            /**
             *  Overrides OnSceneActivated from SceneListener.
             */
            virtual void OnSceneActivated(Scene* scene) override;

        private:
            LodNode *m_lodNode; // LodNode to notify.
    };

    enum { DefaultLodLevelCount = 2 };  ///< 2; Default number of level of detail (lod) levels.

    public:
        /**
         *  LodLevel holds the LOD-level configuration for a certain LOD level.
         */
        struct LodLevel {
            SizeType index;         ///< Lod level index. Default value is 0.
            Node* node;         ///< Lod level Node. Default value is 0.
            Float lowerBound;   ///< Lower bound of visibility range. Default value is 0.
            Float upperBound;   ///< Upper bound of visibility range. Default value is 0.

            /**
             *  Constructor
             */
            LodLevel(): index(0), node(0), lowerBound(0.0F), upperBound(0.0F)  {}

            /**
             *  Constructor
             *  @param initIndex      The Lod level index. Default value is 0.
             *  @param initNode       The Lod level Node. Default value is 0.
             *  @param initLowerBound The lower bound of visibility range. Default value is 0.
             *  @param initUpperBound The upper bound of visibility range. Default value is 0.
             */
            LodLevel(SizeType initIndex, Node* initNode, Float initLowerBound, Float initUpperBound) :
                index(initIndex), node(initNode), lowerBound(initLowerBound), upperBound(initUpperBound)  {}
        };

        /**
         *  Creates a LodNode object. Use Dispose() to delete the instance and possible children, if any.
         *  @param lodLevelCount indicates how many LOD levels shall be pre-allocated and increases automatically,
         *  if a LOD index greater equal the predefined LOD level count is set (see SetLodLevel and GetLodLevelCount).
         *  @return Pointer to the created LOD node.
         */
        static LodNode* Create(SizeType lodLevelCount = DefaultLodLevelCount);

        /**
         * Destructs an LodNode object.
         */
        virtual ~LodNode() override;

        /**
         *  Overrides Node::Clone
         *  The clone does not receive a reference to LodRenderStrategy, as the
         *  lifetime of LodRenderStrategy is not controlled by LodNode.
         *  @return A flat copy of this LodNode.
         */
        virtual LodNode* Clone() const override;

        /**
         *  Defines the LOD configuration per level in which the LOD index 0 identifies the Node with highest detail, the LOD index 1 the LOD with
         *  lower detail and so on. If a given LOD index is greater equal the number of LOD levels (see GetLodLevelCount), the number of LOD
         *  levels increases automatically. The LOD level ranges are expected to be in monotonically increasing order.
         *  Note: SetLodLevel adds the node given as a child node automatically, if not already done manually. Analogous, if a previous node was set,
         *  it will be removed as child of this LodNode automatically, too. Thus, there is no need to call AddChild and RemoveChild explicitly; if done deliberately,
         *  then set corresponding LOD level configuration with SetLodLevel manually as well.
         *
         *  @param lodIndex defines the LOD level index to configure.
         *  @param node assigns a dedicated Node object to the LOD level specified. One Node must not be assigned to more than one LOD level.
         *  @param lowerBound defines the lower bound of the LOD levels visibility range.
         *  @param upperBound defines the upper bound of the LOD levels visibility range.
         *  @return true if the Node given has been set, false otherwise.
         */
        bool SetLodLevel(SizeType lodIndex, Node* node, Float lowerBound, Float upperBound);

        /**
         *  Retrieves the LodLevel.
         *  @param lodIndex The LodIndex for the LodLevel
         *  @return The LodLevel.
         */
        const LodLevel& GetLodLevel(SizeType lodIndex) const;

        /**
         *  Retrieves the LOD index for the criterion value given. A index is found if the criterion value matches a LOD levels
         *  visibility range defined by lower and upper bound: lower bound <= crition value <= upper bound.
         *  @param criterionValue Criterion value to look for the corresponding LOD index.
         *  @return index >= 0 if found, and value -1 if not found.
         */
        Int GetLodIndex(Float criterionValue) const;

        /**
         *  Sets the LOD render strategy, which defines which LOD levels shall be rendered according the the LodCriterion attached.
         *  @param lodRenderStrategy The LOD render strategy to be set.
         */
        void SetLodRenderStrategy(LodRenderStrategy* lodRenderStrategy) { m_lodRenderStrategy = lodRenderStrategy; }

        /**
         *  Retrieves the LOD render strategy, which defines which LOD levels shall be rendered according the the LodCriterion attached.
         *  @return The LOD render strategy of this object.
         */
        const LodRenderStrategy* GetLodRenderStrategy() const { return m_lodRenderStrategy; }

        /**
         *  Returns the number of LOD levels for this LodObject.
         *  @return Number of LOD levels for this LodObject.
         */
        SizeType GetLodLevelCount() const { return m_lodLevels.Size(); }

        /**
         *  Overrides IsRenderPrerequisiteFulfilled from Node. A LodNode object itself does not fulfill render prerequisites.
         *  @return Always false because a LOD object itself does not fulfill render prerequisites.
         */
        virtual bool IsRenderPrerequisiteFulfilled() const override { return false; }

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Deliberately protected constructor, copy constructor and assignment; clients can use Create()/Clone() to create an instance
        LodNode(SizeType lodLevelCount);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(LodNode);

        /**
         *  Updates level of detail configuration, according to attached LodRenderStrategy.
         *  This function is invoked for each render pass of the scene this LodNode object is attached to.
         */
        virtual void Update();

        /**
         *  Overrides Node::Render
         */
        virtual void Render() override {}

        /**
         *  Overrides Node::DisposeSelf
         */
        virtual void DisposeSelf() override;

        /**
         *  Overrides Node::OnAncestorAdded
         *  @param scene Scene to which LOD scene listener shall be added.
         */
        virtual void OnAncestorAdded(Scene* scene) override;

        /**
         *  Overrides Node::OnAncestorRemoved
         *  @param scene Scene to remove LOD scene listener from.
         */
        virtual void OnAncestorRemoved(Scene* scene) override;

    private:
        LodRenderStrategy* m_lodRenderStrategy; // LodRenderStrategy that determines which LOD level to show.
        LodSceneListener m_lodSceneListener;    // SceneListener which notifies this LodNode object, when to update LOD levels.
        Internal::Vector<LodLevel> m_lodLevels;           // Container for all LOD level configurations of this LodNode object.

        /**
         *  Allocates and initializes the LOD levels from index "fromIndex" to index "toIndex" with default LodLevel values (see LodLevel).
         *  @param fromIndex LOD index to start allocating and initializing.
         *  @param toIndex   LOD index to end allocating and initializing.
         */
        bool InitLodLevels(SizeType fromIndex, SizeType toIndex);

        CdaDynamicProperties(Candera::LodNode, Candera::Node);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif    // Candera_LodNode_h

