//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LodNode.h"
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/LodRenderStrategy.h>

namespace Candera {
void LodNode::LodSceneListener::OnSceneActivated(Scene* scene)
{
    if (scene != 0) {
        if (m_lodNode != 0) {
            m_lodNode->Update();
        }
    }
}

void LodNode::Update()
{
    if (m_lodRenderStrategy != 0)  {
        m_lodRenderStrategy->Update(*this);
    }
}

LodNode::LodNode(SizeType lodLevelCount) :
    Base(),
    m_lodRenderStrategy(0)
{
    m_lodSceneListener.SetLodNode(this);
    static_cast<void>(InitLodLevels(0, lodLevelCount - 1));
}

LodNode::LodNode(const Candera::LodNode &other) :
    Base(other),
    m_lodRenderStrategy(0)
{
    m_lodSceneListener.SetLodNode(this);
    static_cast<void>(InitLodLevels(0, other.GetLodLevelCount() - 1));
}

LodNode::~LodNode()
{
    Scene* scene = GetScene();
    if (scene != 0) {
        static_cast<void>(GetScene()->RemoveSceneListener(&m_lodSceneListener));
    }
    m_lodSceneListener.SetLodNode(0);
    m_lodLevels.Clear();

    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::LodNode::m_lodRenderStrategy, CANDERA_LINT_REASON_ASSOCIATION)
}

LodNode* LodNode::Create(SizeType lodLevelCount)
{
    return FEATSTD_NEW(LodNode)(lodLevelCount);
}

void LodNode::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

LodNode* LodNode::Clone() const
{
    LodNode* clone = FEATSTD_NEW(LodNode)(*this);
    return clone;
}

bool LodNode::InitLodLevels(SizeType fromIndex, SizeType toIndex)
{
    SizeType levelCount = toIndex + 1;
    if (!m_lodLevels.Reserve(levelCount)) {
        return false;
    }
    // Init lodLevels from index "fromIndex" to index "toIndex" with default lodLevels.
    LodLevel lodLevel;
    for (SizeType index = fromIndex; index < levelCount; index++) {
        lodLevel.index = static_cast<UInt>(index);
        if (!m_lodLevels.Insert(index, lodLevel)) {
            return false;
        }
    }
    return true;
}

bool LodNode::SetLodLevel(SizeType lodIndex, Node* node, Float lowerBound, Float upperBound)
{
    bool isSuccess = true;

    // Extend LodLevel container if necessary.
    SizeType lodLevelCount = GetLodLevelCount();
    if (lodIndex >= lodLevelCount ) {
        isSuccess = InitLodLevels(static_cast<Int>(lodLevelCount), static_cast<Int>(lodIndex));
    }

    if (isSuccess) {
        // Remove previous child if any.
        if (m_lodLevels[static_cast<Int>(lodIndex)].node != 0) {
            // Note: Child might have been removed manually before.
            static_cast<void>(RemoveChild(m_lodLevels[static_cast<Int>(lodIndex)].node));
            m_lodLevels[static_cast<Int>(lodIndex)].node = 0;
        }

        //Add child if not already done.
        if (! IsParentOf(node) ) {
            if (node != 0) {
                isSuccess = AddChild(node);
            }
        }

        // Set LOD level configuration.
        if (isSuccess) {
            m_lodLevels[static_cast<Int>(lodIndex)].index = lodIndex;
            m_lodLevels[static_cast<Int>(lodIndex)].node = node;
            m_lodLevels[static_cast<Int>(lodIndex)].lowerBound = lowerBound;
            m_lodLevels[static_cast<Int>(lodIndex)].upperBound = upperBound;
        }
    }
    return isSuccess;
}

const LodNode::LodLevel& LodNode::GetLodLevel(SizeType lodIndex) const {
    static const LodLevel invalidLodLevel;
    if (lodIndex < GetLodLevelCount()) {
        return m_lodLevels[lodIndex];
    }
    else {
        return invalidLodLevel;
    }
}

Int LodNode::GetLodIndex(Float criterionValue) const
{
    for(Int index= 0; index < static_cast<Int>(GetLodLevelCount()); index++) {
        if ((m_lodLevels[index].lowerBound <= criterionValue) && (criterionValue <= m_lodLevels[index].upperBound)) {
            return index;
        }
    }
    // No corresponding LOD lodIndex found for criterionValue given.
    return - 1;
}

void LodNode::OnAncestorAdded(Scene* scene)
{
    if (scene != 0) {
        static_cast<void>(scene->AddSceneListener(&m_lodSceneListener));
    }
}

void LodNode::OnAncestorRemoved(Scene* scene)
{
    if (scene != 0) {
        static_cast<void>(scene->RemoveSceneListener(&m_lodSceneListener));
    }
}

FEATSTD_RTTI_DEFINITION(LodNode, Node)
} // namespace Candera

