//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LineList.h"
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/ShaderParamSetters/GenericShaderParamSetter.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/Engine3D/Core/NodeRenderSequence.h>
#include <Candera/Engine3D/Core/VertexGeometryAccessors.h>
#include <Candera/System/Mathematics/Line.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <Candera/Macros.h>

#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

static MemoryManagement::SharedPointer<GenericShaderParamSetter> GetGenericShaderParamSetterInstance()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(MemoryManagement::SharedPointer<GenericShaderParamSetter>, gsps, GenericShaderParamSetter::Create());
    return gsps;
}

static MemoryManagement::SharedPointer<GenericShaderParamSetter> s_forceInitGenericShaderParamSetterInstance = GetGenericShaderParamSetterInstance();

LineList::LineList() :
    Base(),
    m_width(1.0F),
    m_type(Lines)
{
    // Render() changes the state of the RenderDevice. This needs to be refactored for LineList to be batchable.
    m_isBatchable = false;
}

LineList::LineList(const LineList& rhs) :
    Base(rhs),
    m_width(rhs.m_width),
    m_type(rhs.m_type)
{
}

LineList::~LineList()
{
}

LineList* LineList::Create()
{
    return FEATSTD_NEW(LineList);
}

void LineList::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

LineList* LineList::Clone() const
{
    return FEATSTD_NEW(LineList)(*this);
}

bool LineList::ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const
{
    return Math3D::ComputeBoundingBox(m_vertexBuffer, minBounds, maxBounds);
}

static NodeRenderSequence& GetNodeRenderSequence()
{
    FEATSTD_UNSYNCED_STATIC_OBJECT(NodeRenderSequence, renderSequence, GetGenericShaderParamSetterInstance());
    return renderSequence;
}

static NodeRenderSequence& s_forceInitRenderSequence = GetNodeRenderSequence();

void LineList::Render()
{
    NodeRenderSequence& renderSequence = GetNodeRenderSequence();

    if ( !IsRenderPrerequisiteFulfilled() ) {
        FEATSTD_LOG_ERROR("LineListRender precondition is not fulfilled for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
        return;
    }
    if (!RenderDevice::SetLineWidth(m_width)) {
        FEATSTD_LOG_ERROR("LineList width failed to set for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
        return;
    }
    renderSequence.Render(*this, m_vertexBuffer);
}

void LineList::SetLineType(LineType lineType)
{
    m_type = lineType;
    SetVertexBufferLineType();
}

void LineList::SetVertexBuffer(MemoryManagement::SharedPointer<VertexBuffer> vertexBuffer)
{
    m_vertexBuffer = vertexBuffer;
    SetVertexBufferLineType();
}

void LineList::SetVertexBufferLineType()
{
    if (m_vertexBuffer != 0) {
        switch(m_type) {
            case Lines:
                m_vertexBuffer->SetPrimitiveType(VertexBuffer::Lines);
                break;
            case LineLoop:
                m_vertexBuffer->SetPrimitiveType(VertexBuffer::LineLoop);
                break;
            case LineStrip:
                m_vertexBuffer->SetPrimitiveType(VertexBuffer::LineStrip);
                break;
            default:
                break;
        }
    }
}

bool LineList::IsLineIntersectingGeometry(const Line& line, Float& distance) const
{
    if (!IsIntersectionTestEnabled()) {
        return false;
    }

    if ((m_vertexBuffer == 0) || (m_vertexBuffer->GetVertexGeometry() == 0) ||
        (m_vertexBuffer->GetVertexGeometry()->GetVertexCount() == 0)) {
        FEATSTD_LOG_ERROR("Is line intersecting geometry failed, invalid vertexbuffer.");
        return false;
    }

    const VertexGeometry& vertexGeometry = *m_vertexBuffer->GetVertexGeometry();
    ElementFormatAccessor elementFormatAccessor(vertexGeometry);
    const VertexGeometry::VertexElementFormat* positionElement = elementFormatAccessor.GetElementFormatByUsageAndIndex(VertexGeometry::Position, 0);
    if (positionElement == 0) {
        return false;
    }

    bool intersection = false;
    Float min_distance = Math::MaxFloat();

    const Matrix4& worldTransform = GetWorldTransform();
    const VertexAccessor vertexAccessor(vertexGeometry);
    for (VertexBuffer::PrimitiveIterator lineIterator = m_vertexBuffer->GetPrimitiveIterator(); lineIterator.IsValid(); ++lineIterator) {
        const VertexBuffer::Primitive& currentLine = *lineIterator;
        const Float* position1 = FeatStd::Internal::PointerAdd<const Float*>(vertexAccessor.GetVertex(currentLine.m_index[0]), positionElement->Offset);
        const Float* position2 = FeatStd::Internal::PointerAdd<const Float*>(vertexAccessor.GetVertex(currentLine.m_index[1]), positionElement->Offset);

        Vector3 lineVertex1(position1[0], position1[1], position1[2]);
        Vector3 lineVertex2(position2[0], position2[1], position2[2]);

        lineVertex1.TransformCoordinate(worldTransform);
        lineVertex2.TransformCoordinate(worldTransform);
        Line line1(lineVertex1, lineVertex2);

        Vector3 hitPosition;
        // check intersection
        if (Math3D::LineLineIntersection(line1,
            line,
            m_width / 2.0F,
            hitPosition)) {
            // store smallest distance
            Float temp_distance = hitPosition.GetLength();//line.GetStart().GetDistanceTo(hitPosition);
            if (temp_distance < min_distance) {
                min_distance = temp_distance;
                distance = min_distance;
            }
            intersection = true;
            // do not break here as another triangle could have also an intersection but another distance
        }
    }
    return intersection;
}

bool LineList::IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const
{
    return Math3D::IsPickIntersectingGeometry(*this, camera, x, y, distance);
}

bool LineList::Upload()
{
    bool success = true;

    // Upload vertex buffer if not null.
    if (m_vertexBuffer != 0) {
        if( ! m_vertexBuffer->Upload()) {
            FEATSTD_LOG_ERROR("Upload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    // Upload appearance if not null.
    MemoryManagement::SharedPointer<Appearance> app = GetAppearance();
    if (!app.PointsToNull()) {
        if (!app->Upload()) {
            FEATSTD_LOG_ERROR("Upload Appearance failed for Node:\"%s\"", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    return success;
}

bool LineList::Unload()
{
    bool success = true;

    // Unload vertex buffer if not null.
    if (m_vertexBuffer != 0) {
        if ( ! m_vertexBuffer->Unload()) {
            FEATSTD_LOG_ERROR("Unload VertexBuffer failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    // Unload appearance if not null.
    MemoryManagement::SharedPointer<Appearance> app = GetAppearance();
    if (!app.PointsToNull()) {
        if (!app->Unload()) {
            FEATSTD_LOG_ERROR("Unload Appearance failed for Node:\"%s\".", (GetName() == 0) ? "" : GetName());
            success = false;
        }
    }

    return success;
}

FEATSTD_RTTI_DEFINITION(LineList, Renderable)
} // namespace Candera
