//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_LIGHT_H)
#define CANDERA_LIGHT_H

#include <Candera/EngineBase/Common/Color.h>
#include <Candera/Engine3D/Core/3DStringBufferAppenders.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/UniformBuffer.h>
#include <Candera/System/Container/LinkedList.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/Macros.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief   A Node that represents different kinds of light sources.
 *          Light sources are used to determine the color of an object's surface according to its Material attributes.
 *          4 types of light sources are available, described below in the order of their computing complexity:
 *          Ambient light: illuminates the scene from all directions.
 *                         The light intensity is the same everywhere in the scene.
 *                         Position and direction of an ambient light source have no effect.
 *
 *          Directional light: corresponds to sunlight in the real world.
 *                             Illuminates all objects in the scene from the same direction, and with a constant intensity.
 *                             Position and attenuation of a directional light source has no effect.
 *                             Thus, directional light is not transformed into world space.
 *
 *          Point light: casts equal omni directional illumination from the position of the light source.
 *                       The intensity of light coming from a point light source can diminish with distance.
 *                       The direction of a point light has no effect.
 *
 *          Spot light: casts a light cone from a position centered around a direction.
 *                      A spot light has both a direction and a position.
 *
 *
 *          The type of a light source can be changed at any time. This might be useful for switching to a simpler lighting model.
 *          Illumination is enabled, if light has rendering enabled and if the node is in the same scope as the light.
 */
class Light : public Node
{
    friend class RenderDevice;
    friend class Renderer;
    friend class Scene;

    FEATSTD_TYPEDEF_BASE(Node);

    public:
        /**
         *  The light Type.
         */
        enum Type
        {
            Ambient = 0,        ///< An ambient light: Light intensity is the same everywhere in the scene.
            Directional = 1,    ///< A directional light: All objects illuminated with the same light direction and intensity.
            Point = 2,          ///< A point light: Omni-directional illumination from the position of the light source.
            Spot = 3            ///< A spot light: Casts a cone of light from a position in a specific direction.
        };

        /**
         * The coordinate space in which the illumination is calculated.
         */
        enum CoordinateSpace
        {
            Object, ///< Light is transformed into object space to calculate vertex illumination.
            World   ///< Vertex and normal are transformed into world space to calculate vertex illumination.
        };

        /**
         *  The attenuation coefficients for this Light.
         */
        struct Attenuation {
            Float constant;     ///< Independent attenuation factor, usually multiplied as is with the light intensity.
            Float linear;       ///< Linear attenuation factor, usually multiplied with the distance of the receiver to the light source.
            Float quadratic;    ///< Quadratic attenuation factor, usually multiplied with the square distance of the receiver to the light source.
        };

        /**
         *  Light data structure that is used as a shader uniform block.
         *  ATTENTION: Do not add, remove, or change the order of members in this struct
         *             without adapting the 'ub_Lights' uniform block in the shaders.
         */
        struct UniformBlock
        {
            Int32 type;                 // Type of Light (0: Ambient, 1: Diffuse, 2: Point, 3: Spot)
            Int32 enabled;              // Light has been enabled by application (1 = true)
            Float spotCosCutoff;        // Cosine of spot light cutoff angle
            Float spotExponent;         // Spotlight exponent
            Float range;                // Range for point and spot lights; intensity is 0 for positions farther away than range
            Float padding0[3];          // padding to align the following Vector3
            Vector3 position;           // Position of light in model space
            Float padding1;             // padding to align the following Vector3
            Vector3 direction;          // Normalized light direction in model space
            Float padding2;             // padding to align the following Vector3
            Vector3 halfplane;          // Normalized half-plane vector
            Float padding3;             // padding to align the following Vector4
            Vector4 attenuation;        // Attenuation: constant, linear, quadratic, enabled (1.0 = true, 0.0 = false);
            Color::Data ambient;        // Ambient light component
            Color::Data diffuse;        // Diffuse light component
            Color::Data specular;       // Specular light component
            Vector3 cameraLookAtVector; // Camera look at vector.
            Float padding4;             // padding to align start of next UniformBlock element in an array
        };

        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to the created Light object.
         */
        static Light* Create();

        /**
         *  Destructor
         */
        virtual ~Light() override;

        /**
         *  Clones this Node only.
         *  Attached Node resources like Appearance are not deep-copied but referenced.
         *  @return  The pointer to the cloned Node if successful, otherwise NULL.
         */
        virtual Light* Clone() const override;

        /**
         *  Set type of light. This can be ambient, directional, point, or spot light.
         *  @param type Type of the light to be set.
         */
        void SetType(Type type);

        /**
         *  Retrieves the type of the light.
         *  @return The type of the light.
         */
        Type GetType() const { return static_cast<Type>(m_uniformBlock.type); }

        /**
         *  Set ambient color. Relates to the Material's ambient color. The alpha value is ignored.
         *  @param color Ambient color to be set. The alpha value is ignored.
         */
        void SetAmbient(const Color& color);

        /**
         *  Retrieves the ambient color. Relates to the Material's ambient color. The alpha value is ignored.
         *  @return The ambient color.
         */
        Color GetAmbient() const { return Color(m_uniformBlock.ambient); }

        /**
         *  Set diffuse color. Relates to the Material's diffuse color. The alpha value is ignored.
         *  @param color Diffuse color to be set. The alpha value is ignored.
         */
        void SetDiffuse(const Color& color);

        /**
         *  Retrieves the diffuse color. Relates to the Material's diffuse color. The alpha value is ignored.
         *  @return The diffuse color.
         */
        Color GetDiffuse() const { return Color(m_uniformBlock.diffuse); }

        /**
         *  Set specular color. Relates to the Material's specular color. The alpha value is ignored.
         *  @param color Specular color to be set. The alpha value is ignored.
         */
        void SetSpecular(const Color& color);

        /**
         *  Retrieves the specular color. Relates to the Material's specular color. The alpha value is ignored.
         *  @return The specular color.
         */
        Color GetSpecular() const { return m_uniformBlock.specular; }

        /**
         *  Set direction. The default direction of the light is along the negative Z axis of the Light's local coordinate system.
         *  The direction has no effect on Lights of type ambient or point.
         *  @param direction Vector3 describing the direction of this light.
         */
        void SetDirection(const Vector3& direction);

        /**
         *  Retrieves the direction of the light.
         *  The default direction of the light is along the negative Z axis of the Light's local coordinate system.
         *  The direction has no effect on Lights of type ambient or point.
         *  @return The direction of the light.
         */
        Vector3 GetDirection() const { return m_direction; }

        /**
         *  Retrieves the direction vector in world space
         *  Multiplies the direction vector with the world rotation matrix of this Light
         *  @return Direction of this light in world space.
         */
        Vector3 GetWorldDirection() const;

        /**
         *  Set range of light emission from Light source.
         *  Default range is 1000.0F;
         *  @param range Range of light emission to be set. Default range is 1000.f.
         */
        void SetRange(Float range);

        /**
         *  Retrieves the rang of light emission from Light source.
         *  Default range is 1000.0F;
         *  @return The range of light emission.
         */
        Float GetRange() const { return m_uniformBlock.range; }

        /**
         *  Set attenuation coefficients for this Light.
         *  The attenuation factor is: 1 / (constant + (linear * distance) + (quadratic * distance^2)).
         *  The default attenuation coefficients are (1, 0, 0), resulting in no attenuation.
         *  @param constant  Constant part of attenuation, see formula in method description.
         *  @param linear    Linear part of attenuation, see formula in method description.
         *  @param quadratic Quadratic part of attenuation, see formula in method description.
         */
        void SetAttenuation(Float constant, Float linear, Float quadratic);

        /**
         *  Retrieves the attenuation coefficients for this Light.
         *  The attenuation factor is: 1 / (constant + (linear * distance) + (quadratic * distance^2)).
         *  The default attenuation coefficients are (1, 0, 0), resulting in no attenuation.
         *  @return The attenuation coefficients.
         */
        const Attenuation& GetAttenuation() const { return *(FeatStd::Internal::PointerToPointer<const Attenuation*>(&(m_uniformBlock.attenuation))); }

        /**
         *  Set spot angle of the cone in degrees. The lighting is restricted to the cone set.
         *  The spot angle has an effect on Lights of type spotlight, only. The value must lie within [0.0F, 90.0F].
         *  Default value is 45 degrees.
         *  @param angle Angle of spot cone of spotlight in degrees.
         */
        void SetSpotAngle(Float angle);

        /**
         *  Retrieves the spot angle of the cone in degrees. The lighting is restricted to the cone set.
         *  The spot angle has an effect on Lights of type spotlight, only. The value must lie within [0.0F, 90.0F].
         *  Default value is 45 degrees.
         *  @return The spot angle of the cone in degrees.
         */
        Float GetSpotAngle() const { return m_spotAngle; }

        /**
         *  Set the spot exponent. The spot exponent controls the distribution of this Light within the spot cone.
         *  Larger values lead to a more concentrated cone. The value must lie within [0.0F, 128.0F]
         *  The default spot exponent is 0.0, resulting in a uniform light distribution.
         *  @param exponent Spot exponent to be set.
         */
        void SetSpotExponent(Float exponent);

        /**
         *  Retrieves the spot exponent. he spot exponent controls the distribution of this Light within the spot cone.
         *  Larger values lead to a more concentrated cone. The value must lie within [0.0F, 128.0F]
         *  The default spot exponent is 0.0, resulting in a uniform light distribution.
         *  @return The spot exponent.
         */
        Float GetSpotExponent() const { return m_uniformBlock.spotExponent; }

        /**
         *  Enable or disable attenuation of Light.
         *  If disabled, objects illumination appears or disappears abruptly as they enter and exit the light's range.
         *  Has no effect on ambient and directional Light.
         *  Attenuation is disabled by default.
         *  @param enable Enables(true)/Disables(false) attenuation of Light.
         */
        void SetAttenuationEnabled(bool enable) { m_uniformBlock.attenuation[3] = enable ? 1.0F : 0.0F; }

        /**
         *  Retrieves whether the attenuation of the Light is enabled or not.
         *  If disabled, objects illumination appears or disappears abruptly as they enter and exit the light's range.
         *  Has no effect on ambient and directional Light.
         *  Attenuation is disabled by default.
         *  @return The attenuation of the Light, true if enabled otherwise false.
         */
        bool IsAttenuationEnabled() const { return (m_uniformBlock.attenuation[3] != 0.0F); }

        /**
         *  Sets the coordinate space in which the lighting calculations are performed.
         *  Whereas in object-space lighting the light is transformed into object's coordinate space in world-space
         *  lighting the vertex and normal are transformed into world space to perform illumination.
         *  Object space is the default setting as it is less computation intensive.
         *  Nevertheless, to assure accurate illumination of non-uniform scaled objects, lighting in world space is indicated.
         *  Static property as the light coordinate space has to be set per shader and shall not be mixed.
         *  @param coordSpace Coordinate space to be set, lighting calculations are performed in the set coordinate space.
         */
        static void SetCoordinateSpace(Light::CoordinateSpace coordSpace) { m_coordSpace = coordSpace; }

        /**
         *  Retrieves the coordinate space in which the lighting calculations are performed.
         *  Whereas in object-space lighting the light is transformed into object's coordinate space in world-space
         *  lighting the vertex and normal are transformed into world space to perform illumination.
         *  Object space is the default setting as it is less computation intensive.
         *  Nevertheless, to assure accurate illumination of non-uniform scaled objects, lighting in world space is indicated.
         *  Static property as the light coordinate space has to be set per shader and shall not be mixed.
         *  @return The coordinate space.
         */
         static Light::CoordinateSpace GetCoordinateSpace() { return m_coordSpace; }

        /**
         *  Activate light and propagate light parameters to shader
         *  @param shader:    shader that receives and processes light properties
         *  @param node:        node to lit
         *  @param index:    index of light in LightList to process
         *  @return Whether shader uniforms could be set (true) or not(false). Light is activated when shaders have it's values to compute.
         */
        bool Activate(const Shader &shader, const Node& node, UInt index) const;

        /**
         *  Overrides IsRenderPreconditionFulfilled from Node. A Light object itself cannot be rendered.
         *  @return Always false because light's themselves are not rendered.
         */
        virtual bool IsRenderPrerequisiteFulfilled() const override { return false; }

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Light();
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Light);

        /**
         *  Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

        /**
         *  Overrides abstract Render from Node.
         */
        virtual void Render() override {}

        /**
         *  Overrides OnAncestorAdded from Node.
         *  @param scene Scene to add light to.
         */
        virtual void OnAncestorAdded(Scene* scene) override;

        /**
         *  Overrides OnAncestorRemoved from Node.
         *  @param scene Scene to remove light from.
         */       
        virtual void OnAncestorRemoved(Scene* scene) override;

    private:

        mutable UniformBlock m_uniformBlock;
        mutable bool m_isDataDirty;

        Float m_spotAngle;           // Angle of the cone in degrees
        Vector3 m_direction;         // Normalized light direction in Light's local space
        static CoordinateSpace m_coordSpace; // Coordinate space used for lighting
        Candera::Internal::LinkedListNode<Light> LightListNode; // Intrusive linked list node for membership in Scene::LightList

        bool SetUniforms(const Shader &shader, const Node& node, UInt index) const;
        bool SetCommonUniforms(const Shader& shader, const Node& node,  UInt index) const;
        bool SetDirectionalUniforms(const Shader& shader, const Node& node, UInt index) const;
        bool SetPointUniforms(const Shader& shader, const Node& node, UInt index) const;
        bool SetSpotUniforms(const Shader& shader, const Node& node, UInt index) const;
        void SetDirectionData(const Vector3& direction) const;
        void SetHalfPlane(const Vector3& halfplane) const;

        Float GetCosineSpotAngle() const { return m_uniformBlock.spotCosCutoff; }

        static Shader::UniformCacheHandle s_uniformCacheHandle[];
        static const ShaderParamNames::UniformSemantic* s_uniformSemantics[];

        CdaDynamicProperties(Candera::Light, Candera::Node);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

template<> struct FeatStdTypeTrait<Candera::Light::UniformBlock> { enum { IsPodType = true }; };

#endif    // CANDERA_LIGHT_H

