//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Light.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/ShaderParamNames.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Mathematics/Matrix3.h>
#include <Candera/System/Mathematics/Vector4.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Macros.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

Light::CoordinateSpace Light::m_coordSpace = Light::Object;

enum UniformCacheIndices {
    LightType = 0,
    LightEnabled,
    LightAmbient,
    LightDiffuse,
    LightSpecular,
    LightDirection,
    LightHalfplane,
    LightPosition,
    LightCameraLookAtVector,
    LightAttenuation,
    LightRange,
    LightSpotCosCutoff,
    LightSpotExponent,

    NormalModelMatrix3,
    ModelMatrix4,

    UniformCacheIndicesCount
};

static const ShaderParamNames::UniformSemantic UniformCacheParams[UniformCacheIndicesCount] = {
    ShaderParamNames::LightType,
    ShaderParamNames::LightEnabled,
    ShaderParamNames::LightAmbient,
    ShaderParamNames::LightDiffuse,
    ShaderParamNames::LightSpecular,
    ShaderParamNames::LightDirection,
    ShaderParamNames::LightHalfplane,
    ShaderParamNames::LightPosition,
    ShaderParamNames::LightCameraLookAtVector,
    ShaderParamNames::LightAttenuation,
    ShaderParamNames::LightRange,
    ShaderParamNames::LightSpotCosCutoff,
    ShaderParamNames::LightSpotExponent,
    ShaderParamNames::NormalModelMatrix3,
    ShaderParamNames::ModelMatrix4
};

#define CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(LightIndex)                                                                      \
static const ShaderParamNames::UniformSemantic FEATSTD_CONCAT2(UniformCacheParams,LightIndex)[UniformCacheIndicesCount] = { \
    FEATSTD_CONCAT2(ShaderParamNames::LightType,LightIndex),                                                                \
    FEATSTD_CONCAT2(ShaderParamNames::LightEnabled,LightIndex),                                                             \
    FEATSTD_CONCAT2(ShaderParamNames::LightAmbient,LightIndex),                                                             \
    FEATSTD_CONCAT2(ShaderParamNames::LightDiffuse,LightIndex),                                                             \
    FEATSTD_CONCAT2(ShaderParamNames::LightSpecular,LightIndex),                                                            \
    FEATSTD_CONCAT2(ShaderParamNames::LightDirection,LightIndex),                                                           \
    FEATSTD_CONCAT2(ShaderParamNames::LightHalfplane,LightIndex),                                                           \
    FEATSTD_CONCAT2(ShaderParamNames::LightPosition,LightIndex),                                                            \
    FEATSTD_CONCAT2(ShaderParamNames::LightCameraLookAtVector,LightIndex),                                                  \
    FEATSTD_CONCAT2(ShaderParamNames::LightAttenuation,LightIndex),                                                         \
    FEATSTD_CONCAT2(ShaderParamNames::LightRange,LightIndex),                                                               \
    FEATSTD_CONCAT2(ShaderParamNames::LightSpotCosCutoff,LightIndex),                                                       \
    FEATSTD_CONCAT2(ShaderParamNames::LightSpotExponent,LightIndex),                                                        \
                                                                                                                            \
    ShaderParamNames::NormalModelMatrix3,                                                                                   \
    ShaderParamNames::ModelMatrix4                                                                                          \
};

CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(1)
CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(2)
CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(3)
CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(4)
CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(5)
CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(6)
CANDERA_LIGHT_UNIFORM_CACHE_PARAMS(7)

Shader::UniformCacheHandle Light::s_uniformCacheHandle[CANDERA_MAX_LIGHTS_COUNT];
const ShaderParamNames::UniformSemantic* Light::s_uniformSemantics[CANDERA_MAX_LIGHTS_COUNT] = {
    UniformCacheParams,
    UniformCacheParams1,
    UniformCacheParams2,
    UniformCacheParams3,
    UniformCacheParams4,
    UniformCacheParams5,
    UniformCacheParams6,
    UniformCacheParams7
};


Light::Light() :
    Base(),
    m_isDataDirty(true),
    m_spotAngle(45.0F),
    m_direction(Vector3(0.0F,0.0F,-1.0F))
{
    m_uniformBlock.type = static_cast<Int32>(Directional);
    m_uniformBlock.ambient = Color(0.5F, 0.5F, 0.5F).GetData();
    m_uniformBlock.diffuse = Color(0.5F, 0.5F, 0.5F).GetData();
    m_uniformBlock.specular = Color(1.0F, 1.0F, 1.0F).GetData();
    m_uniformBlock.range = 1000.0F;
    m_uniformBlock.spotCosCutoff = Math::Cosine(Math::DegreeToRadian(GetSpotAngle()));
    m_uniformBlock.spotExponent = 0.0F;
    m_uniformBlock.attenuation = Vector4(1.0F, 0.0F, 0.0F, 0.0F); // (constant, linear, quadratic, disabled)
}

Light* Light::Create()
{
    return FEATSTD_NEW(Light);
}

void Light::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

Light::~Light()
{
}

Light* Light::Clone() const
{
    return FEATSTD_NEW(Light)(*this);
}

Light::Light(const Candera::Light &other) :
    Base(other),
    m_uniformBlock(other.m_uniformBlock),
    m_isDataDirty(other.m_isDataDirty),
    m_spotAngle(other.m_spotAngle),
    m_direction(other.m_direction)
{
}

void Light::SetType(Type type)
{
    Int32 lightType = static_cast<Int32>(type);
    if (m_uniformBlock.type != lightType) {
        m_uniformBlock.type = lightType;
        m_isDataDirty = true;
    }
}

void Light::SetAmbient(const Color& color)
{
    if (m_uniformBlock.ambient != color.GetData()) {
        m_uniformBlock.ambient = color.GetData();
        m_isDataDirty = true;
    }
}

void Light::SetDiffuse(const Color& color)
{
    if (m_uniformBlock.diffuse != color.GetData()) {
        m_uniformBlock.diffuse = color.GetData();
        m_isDataDirty = true;
    }
}

void Light::SetSpecular(const Color& color)
{
    if (m_uniformBlock.specular != color.GetData()) {
        m_uniformBlock.specular = color.GetData();
        m_isDataDirty = true;
    }
}

void Light::SetDirection(const Vector3& direction)
{
    m_direction = direction;
    static_cast<void>(m_direction.Normalize());
}

Vector3 Light::GetWorldDirection() const
{
    Vector3 direction = GetDirection();
    //Rotate local direction vector by world rotation.
    direction.TransformCoordinate(GetWorldRotation());
    return direction;
}

void Light::SetRange(Float range)
{
    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
    if (m_uniformBlock.range != range) {
        m_uniformBlock.range = range;
        m_isDataDirty = true;
    }
}

void Light::SetAttenuation(Float constant, Float linear, Float quadratic)
{
    if ((m_uniformBlock.attenuation[0] != constant) ||
        (m_uniformBlock.attenuation[1] != linear) ||
        (m_uniformBlock.attenuation[2] != quadratic)) {
        m_uniformBlock.attenuation[0] = constant;
        m_uniformBlock.attenuation[1] = linear;
        m_uniformBlock.attenuation[2] = quadratic;
        m_isDataDirty = true;
    }
}

void Light::SetSpotAngle(Float angle)
{
    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
    if (m_spotAngle != angle) {
        m_spotAngle = angle;
        m_uniformBlock.spotCosCutoff = Math::Cosine(Math::DegreeToRadian(GetSpotAngle()));
        m_isDataDirty = true;
    }
}

void Light::SetSpotExponent(Float exponent)
{
    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
    if (m_uniformBlock.spotExponent != exponent) {
        m_uniformBlock.spotExponent = exponent;
        m_isDataDirty = true;
    }
}

bool Light::SetCommonUniforms(const Shader& shader, const Node& node, UInt index) const
{
    Int uniformLocation;
    bool finalResult = true;
    const Type type = GetType();

    const Int32 enabled = (IsEffectiveRenderingEnabled() && node.IsInScopeOf(GetScopeMask())) ? 1 : 0;
    if (m_uniformBlock.enabled != enabled) {
        m_uniformBlock.enabled = enabled;
        m_isDataDirty = true;
    }

    // 1. Set type of light source
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightType, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::Integer>(uniformLocation, &type);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set common uniform type failed.");
            finalResult = false;
        }
    }

    // 2. Set light enabled or disabled
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightEnabled, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::Integer>(uniformLocation, &enabled);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set common uniform enabled failed.");
            finalResult = false;
        }
    }

    // 3. Set ambient color component
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightAmbient, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec4>(uniformLocation, &GetAmbient().GetData());
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set common uniform ambient color failed.");
            finalResult = false;
        }
    }

    return finalResult;
}

bool Light::SetDirectionalUniforms(const Shader& shader, const Node& node, UInt index) const
{
    Int uniformLocation;
    bool finalResult = true;
    Matrix4  inverseWorldRotation;

    // 1. Set diffuse color component
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightDiffuse, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec4>(uniformLocation, &GetDiffuse().GetData());
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set directional uniform diffuse color failed.");
            finalResult = false;
        }
    }

    // 2. Set specular color component
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightSpecular, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec4>(uniformLocation, &GetSpecular().GetData());
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set directional uniform specular color failed.");
            finalResult = false;
        }
    }

    // 3. Set light direction

    // Update light direction for uniform buffer.
    Vector3 lightDir = GetDirection();
    if (m_coordSpace == Object) {
        // Transform light direction into object space (This saves calculating each normal attribute into world space)
        inverseWorldRotation = node.GetWorldRotation();
        inverseWorldRotation.Inverse();

        lightDir.TransformCoordinate(inverseWorldRotation);
        static_cast<void>(lightDir.Normalize());
    }

    // Invert light direction, as shader expects direction to light, not the direction of light
    SetDirectionData(-lightDir);

    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightDirection, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec3>(uniformLocation, &m_uniformBlock.direction);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set directional uniform dir failed.");
            finalResult = false;
        }
    }

    // 4. Set half vector

    // Update half plane for uniform buffer.

    // Calculate half vector for specular highlighting: ||vector to Camera + vector to light position (== light direction)||
    // First calculate normalized vector from object to camera
    Vector3 halfVec = RenderDevice::GetActiveCamera()->GetWorldPosition() - node.GetWorldPosition();
    static_cast<void>(halfVec.Normalize());
    // Then add normalized vector to light source, which is exactly the inverted light direction. Thus, simply subtract the light direction.
    halfVec -= GetDirection();
    if (m_coordSpace == Object) {
        // Transform calculated half vector into object space
        halfVec.TransformCoordinate(inverseWorldRotation);
    }
    static_cast<void>(halfVec.Normalize());
    SetHalfPlane(halfVec);

    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightHalfplane, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec3>(uniformLocation, &m_uniformBlock.halfplane);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set directional uniform half plane failed.");
            finalResult = false;
        }
    }

    // 5. Set Normal transformation matrix if shader expects normals in world coordinate space
    if (m_coordSpace == World) { // it's faster to check for world space first
        if (shader.GetUniformLocation(s_uniformCacheHandle[index], NormalModelMatrix3, uniformLocation)) {
            Matrix3 normalTransform = node.GetWorldScale() * node.GetWorldRotation();
            normalTransform.Inverse();
            normalTransform.Transpose();
            const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatMat3>(uniformLocation, normalTransform.GetData());
            if (!tmpResult) {
                FEATSTD_LOG_ERROR("Set directional uniform normal transform failed.");
                finalResult = false;
            }
        }
    }

    return finalResult;
}

bool Light::SetPointUniforms(const Shader& shader, const Node& node, UInt index) const
{
    Int uniformLocation;
    bool finalResult = true;

    // 1. Set diffuse color component
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightDiffuse, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec4>(uniformLocation, &GetDiffuse().GetData());
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set point uniform diffuse color failed.");
            finalResult = false;
        }
    }

    // 2. Set specular color component
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightSpecular, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec4>(uniformLocation, &GetSpecular().GetData());
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set point uniform specular color failed.");
            finalResult = false;
        }
    }

    Vector3 lightPos = GetWorldPosition();
    if (m_coordSpace == Object) {
        // transform light position into object space
        Matrix4 inverseWorldTransform = node.GetWorldTransform();
        inverseWorldTransform.Inverse();
        lightPos.TransformCoordinate(inverseWorldTransform);
    }

    if (m_uniformBlock.position != lightPos) {
        m_uniformBlock.position = lightPos;
        m_isDataDirty = true;
    }

    // 3. Set light position
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightPosition, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec3>(uniformLocation, lightPos.GetData());
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set point uniform light position failed.");
            finalResult = false;
        }
    }

    // Transform camera direction into object space
    Vector3 camVec;
    Matrix4 viewMat = RenderDevice::GetActiveCamera()->GetViewMatrix();
    //Get look-at-vector of camera
    camVec.SetX(viewMat.Get(0, 2));
    camVec.SetY(viewMat.Get(1, 2));
    camVec.SetZ(viewMat.Get(2, 2));
    if (m_coordSpace == Object) {
        // transform camera look-at-vector into object space
        Matrix4 inverseWorldRotation = node.GetWorldRotation();
        inverseWorldRotation.Inverse();
        camVec.TransformCoordinate(inverseWorldRotation);
    }
    // camVec.Normalize(); isn't executed any more because halfVector calculation needs lightDirection and CamDirection not normalized.
    if (m_uniformBlock.cameraLookAtVector != camVec) {
        m_uniformBlock.cameraLookAtVector = camVec;
        m_isDataDirty = true;
    }

    // 4. Set camera direction
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightCameraLookAtVector, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec3>(uniformLocation, camVec.GetData());
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set point uniform camera direction failed.");
            finalResult = false;
        }
    }

    // 5. Set attenuation of light
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightAttenuation, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec4>(uniformLocation, &m_uniformBlock.attenuation);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set point uniform attenuation failed.");
            finalResult = false;
        }
    }

    // 6. Set range of light
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightRange, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::Float>(uniformLocation, &m_uniformBlock.range);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set point uniform range failed.");
            finalResult = false;
        }
    }

    // 7. Set Normal transformation matrix if shader expects normals in world coordinate space
    if (m_coordSpace == World) {
        if (shader.GetUniformLocation(s_uniformCacheHandle[index], NormalModelMatrix3, uniformLocation)) {
            Matrix3  normalTransform = node.GetWorldScale() * node.GetWorldRotation();
            normalTransform.Inverse();
            normalTransform.Transpose();
            const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatMat3>(uniformLocation, normalTransform.GetData());
            if (!tmpResult) {
                FEATSTD_LOG_ERROR("Set point uniform normal transform failed.");
                finalResult = false;
            }
        }

        //Set Model Matrix
        if (shader.GetUniformLocation(s_uniformCacheHandle[index], ModelMatrix4, uniformLocation)) {
            Matrix4 mMatrix = node.GetWorldTransform();
            const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatMat4>(uniformLocation, mMatrix.GetData());
            if (!tmpResult) {
                FEATSTD_LOG_ERROR("Set point uniform model matrix failed.");
                finalResult = false;
            }
        }
    }

    return finalResult;
}

bool Light::SetSpotUniforms(const Shader& shader, const Node& node, UInt index) const
{
    Int uniformLocation;
    bool finalResult = true;
    Float tmpFloat;

    // Update light direction for uniform buffer.
    Vector3 lightVec = GetDirection();
    // Rotate light direction by world rotation of this light.
    lightVec.TransformCoordinate(GetWorldRotation());
    static_cast<void>(lightVec.Normalize());
    if (m_coordSpace == Object) {
        // Transform light direction into object space (This saves calculating each normal attribute into world space)
        Matrix4  inverseWorldRotation = node.GetWorldRotation();
        inverseWorldRotation.Inverse();
        lightVec.TransformCoordinate(inverseWorldRotation);
        static_cast<void>(lightVec.Normalize());
    }
    //Invert light direction which describes light direction of light, but shader expects direction to light.
    SetDirectionData(-lightVec);

    // 1. Set Light direction
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightDirection, uniformLocation)) {
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::FloatVec3>(uniformLocation, &m_uniformBlock.direction);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set spot uniform light direction failed.");
            finalResult = false;
        }
    }

    // 2. Set cosine of spot angle
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightSpotCosCutoff, uniformLocation)) {
        tmpFloat = GetCosineSpotAngle();
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::Float>(uniformLocation, &tmpFloat);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set uniform light spot cos cutoff failed.");
            finalResult = false;
        }
    }

    // 3. Set spot exponent
    if (shader.GetUniformLocation(s_uniformCacheHandle[index], LightSpotExponent, uniformLocation)) {
        tmpFloat = GetSpotExponent();
        const bool tmpResult = Renderer::SetUniformInRenderDevice<Shader::Float>(uniformLocation, &tmpFloat);
        if (!tmpResult) {
            FEATSTD_LOG_ERROR("Set uniform spot exponent failed.");
            finalResult = false;
        }
    }
    return finalResult;
}

void Light::SetDirectionData(const Vector3& direction) const
{
    if (m_uniformBlock.direction != direction) {
        m_uniformBlock.direction = direction;
        m_isDataDirty = true;
    }
}

void Light::SetHalfPlane(const Vector3& halfplane) const
{
    if (m_uniformBlock.halfplane != halfplane) {
        m_uniformBlock.halfplane = halfplane;
        m_isDataDirty = true;
    }
}

bool Light::SetUniforms(const Shader &shader, const Node& node, UInt index) const
{
    if (index >= CANDERA_MAX_LIGHTS_COUNT) {
        FEATSTD_DEBUG_FAIL();
        return false;
    }

    const void* accessorId = s_uniformSemantics[index];
    s_uniformCacheHandle[index] = shader.GetUniformCacheHandle(accessorId);
    if (!s_uniformCacheHandle[index].IsValid()) {
        if (!shader.RegisterUniformCacheAccessor(accessorId, s_uniformSemantics[index], UniformCacheIndicesCount, s_uniformCacheHandle[index])) {
            FEATSTD_LOG_ERROR("Light::SetUniforms failed, RegisterUniformCacheHandle failed.");
            return false;
        }
    }

    bool finalResult = true;

    // Set basic uniforms independent of light type
    const bool setCommonOk = SetCommonUniforms(shader, node, index);
    if (!setCommonOk) {
        FEATSTD_LOG_ERROR("Set common uniforms failed.");
        finalResult = false;
    }

    // Set light type specific uniforms in shader
    switch(GetType()) {
        case Ambient: {
                //  nothing specific to do
                break;
            }

        case Directional: {
                const bool directionalOk = SetDirectionalUniforms(shader, node, index);
                if (!directionalOk) {
                    FEATSTD_LOG_ERROR("Set directional uniforms failed.");
                    finalResult = false;
                }
                break;
            }
        case Spot:
            {
                const bool spotOk = SetSpotUniforms(shader, node, index);
                if (!spotOk) {
                    FEATSTD_LOG_ERROR("Set spot uniforms failed.");
                    finalResult = false;
                }
            }
            //Additionally, process point related uniform settings, which also apply to spot lights.
            CANDERA_LINT_FALLTHROUGH()
        case Point:
            {
                const bool pointOk = SetPointUniforms(shader, node, index);
                if (!pointOk) {
                    FEATSTD_LOG_ERROR("Set point uniforms failed.");
                    finalResult = false;
                }
                break;
            }
        default: {
                FEATSTD_LOG_ERROR("Reached unexpected default case.");
            }
    }// switch

    Renderer::SetLight(this, index);
    return finalResult;
}

bool Light::Activate(const Shader& shader, const Node& node, UInt index) const
{
    return SetUniforms(shader, node, index);
}

void Light::OnAncestorAdded(Scene* scene)
{
    if (scene != 0) {
        static_cast<void>(scene->AddLight(this));
    }
}

void Light::OnAncestorRemoved(Scene* scene)
{
    if (scene != 0) {
        static_cast<void>(scene->RemoveLight(this));
    }
}

FEATSTD_RTTI_DEFINITION(Light, Node)

} // namespace Candera
