//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GROUP_H)
#define CANDERA_GROUP_H

#include <Candera/Engine3D/Core/3DStringBufferAppenders.h>
#include <Candera/Engine3D/Core/Node.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Candera/Macros.h>

namespace Candera {
/** @addtogroup Core3D
 *  @{
 */

/**
 * @brief   The class Group is a scene graph node that stores a set of Nodes as its children.
 *          The parent-child relationship between Groups and Nodes is bidirectional. Whereas
 *          a Group can hold a group of Nodes, a Node can at most have one parent at a time.
 *          Cycles are prohibited. Changing the Groups transformation or alpha value affects
 *          the children Nodes of the Group.
 */
class Group:  public Node
{
    FEATSTD_TYPEDEF_BASE(Node);

    public:
        /**
        * Creates an instance of this class.
        * Use Dispose() to delete the instance and possible children, if any.
        * @return Pointer to the created Group node.
        */
        static Group* Create();

        /**
         * Destructor.
         */
        virtual ~Group() override {}

        /**
        *  Clones this Node only.
        *  Attached Node resources like Appearance are not deep-copied but referenced.
        *  @return  The pointer to the cloned Node if successful, otherwise NULL.
        */
        virtual Group* Clone() const override;

        /**
         * Overrides IsRenderPrerequisiteFulfilled from Node. A Group object itself does not fulfill render prerequisites.
         * @return Always false because groups aren't rendered.
         */
        virtual bool IsRenderPrerequisiteFulfilled() const override { return false; }

        /**
         * Concatenates the given movement (translation) with current position component.
         * The children transformations are adjusted so that their world transformation remains unchanged.
         * @param translation Vector3 describing the translation of the pivot.
         */
        void TranslatePivot(const Vector3& translation);

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Group() {}
        FEATSTD_LINT_NEXT_EXPRESSION(1704, "Copy contructor is only used internaly by Clone")
        Group(const Group&);

        // Overridden methods -----------------------------------------------------------------------------

        /**
        * Computes axis-aligned bounding box (AABB) in object coordinate space and overrides function Node::ComputeBoundingBoxImpl.
        * @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
        * @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
        * @return true if bounding box was computed successfully, false otherwise.
        */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

        /**
        * Disposes the instance of this class.
        */
        virtual void DisposeSelf() override;
        /**
         * Overrides Render from base class to allow instantiation.
         */
        virtual void Render() override {}

    private:
        Group& operator=(const Group&);

        CdaDynamicProperties(Candera::Group, Candera::Node);
        CdaDynamicPropertiesEnd();
};

/** @} */ // end of Core3D
} // namespace Candera

#endif    // CANDERA_GROUP_H
