//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Group.h"
#include <Candera/System/Mathematics/Matrix3.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>

namespace Candera {
/**
 *  BoundingBoxTraverser computes axis aligned bounding box in group's local space.
 */
class BoundingBoxTraverser: public ConstTreeTraverser {
    typedef ConstTreeTraverser Base;

    public:
        BoundingBoxTraverser(const Group&  group) :
            Base(),
            m_minBounds(Math::MaxFloat(), Math::MaxFloat(), Math::MaxFloat()),    // Init to invalid min bounds.
            m_maxBounds(-Math::MaxFloat(), -Math::MaxFloat(), -Math::MaxFloat())  // Init to invalid max bounds.
        {
            m_inverseGroupTransform = group.GetWorldTransform();
            m_inverseGroupTransform.Inverse();
        }

        Vector3 GetMinBounds() const { return m_minBounds; }
        Vector3 GetMaxBounds() const { return m_maxBounds; }

    protected:
        virtual TraverserAction ProcessNode(const Node& node) override
        {
            if (node.IsBoundingBoxValid() && (!node.IsTypeOf(Group::GetTypeId())) ) {
                Vector3 minBounds;
                Vector3 maxBounds;  //Node-oriented bounding box.
                Vector3 groupMinBounds;
                Vector3 groupMaxBounds;  //Group-oriented bounding box.
                node.GetBoundingBox(minBounds, maxBounds);

                // Compute transformation of given node into group's local space.
                Matrix4 transform = node.GetWorldTransform() * m_inverseGroupTransform;

                // Transform node's bounding box into group's local space.
                Math3D::TransformBoundingBox(minBounds, maxBounds, transform, groupMinBounds, groupMaxBounds);

                // Accumulate boundaries if they exceed current group's axis aligned bounding box.
                if (groupMinBounds.GetX() < m_minBounds.GetX()) { m_minBounds.SetX(groupMinBounds.GetX()); }
                if (groupMinBounds.GetY() < m_minBounds.GetY()) { m_minBounds.SetY(groupMinBounds.GetY()); }
                if (groupMinBounds.GetZ() < m_minBounds.GetZ()) { m_minBounds.SetZ(groupMinBounds.GetZ()); }

                if (groupMaxBounds.GetX() > m_maxBounds.GetX()) { m_maxBounds.SetX(groupMaxBounds.GetX()); }
                if (groupMaxBounds.GetY() > m_maxBounds.GetY()) { m_maxBounds.SetY(groupMaxBounds.GetY()); }
                if (groupMaxBounds.GetZ() > m_maxBounds.GetZ()) { m_maxBounds.SetZ(groupMaxBounds.GetZ()); }
            }

            return ProceedTraversing;
        }

    private:
        Matrix4 m_inverseGroupTransform;
        Vector3 m_minBounds;
        Vector3 m_maxBounds;
};

Group* Group::Create()
{
    return FEATSTD_NEW(Group);
}

void Group::DisposeSelf()
{
    FEATSTD_DELETE(this);
}

Group* Group::Clone() const
{
    return FEATSTD_NEW(Group)(*this);
}

Group::Group(const Candera::Group &other) :
    Node(other)
{
}

FEATSTD_RTTI_DEFINITION(Group, Node)

void Group::TranslatePivot(const Vector3& translation)
{
    Translate(translation);

    Vector3 compensation = -translation;
    Matrix3 composite = GetCompositeTransform();
    composite.Inverse();

    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(864, Vector3 constructor copies it's arguments and therefore doesn't change the passed values.)
    Vector3 result(
        compensation.GetDotProduct(Vector3(composite(0, 0), composite(1, 0), composite(2, 0))),
        compensation.GetDotProduct(Vector3(composite(0, 1), composite(1, 1), composite(2, 1))),
        compensation.GetDotProduct(Vector3(composite(0, 2), composite(1, 2), composite(2, 2))));

    Node* node = GetFirstChild();
    while (node != 0) {
        node->Translate(result);
        node = node->GetNextSibling();
    }
}

bool Group::ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const
{
    BoundingBoxTraverser traverser(*this);
    traverser.Traverse(*this);
    minBounds = traverser.GetMinBounds();
    maxBounds = traverser.GetMaxBounds();

    return true;
}
} // namespace Candera
