//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_GLYPHATLASTEXTNODERENDERCONTEXT_H
#define CANDERA_GLYPHATLASTEXTNODERENDERCONTEXT_H

#ifdef CANDERA_2D_OVER_3D_ENABLED

#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/EngineBase/Common/GlyphAtlas.h>
#include <Candera/EngineBase/Common/GlyphAtlasGlyphCacheAccess.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/TextEngine/TextRenderContext.h>

namespace Candera
{

namespace TextRendering {
    struct GlyphBitmap;
    class TextRect;
}
namespace Internal {
    class GlyphAtlasTextContainer;
}

class GlyphAtlasTextRenderContext : public TextRendering::TextRenderContext
{
public:
    /**
     *  Constructor
     */
    GlyphAtlasTextRenderContext();

    /**
     * Destructor 
     */
    virtual ~GlyphAtlasTextRenderContext() { UnloadSelf(); }

    /**
     *  Candera text rendering will use this clipping rectangle to avoid blitting
     *  of invisible (i.e. completely outside clipping area) glyphs.
     *  @return current clipping rectangle
     *  @see TextRenderContext::GetClipRect
     */
    virtual const TextRendering::TextRect& GetClipRect() const;

    /**
     *  Blit the given glyph bitmap to the specified position in the render target
     *  @param x left coordinate of the glyph bitmap
     *  @param y top coordinate of the glyph bitmap
     *  @param glyph the glyph bitmap to be blitted
     *  @see TextRenderContext::Blit
     */
    virtual void Blit(Int16 x, Int16 y, const TextRendering::GlyphBitmap &glyph);

    /**
     *  Retrieves the glyph cache access.
     *  @return The current glyph cache access
     *  @see TextRenderContext::GetGlyphCacheAccess
     */
    virtual TextRendering::GlyphCacheAccess* GetGlyphCacheAccess() const override { return &GlyphAtlasGlyphCacheAccess::GetInstance(); }

    /**
     *  Prepare drawing by creating the necessary GPU resources.
     *  @param width   The width of the text to be drawn.
     *  @param height  The height of the text to be drawn.
     *  @return  True, if preparation of draw calls was successful. False, otherwise.
     */
    bool PrepareDrawCalls(Float width, Float height);

    /**
     *  Clear the context before re-using it.
     */
    void Clear() { m_GlyphInfos.Clear(); }

    /**
     *  Get the number of draw calls that were prepared.
     *  @return  The number of draw calls that were prepared.
     */
    SizeType GetDrawCallCount() const { return m_drawCallCount; }

    /**
     *  Get the vertex buffer that was prepared for the draw call of the given index.
     *  @param drawCallIndex  The index of the draw call to get the vertex buffer for.
     *  @return  The vertex buffer for the draw call of the given index.
     */
    const VertexBuffer::SharedPointer& GetVertexBuffer(SizeType drawCallIndex) const {
        FEATSTD_DEBUG_ASSERT(drawCallIndex < m_drawCallCount); return m_drawCallInfos[drawCallIndex].vertexBuffer;
    }

    /**
     *  Get the index of the bitmap image of the glyph atlas that was prepared for the draw call of the given index.
     *  @param drawCallIndex  The index of the draw call to get the bitmap image index for.
     *  @return  The index of the bitmap image for the draw call of the given index.
     */
    SizeType GetBitmapImageIndex(SizeType drawCallIndex) const {
        FEATSTD_DEBUG_ASSERT(drawCallIndex < m_drawCallCount); return m_drawCallInfos[drawCallIndex].BitmapImageIndex;
    }

    /**
     *  Uploads GPU resources that are used by a TextNodeRenderer derived class.
     *  @return  True if successful, false otherwise.
     */
    bool UploadSelf();

    /**
     *  Unloads GPU resources that are used by a TextNodeRenderer derived class.
     *  @return  True if successful, false otherwise.
     */
    bool UnloadSelf();

private:
    friend class Candera::Internal::GlyphAtlasTextContainer;

    struct GlyphInfo {
        SizeType BitmapImageIndex;
        GlyphAtlasBase::Entry::Rect UvRect;
        Int16 X;
        Int16 Y;
        Int16 Width;
        Int16 Height;
    };

    struct DrawCallInfo {
        VertexBuffer::SharedPointer vertexBuffer;
        SizeType BitmapImageIndex;
    };

    SizeType GetGlyphCount() const { return m_GlyphInfos.Size(); }
    const GlyphInfo& GetGlyphInfo(SizeType i) const { FEATSTD_DEBUG_ASSERT(i < m_GlyphInfos.Size()); return m_GlyphInfos[i]; }
    bool IsCleared() const { return (m_GlyphInfos.Size() == 0); }

    VertexBuffer::SharedPointer CreateVertexBuffer(SizeType glyphCount) const;
    bool PrepareVertexBuffer(Float width, Float height, DrawCallInfo& drawCallInfo, SizeType glyphInfoIndex,
        SizeType startingGlyphInfoIndex) const;

    struct SortCompare
    {
        bool operator()(const GlyphAtlasTextRenderContext::GlyphInfo& a, const GlyphAtlasTextRenderContext::GlyphInfo& b) const
        {
            return a.BitmapImageIndex < b.BitmapImageIndex;
        }
    };

    void SortGlyphInfosByBitmapImage();

    SizeType m_drawCallCount;
    Candera::Internal::Vector<GlyphInfo> m_GlyphInfos;
    Candera::Internal::Vector<DrawCallInfo> m_drawCallInfos;
};

}

#endif

#endif
