//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifdef CANDERA_2D_OVER_3D_ENABLED

#include "GlyphAtlasTextRenderContext.h"

#include <Candera/TextEngine/GlyphBitmap.h>
#include <Candera/TextEngine/TextRect.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDevicePool.h>

namespace Candera
{

using namespace TextRendering;

GlyphAtlasTextRenderContext::GlyphAtlasTextRenderContext()
    :
    m_drawCallCount(0)
{
}

const TextRect& GlyphAtlasTextRenderContext::GetClipRect() const
{
    return TextRect::GetMax();
}

void GlyphAtlasTextRenderContext::Blit(Int16 x, Int16 y, const GlyphBitmap &glyph)
{
    if ((glyph.width > 0) && (glyph.height > 0)) {
        GlyphAtlas& glyphAtlas = GlyphAtlas::GetInstance();
        GlyphAtlasBase::EntryHandle entryHandle = glyphAtlas.Get(glyph);
        const GlyphAtlasBase::Entry* atlasEntry = glyphAtlas.Get(entryHandle);
        FEATSTD_DEBUG_ASSERT(0 != atlasEntry);
        if (0 == atlasEntry) {
            return;
        }

        GlyphInfo glyphInfo;
        glyphInfo.BitmapImageIndex = atlasEntry->AtlasIndex;
        glyphInfo.UvRect = atlasEntry->UvRect;
        glyphInfo.X = x;
        glyphInfo.Y = y;
        glyphInfo.Width = glyph.width;
        glyphInfo.Height = glyph.height;

        static_cast<void>(m_GlyphInfos.Add(glyphInfo));
    }
}

void GlyphAtlasTextRenderContext::SortGlyphInfosByBitmapImage()
{
    SortCompare sortCompare;
    m_GlyphInfos.Sort(sortCompare);
}

bool GlyphAtlasTextRenderContext::PrepareDrawCalls(Float width, Float height)
{
    m_drawCallCount = 0;

    const SizeType glyphCount = GetGlyphCount();
    if (0 == glyphCount) {
        return true;
    }

    SortGlyphInfosByBitmapImage();
    SizeType startingGlyphInfoIndex = 0;
    
    if (m_drawCallInfos.Size() != 0) {
        m_drawCallInfos[0].BitmapImageIndex = GetGlyphInfo(0).BitmapImageIndex;
    }
    else {
        GlyphAtlasTextRenderContext::DrawCallInfo drawCallinfo;
        drawCallinfo.BitmapImageIndex = GetGlyphInfo(0).BitmapImageIndex;
        if (!m_drawCallInfos.Add(drawCallinfo)) {
            return false;
        }
    }
    ++m_drawCallCount;

    for (SizeType glyphInfoIndex = 0; glyphInfoIndex < glyphCount; ++glyphInfoIndex) {
        SizeType bitmapImageIndex = GetGlyphInfo(glyphInfoIndex).BitmapImageIndex;
        GlyphAtlasTextRenderContext::DrawCallInfo& drawCallInfo = m_drawCallInfos[m_drawCallCount - 1];
        if (drawCallInfo.BitmapImageIndex != bitmapImageIndex) {
            if (!PrepareVertexBuffer(width, height, drawCallInfo, glyphInfoIndex, startingGlyphInfoIndex)) {
                return false;
            }
            startingGlyphInfoIndex = glyphInfoIndex;
            if (m_drawCallInfos.Size() <= m_drawCallCount) {
                GlyphAtlasTextRenderContext::DrawCallInfo newDrawCallInfo;
                newDrawCallInfo.BitmapImageIndex = bitmapImageIndex;
                if (!m_drawCallInfos.Add(newDrawCallInfo)) {
                    return false;
                }
            }
            else {
                m_drawCallInfos[m_drawCallCount].BitmapImageIndex = bitmapImageIndex;
            }
            ++m_drawCallCount;
        }
    }

    FEATSTD_DEBUG_ASSERT(m_drawCallCount > 0);
    return PrepareVertexBuffer(width, height, m_drawCallInfos[m_drawCallCount - 1], glyphCount, startingGlyphInfoIndex);
}

VertexBuffer::SharedPointer GlyphAtlasTextRenderContext::CreateVertexBuffer(SizeType glyphCount) const
{
    using namespace Candera::Internal;

    VertexBuffer::SharedPointer vertexBuffer;

    SizeType vertexCount = glyphCount * 4;
    Context2DOver3DDevicePool::Vertex* vertices = CANDERA_NEW_ARRAY(Context2DOver3DDevicePool::Vertex,
        (vertexCount)* sizeof(Context2DOver3DDevicePool::Vertex));
    if (0 == vertices) {
        return vertexBuffer;
    }

    SizeType indexCount = glyphCount * 6;
    UInt16* indexBuffer = CANDERA_NEW_ARRAY(UInt16, indexCount);
    if (0 == indexBuffer) {
        CANDERA_DELETE_ARRAY(vertices);
        return vertexBuffer;
    }

    for (SizeType i = 0; i < glyphCount; ++i) {
        const UInt16 vertexOffset = FeatStd::Internal::NumericConversion<UInt16>(i * 4);
        const SizeType idx = i * 6;

        indexBuffer[idx] = 0 + vertexOffset;
        indexBuffer[idx + 1] = 1 + vertexOffset;
        indexBuffer[idx + 2] = 2 + vertexOffset;

        indexBuffer[idx + 3] = 2 + vertexOffset;
        indexBuffer[idx + 4] = 1 + vertexOffset;
        indexBuffer[idx + 5] = 3 + vertexOffset;
    }

    VertexGeometry* vertexGeometry = CANDERA_NEW(VertexGeometry)(
        vertices, MemoryManagement::AdaptedArrayDisposer<const void*, const Context2DOver3DDevicePool::Vertex*>::Dispose,
        Context2DOver3DDevicePool::VertexFormat, 0,
        indexBuffer, MemoryManagement::AdaptedArrayDisposer<const void*, const UInt16*>::Dispose,
        static_cast<UInt32>(vertexCount), sizeof(Context2DOver3DDevicePool::Vertex), 2, static_cast<UInt32>(indexCount),
        VertexGeometry::VideoMemory,
        VertexGeometry::UInt16IndexedArrayBuffer,
        VertexGeometry::DynamicWrite);

    if (0 == vertexGeometry) {
        CANDERA_DELETE_ARRAY(indexBuffer);
        CANDERA_DELETE_ARRAY(vertices);
        return vertexBuffer;
    }

    vertexBuffer = VertexBuffer::Create();
    if (vertexBuffer != 0) {
        static_cast<void>(vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));
    }

    return vertexBuffer;
}

bool GlyphAtlasTextRenderContext::PrepareVertexBuffer(Float width, Float height, DrawCallInfo& drawCallInfo,
    SizeType glyphInfoIndex, SizeType startingGlyphInfoIndex) const
{
    using namespace Candera::Internal;

    const SizeType glyphCount = glyphInfoIndex - startingGlyphInfoIndex;
    const SizeType vertexCount = glyphCount * 4;

    if ((drawCallInfo.vertexBuffer == 0) || (drawCallInfo.vertexBuffer->GetVertexGeometry()->GetVertexCount() < vertexCount)) {
        drawCallInfo.vertexBuffer = CreateVertexBuffer(glyphCount);
    }

    const VertexBuffer::SharedPointer& vertexBuffer = drawCallInfo.vertexBuffer;
    if (vertexBuffer == 0) {
        return false;
    }

    VertexGeometry::VertexArrayResource vertexResource(vertexBuffer->GetVertexGeometry()->GetVertexArrayResourceHandle());
    Context2DOver3DDevicePool::Vertex* vertices = const_cast<Context2DOver3DDevicePool::Vertex*>(
        FeatStd::Internal::PointerToPointer<const Context2DOver3DDevicePool::Vertex*>(vertexResource.GetData()));

    for (SizeType i = 0; i < vertexCount;) {
        const GlyphAtlasTextRenderContext::GlyphInfo& glyphInfo = GetGlyphInfo(startingGlyphInfoIndex + (i / 4));
        const Vector2& topLeft = glyphInfo.UvRect.TopLeft;
        const Vector2& bottomRight = glyphInfo.UvRect.BottomRight;
        const Float topU = topLeft.GetX();
        const Float topV = topLeft.GetY();
        const Float bottomU = bottomRight.GetX();
        const Float bottomV = bottomRight.GetY();

        Float topX = static_cast<Float>(glyphInfo.X) / width;
        Float topY = static_cast<Float>(glyphInfo.Y) / height;
        Float bottomX = (static_cast<Float>(glyphInfo.X + glyphInfo.Width)) / width;
        Float bottomY = (static_cast<Float>(glyphInfo.Y + glyphInfo.Height)) / height;

        vertices[i].x = topX;
        vertices[i].y = topY;
        vertices[i].u = topU;
        vertices[i++].v = topV;

        vertices[i].x = topX;
        vertices[i].y = bottomY;
        vertices[i].u = topU;
        vertices[i++].v = bottomV;

        vertices[i].x = bottomX;
        vertices[i].y = topY;
        vertices[i].u = bottomU;
        vertices[i++].v = topV;

        vertices[i].x = bottomX;
        vertices[i].y = bottomY;
        vertices[i].u = bottomU;
        vertices[i++].v = bottomV;
    }


    if (!vertexBuffer->IsUploaded()) {
        if (!vertexBuffer->Upload(DeviceObject::Force)) {
            return false;
        }
    }
    else {
        if (vertexBuffer->Update(0, static_cast<UInt32>(glyphCount * 4))) {
            vertexBuffer->SetElementCount(glyphCount * 6);
        }
        else {
            return false;
        }
    }

    return true;
}

bool GlyphAtlasTextRenderContext::UploadSelf()
{
    bool success = true;
    for (SizeType i = 0; i < m_drawCallInfos.Size(); ++i) {
        VertexBuffer* vertexBuffer = m_drawCallInfos[i].vertexBuffer.GetPointerToSharedInstance();
        if ((0 != vertexBuffer) && (!vertexBuffer->IsUploaded())) {
            success = vertexBuffer->Upload() && success;
        }
    }

    return success;
}

bool GlyphAtlasTextRenderContext::UnloadSelf()
{
    bool success = true;
    for (SizeType i = 0; i < m_drawCallInfos.Size(); ++i) {
        VertexBuffer* vertexBuffer = m_drawCallInfos[i].vertexBuffer.GetPointerToSharedInstance();
        if ((0 != vertexBuffer) && (vertexBuffer->IsUploaded())) {
            success = vertexBuffer->Unload() && success;
        }
    }

    return success;
}

}

#endif
