//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifdef CANDERA_2D_OVER_3D_ENABLED

#include "GlyphAtlasTextNodeRenderer.h"

#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRendererTools.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDevicePool.h>
#include <Candera/TextEngine/TextCoordinate.h>
#include <FeatStd/MemoryManagement/Heap.h>

namespace Candera
{

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine3D);

GlyphAtlasTextNodeRenderer::GlyphAtlasTextNodeRenderer()
    :
    m_hasChanged(false)
{
}

GlyphAtlasTextNodeRenderer* GlyphAtlasTextNodeRenderer::Create()
{
    return FEATSTD_NEW(GlyphAtlasTextNodeRenderer);
}

GlyphAtlasTextNodeRenderer* GlyphAtlasTextNodeRenderer::Clone() const
{
    GlyphAtlasTextNodeRenderer* clone = Create();
    if (0 != clone) {
        clone->m_textRenderContext = m_textRenderContext;
        clone->m_hasChanged = m_hasChanged;
    }

    return clone;
}

void GlyphAtlasTextNodeRenderer::Dispose()
{
    FEATSTD_DELETE(this);
}

const TextRendering::TextRenderContext* GlyphAtlasTextNodeRenderer::GetMeasureReferenceContext() const
{
    return &m_textRenderContext;
}

bool GlyphAtlasTextNodeRenderer::PreRenderInternal(const TextNode2D& textNode)
{
    const TextRendering::TextRect& boundRect = textNode.GetBoundingTextRectangle();
    TextRendering::TextCoordinate offset(-boundRect.GetLeft(), -boundRect.GetTop());
    m_textRenderContext.Clear();
    m_hasChanged = Internal::TextNodeRendererTools::RenderText(m_textRenderContext, textNode, offset);
    return true;
}

void GlyphAtlasTextNodeRenderer::RenderInternal(TextNode2D& textNode, RenderTarget2D* renderTarget, const Matrix3x2& localTransform)
{
    if (m_hasChanged)
    {
        m_hasChanged = false;

        const TextRendering::TextRect& boundRect = textNode.GetBoundingTextRectangle();
        if (!m_textRenderContext.PrepareDrawCalls(static_cast<Float>(boundRect.GetWidth()), static_cast<Float>(boundRect.GetHeight()))) {
            FEATSTD_LOG_ERROR("GlyphAtlasTextNodeRenderer failed to create or set vertex buffer.");
        }
    }

    const GlyphAtlas& atlas = GlyphAtlas::GetInstance();
    using namespace Internal;
    Context2DOver3DDevicePool& pool = Context2DOver3DDevicePool::GetInstance();

    for (SizeType i = 0; i < m_textRenderContext.GetDrawCallCount(); ++i) {
        const SizeType bitmapImageIndex = m_textRenderContext.GetBitmapImageIndex(i);
        SharedPointer<BitmapImage2D> glyphBitmap;

        if (atlas.GetBitmapImageCount() <= bitmapImageIndex) {
            // Try to get it from previous used (and uploaded) glyph bitmaps.
            glyphBitmap = m_bitmapTracker.Get(bitmapImageIndex);
        } else {
            glyphBitmap = atlas.GetBitmapImage(bitmapImageIndex);
            if (!glyphBitmap.PointsToNull())
            {
                m_bitmapTracker.Add(bitmapImageIndex, glyphBitmap);
            }
            else
            {
                glyphBitmap = m_bitmapTracker.Get(bitmapImageIndex);
            }
        }
        if (glyphBitmap.PointsToNull())
        {
            FEATSTD_LOG_ERROR("GlyphAtlasTextNodeRenderer cannot render the text because the atlas has been cleared or the font has been removed.");
            continue;
        }

        const TextRendering::TextRect& boundRect = textNode.GetBoundingTextRectangle();
        Rectangle rect = Rectangle(0.0F, 0.0F, static_cast<Float>(boundRect.GetWidth()), static_cast<Float>(boundRect.GetHeight()));
        pool.SetCustomVertexBuffer(m_textRenderContext.GetVertexBuffer(i), rect);

        const Vector2 position(static_cast<Float>(boundRect.GetLeft()), static_cast<Float>(boundRect.GetTop()));
        TextNodeRendererTools::RenderImage(glyphBitmap.GetSharedInstance(),
            textNode,
            renderTarget,
            localTransform,
            position);

        pool.SetCustomVertexBuffer(VertexBuffer::SharedPointer(0));
    }
    // Unload unused atlas bitmaps
    m_bitmapTracker.FinishIteration();
}

bool GlyphAtlasTextNodeRenderer::UploadSelf()
{
    m_bitmapTracker.UploadAll();
    return m_textRenderContext.UploadSelf();
}

bool GlyphAtlasTextNodeRenderer::UnloadSelf()
{
    m_bitmapTracker.UnloadAll();
    return m_textRenderContext.UnloadSelf();
}

}

#endif
