//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlyphAtlasTextContainer.h"
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/TextEngine/GlyphCacheAccess.h>
#include <Candera/TextEngine/TextRenderContext.h>
#include <Candera/TextEngine/Internal/TextProcessHelper.h>
#include <Candera/TextEngine/Internal/TextProcessResult.h>
#include <Candera/Engine3D/Core/GlyphAtlas3DGlyphCacheAccess.h>

#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

using namespace Candera::TextRendering;
using namespace Candera::TextRendering::Internal;

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine3D);
    namespace Internal {

        class GlyphAtlas3DTextRenderContext : public TextRendering::TextRenderContext {
        public:
            GlyphAtlas3DTextRenderContext(GlyphAtlasTextContainer& container) :m_textContainer(container) {}

            virtual const TextRect& GetClipRect() const override
            {
                return TextRect::GetMax();
            }

            virtual void Blit(Int16 x, Int16 y, const GlyphBitmap &glyph) override
            {
                if ((glyph.width > 0) && (glyph.height > 0)) {
                    GlyphAtlas3D& glyphAtlas = GlyphAtlas3D::GetInstance();
                    const GlyphAtlasBase::EntryHandle entryHandle = glyphAtlas.Get(glyph);
                    const GlyphAtlas3D::Entry* atlasEntry = glyphAtlas.Get(entryHandle);
                    if (0 == atlasEntry) {
                        FEATSTD_LOG_ERROR("Requested glyph cannot be found within the Glyph Atlas!");
                        return;
                    }

                    GlyphAtlasTextContainer::GlyphInfo glyphInfo;
                    glyphInfo.BitmapImageIndex = atlasEntry->AtlasIndex;
                    glyphInfo.UvRect = atlasEntry->UvRect;
                    glyphInfo.X = x;
                    glyphInfo.Y = y;
                    glyphInfo.Width = glyph.width;
                    glyphInfo.Height = glyph.height;
                    static_cast<void>(m_textContainer.m_glyphInfos.Add(glyphInfo));

                }
            }

            virtual GlyphCacheAccess* GetGlyphCacheAccess() const override
            {
                if (m_textContainer.GetTextNode() != 0) {
                    // It has to be the GlyphAtlas cache - otherwise this system would fail completely
                    FEATSTD_DEBUG_ASSERT(m_textContainer.GetTextNode()->GetGlyphCache() == &GlyphAtlas3DGlyphCacheAccess::GetInstance());
                    return m_textContainer.GetTextNode()->GetGlyphCache();
                }
                return 0;
            }

        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(GlyphAtlas3DTextRenderContext);
            GlyphAtlasTextContainer& m_textContainer;
        };

        struct SortCompare
        {
            bool operator()(const Candera::Internal::GlyphAtlasTextContainer::GlyphInfo& a,
                const Candera::Internal::GlyphAtlasTextContainer::GlyphInfo& b) const
            {
                return a.BitmapImageIndex < b.BitmapImageIndex;
            }
        };

        static void SortGlyphInfosByBitmapImage(Candera::Internal::GlyphAtlasTextContainer::GlyphInfoVector& glyphInfos)
        {
            SortCompare sortCompare;
            glyphInfos.Sort(sortCompare);
        }

        Candera::Internal::GlyphAtlasTextContainer::TextNode * GlyphAtlasTextContainer::GetTextNode() const
        {
            return m_textNode;
        }

        void GlyphAtlasTextContainer::SetTextNode(Candera::Internal::GlyphAtlasTextContainer::TextNode * textNode)
        {
            if (m_textNode != textNode) {
                Clear();
                m_textNode = textNode;
            }
        }

        void GlyphAtlasTextContainer::UpdateContainer(TextNode * textNode)
        {
            SetTextNode(textNode);

            if (m_textNode != 0) {
                TextProcessResult::SharedPointer result = m_textNode->GetActiveProcessResult();
                if ((!result.PointsToNull()) && (!result->IsResultConsumed())) {
                    m_glyphInfos.Clear();
                    GlyphAtlas3DTextRenderContext context(*this);
                    static_cast<void>(TextRendering::Internal::TextProcessHelper::RenderText(result.GetSharedInstance(), context));
                    result->SetResultConsumed(true);
                    SetContainerChangeConsumed(false);
                    SortGlyphInfosByBitmapImage(m_glyphInfos);
                }
                if (m_textNode->GetTextColor() != m_currentTextColor) {
                    m_currentTextColor = m_textNode->GetTextColor();
                    SetContainerChangeConsumed(false);
                }
            }
        }

        Candera::Internal::GlyphAtlasTextContainer::GlyphInfo const * GlyphAtlasTextContainer::operator[](SizeType index) const
        {
            if (index >= m_glyphInfos.Size()) {
                return 0;
            }
            return &m_glyphInfos[index];
        }

        GlyphAtlasTextContainer::GlyphAtlasTextContainer() :
            m_textNode(0), 
            m_currentTextColor(1.0F, 1.0F, 1.0F),
            m_containerChangeConsumed(false)
        {

        }

        void GlyphAtlasTextContainer::Clear()
        {
            m_textNode = 0;
            m_glyphInfos.Clear();
        }

    }
}

