//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_FRUSTUM_H)
#define CANDERA_FRUSTUM_H

#include <Candera/System/Mathematics/Plane.h>

namespace Candera {

/**  @addtogroup Core3D
 *   @{
 */
 
class Mesh;
class Camera;

/**
*  @brief The Frustum class serves as base class for 
*         all further derived frustum implementations.  
*         It provides methods to determine whether a point, sphere or mesh is within 
*         the frustum or not.
*         Frustum planes in derivations should be in world space in order to be able
*         to test against world space points, bounding spheres or meshes.
*/

class Frustum {
    public:        

        /**
         *  Constructor
         */
        Frustum();

        /**
         *  Destructor
         */
        virtual ~Frustum(); //virtual because of base class.


        /**   
         *  Use this method to calculate the frustum planes in derivations.
         */
        virtual void Recalculate() = 0;  

        /**   
         *  Determines if a point is inside the frustum.
         *  @param point Point to be determined
         *  @return      Whether or not the point is inside the frustum
         */
        bool IsPointInFrustum(const Vector3& point) const; 

        /**  
         *  Determines if (part of) a bounding sphere is inside the frustum.
         *  @param center Center of the bounding sphere
         *  @param radius Radius of the bounding sphere
         *  @return       True if part of the sphere is inside the frustum
         */   
        bool IsSphereInFrustum(const Vector3& center, Float radius) const;  

        /**  
         *  Determines if a polygon is inside the frustum.
         *  @param noOfPoints Number of points in polygon
         *  @param pointList  List of points in polygon
         *  @return           True if all points of the polygon are inside the frustum
         */
        bool IsPolygonInFrustum(UInt noOfPoints, const Vector3* pointList) const;        

        /**  
         *  Determines if (part of) a Mesh is inside the frustum.
         *  @param mesh Mesh to be determined
         *  @return     True if part of the mesh is inside the frustum
         */
        bool IsMeshInFrustum(const Mesh& mesh) const;

        /**
         *  Returns the distance of a point in world space to the near plane of the frustum.
         *  Use this value e.g in conjunction with ViewingFrustums and with the bounding sphere radius to alter the level of detail (LOD).  
         *  @param point specifies the position in world space.
         *  @return distance of point in world space to near plane. The distance is positive if in front of near plane and
         *  negative if behind of near plane.
         */
        Float GetDistanceToNearPlane(const Vector3& point) const;  
        
    protected:
        
        enum FrustumPlane {
            NearPlane,      ///< In a view frustum this is the plane closest to the viewer and is perpendicular to the viewing direction.
            FarPlane,       ///< In a view frustum this is the plane furthest from the viewer and is perpendicular to the viewing direction.
            LeftPlane,      ///< The left plane.
            RightPlane,     ///< The right plane.
            TopPlane,       ///< The top plane.
            BottomPlane,    ///< The bottom plane.
            NumberOfPlanes  ///< Number of planes of a frustum (6)
        };

        Plane m_planes[NumberOfPlanes];       
};
 
/**  @} */ // end of Core3D
 
} //namespace Candera

#endif // CANDERA_FRUSTUM_H
