//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DEVICE_OBJECT_LISTENER_H)
#define CANDERA_DEVICE_OBJECT_LISTENER_H

#include <Candera/Environment.h>

namespace Candera {

    class DeviceObject;
    class ContextResourcePool;

    /** @addtogroup Core3D
     *  @{
     */

    /**
     * @brief Interface for DeviceObject listener implementations.
     *
     * DeviceObjectListener receive notifications whenever the object that is
     *  listened to is uploaded, unloaded or destroyed.
     */
    class DeviceObjectListener {
            friend class DeviceObject;
        public:
            virtual ~DeviceObjectListener() {}

        protected:
            DeviceObjectListener() {}

            /**
             * @brief Notification of DeviceObject start uploading.
             *
             * OnPreUpload notification is triggered by a DeviceObject before
             *  its data is uploaded in VRAM. If Upload is called on an already
             *  uploaded device object, no new notification will be triggered.
             * @param deviceObject The DeviceObject that triggered the
             *  notification.
             * @param contextResourcePool ContextResourcePool in which the
             *  device object will be uploaded.
             * @see DeviceObject::Upload
             */
            virtual void OnPreUpload(
                DeviceObject& deviceObject,
                const ContextResourcePool& contextResourcePool) = 0;

            /**
             * @brief Notification of DeviceObject finish uploading.
             *
             * OnPostUpload notification is triggered by a DeviceObject after
             *  its data is uploaded in VRAM. The notification is triggered even
             *  if upload fails. DeviceObject::IsUploaded() can be called to
             *  check the upload state.
             * @param deviceObject The DeviceObject that triggered the
             *  notification.
             * @param contextResourcePool ContextResourcePool in which the
             *  device object is uploaded.
             * @see DeviceObject::Upload
             */
            virtual void OnPostUpload(
                DeviceObject& deviceObject,
                const ContextResourcePool& contextResourcePool) = 0;

            /**
             * @brief Notification of DeviceObject start unloading.
             *
             * OnPreUnload notification is triggered by a DeviceObject before
             *  its data is unloaded from VRAM. If Unload does not free VRAM,
             *  but only decrease the upload reference count, no notification
             *  will be triggered.
             * @param deviceObject The DeviceObject that triggered the
             *  notification.
             * @param contextResourcePool ContextResourcePool from which the
             *  device object will be uploaded.
             * @see DeviceObject::Unload
             */
            virtual void OnPreUnload(
                DeviceObject& deviceObject,
                const ContextResourcePool& contextResourcePool) = 0;

            /**
             * @brief Notification of DeviceObject finish unloading.
             *
             * OnPostUnload notification is triggered by a DeviceObject after
             *  its data is unloaded from VRAM. The notification is triggered
             *  even if unload fails. DeviceObject::IsUploaded() can be called
             *  to check the upload state.
             * @param deviceObject The DeviceObject that triggered the
             *  notification.
             * @param contextResourcePool ContextResourcePool from which the
             *  device object will be uploaded.
             * @see DeviceObject::Unload
             */
            virtual void OnPostUnload(
                DeviceObject& deviceObject,
                const ContextResourcePool& contextResourcePool) = 0;
            /**
             * @brief Notification of DeviceObject destroy.
             *
             * OnDestruction notification is triggered by a DeviceObject whenever
             *  it is destroyed.
             * @param deviceObject The DeviceObject that triggered the
             *  notification.
             */
            virtual void OnDestruction(
                DeviceObject& deviceObject) = 0;
        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(DeviceObjectListener);
    };
    /** @} */ // end of Core3D
}

#endif
