//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CameraRenderStrategy.h"

#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/RenderOrder/RenderOrder.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {

using namespace Diagnostics;
FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

FEATSTD_RTTI_BASECLASS_DEFINITION(CameraRenderStrategy)

CameraRenderStrategy::CameraRenderStrategy(bool isDrawcallBatchingPathUsed) :
m_previousNodeRendered(0),
m_drawcallBatchingRenderOrder(0),
m_isRenderPassCompleted(true)
{
    if (isDrawcallBatchingPathUsed) {
        m_drawcallBatchingRenderOrder = RenderOrder::Create(10, 0);
    }
}

CameraRenderStrategy::~CameraRenderStrategy()
{
    if (0 != m_drawcallBatchingRenderOrder) {
        m_drawcallBatchingRenderOrder->Dispose();
    }
    m_drawcallBatchingRenderOrder = 0;
    m_previousNodeRendered = 0;
}

bool CameraRenderStrategy::IsRenderPassCompleted() const
{
    return m_isRenderPassCompleted;
}

void CameraRenderStrategy::SetRenderPassCompleted(bool complete)
{
    m_isRenderPassCompleted = complete;
}

const Node* CameraRenderStrategy::GetPreviousNodeRendered() const
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1763, Candera::CameraRenderStrategy::GetPreviousNodeRendered, "semantically, modification rights for most recent node belong to Renderer")
    return m_previousNodeRendered;
}

void CameraRenderStrategy::SetPreviousNodeRendered(const Node* n)
{
    m_previousNodeRendered = n;
}

bool CameraRenderStrategy::Render(AbstractRenderOrder* renderOrder)
{
    const bool isDrawcallBatchingPathUsed = (0 != m_drawcallBatchingRenderOrder);
    if (isDrawcallBatchingPathUsed) {
        m_drawcallBatchingRenderOrder->Clear();
    }

    renderOrder->IterationBegin();

    // Move render order iterator to the node where to start rendering from.
    if (IsRenderPassCompleted()) {
        // Start rendering with first node.
        SetPreviousNodeRendered(0);
        OnRenderPassBegins();
    }
    else {
        // Iterate to the successor of previous node rendered before the render pass has been paused by CameraRenderStrategy.
        const Node* const previousNodeRendered = GetPreviousNodeRendered();
        Node* currentNode = 0;
        while ((currentNode != previousNodeRendered) && renderOrder->IterationHasMoreNodes()) {
            currentNode = renderOrder->IterationGetNode();
            renderOrder->IterationMoveToNextNode();
        }
        OnRenderPassResumes();
    }

    bool wasSuccessful = true;
    bool isRenderPassCompleted = true;
    // Loop RenderOrder to render all Nodes. Only a RenderCameraStrategy object can interrupt the render loop.
    for (/* Node from where rendering starts */; renderOrder->IterationHasMoreNodes(); renderOrder->IterationMoveToNextNode()) {
        Node* const currentNode = renderOrder->IterationGetNode();
        if (currentNode == 0) {
            FEATSTD_LOG_ERROR("Render camera failed, currentNode == 0.");
            return false;
        }

        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::CameraRenderStrategyRenderLoop, currentNode->GetName()));

        // If CameraRenderStrategy is available, check if render pass shall proceed with current node.
        const CameraRenderStrategy::RenderPassAction action = GetRenderPassAction(currentNode);
        if (action == CameraRenderStrategy::ProceedRenderPass) {
            SetPreviousNodeRendered(currentNode);
        }
        else if (action == CameraRenderStrategy::PauseRenderPass) {
            isRenderPassCompleted = false;
            break;
        }
        else if (action == CameraRenderStrategy::SkipNode) {
            continue;
        }
        else { /*if (action == CameraRenderStrategy::StopRenderPass)*/
            break;
        }

        if (isDrawcallBatchingPathUsed) {
            wasSuccessful = wasSuccessful && m_drawcallBatchingRenderOrder->AssignVerifiedNodeToBin(currentNode);
        }
        else {
            // Finally, render Node if all culling tests passed.
            Renderer::NotifyRendererListenersOnNodePreRender(currentNode);
            currentNode->Render();
            Renderer::NotifyRendererListenersOnNodePostRender(currentNode);
        }
    }

    if (isDrawcallBatchingPathUsed) {
        wasSuccessful = wasSuccessful && Renderer::Render(m_drawcallBatchingRenderOrder, /* useImmediateRendering */ false);
    }

    SetRenderPassCompleted(isRenderPassCompleted);

    return wasSuccessful;
}

} // namespace Candera
