//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CameraListener_H)
#define CANDERA_CameraListener_H

#include <Candera/Environment.h>

namespace Candera {

    /** @addtogroup Core3D
     *  @{
     */

    class Camera;
    class RenderTarget3D;
    class CameraRenderStrategy;
    
    /**
    * @brief    A CameraListener defines hooks that are called before or after a Camera is rendered.
    *           In order to register a CameraListener to a Camera simply derive from CameraListener and override pure virtual functions with custom code.
    */
    class CameraListener {
        public:
            /**
            * Constructs a CameraListener object.
            */
            CameraListener() {}

            /**
            * Destructs a CameraListener object.
            */
            virtual ~CameraListener() {}
            
            /**
            * Is invoked before a camera is rendered.
            * At the time when this function is called, the scene and its lights are already activated.
            * @param camera Camera to listen to.
            */
            virtual void OnPreRender(Camera* camera) = 0;

            /**
            * Is invoked after a camera is rendered.
            * @param camera Camera to listen to.
            */
            virtual void OnPostRender(Camera* camera) = 0;

            /**
            * Is invoked after camera projection was changed.
            * @param camera Camera to listen to.
            */
            virtual void OnProjectionChanged(Camera* camera);

            /**
            * Is invoked after camera projection was changed.
            * @param camera Camera to listen to.
            */
            virtual void OnViewChanged(Camera* camera);

            /**
            * Is invoked after the RenderTarget3D of the Camera was changed.
            * Use Camera::GetRenderTarget() to retrieve the new associated RenderTarget3D.
            * @param camera Camera that triggered the notification.
            * @param previousRenderTarget RenderTarget3D that was replaced.
            */
            virtual void OnRenderTargetChanged(Camera* camera, RenderTarget3D* previousRenderTarget);

            /**
            * Is invoked after the CameraRenderStrategy of the Camera was changed.
            * Use Camera::GetCameraRenderStrategy() to retrieve the new associated CameraRenderStrategy.
            * @param camera Camera that triggered the notification.
            * @param previousCameraRenderStrategy CameraRenderStrategy that was replaced.
            */
            virtual void OnCameraRenderStrategyChanged(Camera* camera, CameraRenderStrategy* previousCameraRenderStrategy);

            /**
            *  Is invoked before a camera is activated.
            *  @param camera The camera that will be activated.
            */
            virtual void OnPreActivate(const Camera* camera);

            /**
            *  Is invoked after a camera is activated.
            *  @param camera The camera that was activated.
            */
            virtual void OnPostActivate(const Camera* camera);

            /**
            *  Is invoked before clearing the framebuffer.
            *  @param camera The camera that will be activated.
            */
            virtual void OnPreClear(const Camera* camera);

            /**
            *  Is invoked after a clearing the framebuffer.
            *  @param camera The camera that was activated.
            */
            virtual void OnPostClear(const Camera* camera);
    
        protected:
            CameraListener(const CameraListener&);
            CameraListener& operator=(const CameraListener&);
    };

    inline void CameraListener::OnProjectionChanged(Camera* /*camera*/)
    {
    }

    inline void CameraListener::OnViewChanged(Camera* /*camera*/)
    {
    }

    inline void CameraListener::OnRenderTargetChanged(Camera* /*camera*/, RenderTarget3D* /*previousRenderTarget*/)
    {
    }

    inline void CameraListener::OnCameraRenderStrategyChanged(Camera* /*camera*/, CameraRenderStrategy* /*previousCameraRenderStrategy*/)
    {
    }

    inline void CameraListener::OnPreActivate(const Camera* /*camera*/)
    {
    }

    inline void CameraListener::OnPostActivate(const Camera* /*camera*/)
    {
    }

    inline void CameraListener::OnPreClear(const Camera* /*camera*/)
    {
    }

    inline void CameraListener::OnPostClear(const Camera* /*camera*/)
    {
    }

    /** @} */ // end of Core3D

} // namespace Candera

#endif// CANDERA_CameraListener_H

