//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_BlendLodRenderStrategy_h
#define Candera_BlendLodRenderStrategy_h

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/LodRenderStrategy.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
/**
 *  @brief The class BlendLodRenderStrategy defines an interface to configure the LodNode's visual appearance according to the LodCriterion retrieved.
 *         This render strategy renders LOD levels opaque if criterion value is within their lower and upper bound. If criterion value falls between 
 *         two non-adjacent boundaries of contiguous LOD levels, then these LOD levels become blended accordingly. During blend transition at least 
 *         one LOD level is always opaque to avoid undesired "see-through" effects.
 *         If boundaries are adjacent and criterion value coincides with boundary, then the LOD level with lower index is shown (e.g LOD 0 before LOD 1).
 *         BlendLodRenderStrategy can be shared among different LodNodes.
 *
 *         Note: Blending is realized with RenderMode's constant color property and does therefore not affect any shader code. Make sure that all LodNode's children
 *         have their own RenderMode object set in order to enable correct blending. Custom alpha values set in Shaders (with e.g Material, Textures) are ignored.
 *         
 *  Blending Example for LOD level 0: lowerBound = 0, upperBound = 40;  LOD level 1: lowerBound = 60, upperBound = 100.
 *  Step 1: criterion value: 40 -> LOD level 0: Opaque; LOD level 1: Not rendered.
 *  Step 2: criterion value: 45 -> LOD level 0: Opaque; LOD level 1: 50% transparent.
 *  Step 3: criterion value: 50 -> LOD level 0: Opaque; LOD level 1: Opaque.
 *  Step 4: criterion value: 55 -> LOD level 0: 50% transparent; LOD level 1: Opaque.
 *  Step 5: criterion value: 60 -> LOD level 0: Not rendered; LOD level 1: Opaque.
 */
class BlendLodRenderStrategy: public LodRenderStrategy
{
    FEATSTD_TYPEDEF_BASE(LodRenderStrategy);

    public:

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Constructs a BlendLodRenderStrategy object.                      
         */
        BlendLodRenderStrategy();

        /**
         *  Destructs a BlendLodRenderStrategy object.                         
         */ 
        virtual ~BlendLodRenderStrategy() override{
        m_opaqueBinName = 0;
        m_transparentBinName = 0;
        }
        
        /**
         *  Enables depth bias for transparent objects during blend transition to avoid Z-fighting artifacts for coplanar surfaces.
         *  Per default depth bias is disabled.                         
         *  @param enable True enables depth bias, false disables it.
         */
        void SetDepthBiasEnabled(bool enable) { m_isDepthBiasEnabled = enable; }
        /**
         *  Gets whether depth bias for transparent objects during blend transitions is enabled or not.
         *  Used to avoid Z-fighting artifacts for coplanar surfaces.
         *  @return True if depth bias is enabled, False otherwise.
         */
        bool IsDepthBiasEnabled() const { return m_isDepthBiasEnabled; }

        /**
         *  Sets render order bin assignment for LOD levels, if they are rendered opaquely.
         *  Per default no explicit bin is set, which means each opaque LOD node is automatically assigned to render order bin 
         *  with name RenderOrder::OpaqueBinName.
         *  @param binName name of bin to assign opaquely rendered LOD node to.
         */
        void SetOpaqueRenderOrderBinAssignment(const Char* binName) { m_opaqueBinName = binName; }
        const Char* GetOpaqueOpaqueBinAssignment() const { return m_opaqueBinName; }

        /**
         *  Sets render order bin assignment for LOD levels, if they are rendered transparently during blend transition.
         *  Per default no explicit bin is set, which means each transparent LOD node is automatically assigned to render order bin
         *  with name RenderOrder::TransparentBinName.
         *  @param binName name of bin to assign transparently rendered LOD node to.
         */
        void SetTransparentRenderOrderBinAssignment(const Char* binName) { m_transparentBinName = binName; }

        /**
         *  Retrieves the render order bin assignment for LOD levels.
         *  Per default no explicit bin is set, which means each transparent LOD node is automatically assigned to render order bin
         *  with name RenderOrder::TransparentBinName.
         *  @return The render order bin assignment for LOD levels.
         */
        const Char* GetTransparentRenderOrderBinAssignment() const { return m_transparentBinName; }

        /**
         *  Configures LodNode before rendering. E.g. sets level of detail level to render according to LodCrition value.   
         *  @param lodNode LOD node to update.
         */
        virtual void Update(LodNode& lodNode) override;

    private:        
        bool m_isDepthBiasEnabled;        // Indicates if depth bias shall be used for transparent objects. Default value is false.       
        const Char* m_opaqueBinName;      // Render order bin assignment for opaque LOD nodes. Default value is 0, which means automatic assignment.    
        const Char* m_transparentBinName; // Render order bin assignment for transparent LOD nodes. Default value is 0, which means automatic assignment.

        /**
         *  Activates distinctive LOD level.  
         *  @param lodNode  LOD node containing LOD levels.
         *  @param lodIndex LOD level to activate. All others are deactivated.
         */      
        void ActivateLodLevel(LodNode& lodNode, UInt lodIndex) const;

        /**
         *  Activates the blend transition between two adjacent LOD levels, if criterion value falls between two LOD's boundaries.     
         *  @param lodNode            LOD node containing LOD levels.
         *  @param lowerLodLevelIndex Index of lower LOD level.
         *  @param criterionValue     LOD criterion describing which level is active and how much it should be blended.
         */     
        void ActivateBlendTransition(LodNode& lodNode, UInt lowerLodLevelIndex, Float criterionValue) const;

        /**
         *  Retrieves LOD index for given criterion value.
         *  @param lodNode is the LodNode to find LOD index in.                         
         *  @param criterionValue is the criterion value to map onto LOD boundaries.
         *  @param isLodIndexUnique is an out-parameter: If true the index returned is a unique LOD index; if false it is the lower index of two
         *  indices to be blended.   
         *  @return the LOD index that coincides with the given criterion value. If value is -1, then a matching LOD index could not be found.
         */
        Int GetLodIndex(const LodNode& lodNode, Float criterionValue, bool& isLodIndexUnique /*out*/) const;
};
 
/** @} */ // end of Core3D
 
}

#endif // Candera_BlendLodRenderStrategy_h
