//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BlendLodRenderStrategy.h"
#include <Candera/Engine3D/Core/LodNode.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Core/LodCriterion.h>

namespace Candera {

using MemoryManagement::SharedPointer;

class BlendTransitionTraverser: public TreeTraverser {
    typedef TreeTraverser Base;

public:
    BlendTransitionTraverser() : Base(), m_alphaValue(1.0F), m_isDepthBiasEnabled(false), m_opaqueBinName(0), m_transparentBinName(0) {}
    ~BlendTransitionTraverser()
    {
        m_opaqueBinName = 0;
        m_transparentBinName = 0;
    }

    void SetAlphaValue(Float alphaValue) { m_alphaValue = alphaValue; }
    Float GetAlphaValue() const { return m_alphaValue; }

    void SetDepthBiasEnabled(bool enable) { m_isDepthBiasEnabled = enable; }
    bool IsDepthBiasEnabled() const { return m_isDepthBiasEnabled; }

    void SetOpaqueRenderOrderBinAssignment(const Char* binName) { m_opaqueBinName = binName; }
    const Char* GetOpaqueRenderOrderBinAssignment() const { return m_opaqueBinName; }

    void SetTransparentRenderOrderBinAssignment(const Char* binName) { m_transparentBinName = binName; }
    const Char* GetTransparentRenderOrderBinAssignment() const { return m_transparentBinName; }

protected:
    virtual TraverserAction ProcessNode(Node& node) override
    {
        BlendNode(node);
        return ProceedTraversing;
    }

private:
    Float m_alphaValue;
    bool m_isDepthBiasEnabled;
    const Char* m_opaqueBinName;
    const Char* m_transparentBinName;

    void BlendNode(Node& n) const;
};

void BlendTransitionTraverser::BlendNode(Node& n) const
{
    SharedPointer<Appearance> appearance = n.GetAppearance();
    if (appearance != 0) {
        SharedPointer<RenderMode> renderMode = appearance->GetRenderMode();
        if (renderMode != 0) {
            // Make sure that subsequent RenderMode configuration is not inherited by default RenderMode.
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(655, Bitwise operations intended by Design.);
            renderMode->SetInheritanceBitsToZero(static_cast<UInt32>(RenderMode::BlendingEnabledBit) | static_cast<UInt32>(RenderMode::BlendModeBit) | static_cast<UInt32>(RenderMode::BlendColorBit) | static_cast<UInt32>(RenderMode::DepthBiasBit));
            if (m_alphaValue >= 1.0F) {
                //Render opaque.
                renderMode->SetBlendingEnabled(false);
                n.SetRenderOrderBinAssignment(m_opaqueBinName);
            }
            else {
                //Render transparent.
                renderMode->SetBlendingEnabled(true);
                renderMode->SetBlendModeSeparate(RenderMode::ConstantAlpha, RenderMode::InverseConstantAlpha, RenderMode::Add,
                                                 RenderMode::InverseDestAlpha, RenderMode::One, RenderMode::Add);
                renderMode->SetBlendColor(Color(0.0F, 0.0F ,0.0F, m_alphaValue));
                if (m_isDepthBiasEnabled) {
                    renderMode->SetDepthBias(1.0F, 1.0F);
                }
                n.SetRenderOrderBinAssignment(m_transparentBinName);
            }
        }
    }
}

FEATSTD_RTTI_DEFINITION(BlendLodRenderStrategy, LodRenderStrategy)

BlendLodRenderStrategy::BlendLodRenderStrategy() :
    Base(),
    m_isDepthBiasEnabled(false),
    m_opaqueBinName(0),
    m_transparentBinName(0)
{
    // nothing to do.
}

void BlendLodRenderStrategy::ActivateLodLevel(LodNode& lodNode, UInt lodIndex) const
{
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, lodNode, CANDERA_LINT_REASON_NONCONST)
    for(Int index= 0; index < static_cast<Int>(lodNode.GetLodLevelCount()); index ++) {
        Node* node= lodNode.GetLodLevel(static_cast<UInt>(index)).node;
        if (node != 0) {
            // Activate the LOD level given, disable all others.
            node->SetRenderingEnabled(static_cast<UInt>(index) == lodIndex);
            if (node->IsRenderingEnabled()) {
                BlendTransitionTraverser blendTraverser;
                blendTraverser.SetAlphaValue(1.0F);
                blendTraverser.Traverse(*node);
            }
        }
    }
}

void BlendLodRenderStrategy::ActivateBlendTransition(LodNode& lodNode, UInt lowerLodLevelIndex, Float criterionValue) const
{
     CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, lodNode, CANDERA_LINT_REASON_NONCONST)
    // Calculate params needed for blending transition.
    LodNode::LodLevel lowerLodLevel = lodNode.GetLodLevel(lowerLodLevelIndex);
    LodNode::LodLevel upperLodLevel = lodNode.GetLodLevel(lowerLodLevelIndex + 1);

    Float range= upperLodLevel.lowerBound - lowerLodLevel.upperBound;
    Float factor= (criterionValue - lowerLodLevel.upperBound) / range;

    Float lowerBlendFactor = 1.0F;
    if (factor > 0.5F) {
        lowerBlendFactor = 2.0F - (factor * 2.0F);
    }

    Float upperBlendFactor = 1.0F;
    if (factor < 0.5F) {
        upperBlendFactor = factor * 2.0F;
    }

    // Enable rendering for blended LOD levels. Disable rendering for all others.
    for(Int lodIndex= 0; lodIndex < static_cast<Int>(lodNode.GetLodLevelCount()); lodIndex++) {
        Node* node= lodNode.GetLodLevel(static_cast<UInt>(lodIndex)).node;
        if (node != 0) {
            node->SetRenderingEnabled((static_cast<UInt>(lodIndex) == lowerLodLevelIndex) || (static_cast<UInt>(lodIndex) == (lowerLodLevelIndex + 1)));
        }
    }

    // Init BlendTransitionTraverser
    BlendTransitionTraverser blendTraverser;
    blendTraverser.SetDepthBiasEnabled(m_isDepthBiasEnabled);
    blendTraverser.SetOpaqueRenderOrderBinAssignment(m_opaqueBinName);
    blendTraverser.SetTransparentRenderOrderBinAssignment(m_transparentBinName);

    // Blend lower LOD level.
    if (lowerLodLevel.node != 0)  {
        blendTraverser.SetAlphaValue(lowerBlendFactor);
        blendTraverser.Traverse(*lowerLodLevel.node);
    }

    // Blend upper LOD level.
    if (upperLodLevel.node != 0) {
        blendTraverser.SetAlphaValue(upperBlendFactor);
        blendTraverser.Traverse(*upperLodLevel.node);
    }
}

Int BlendLodRenderStrategy::GetLodIndex(const LodNode& lodNode, Float criterionValue, bool& isLodIndexUnique /*out*/) const
{
    isLodIndexUnique= false;
    // As LOD levels are expected in monotonically increasing order, no Index is found, if criterion is lower than first LOD levels lower bound.
    if (criterionValue < lodNode.GetLodLevel(0).lowerBound) {
        return -1;
    }
    // Iteratively take next bound until criterionValue is below.
    Int maxLodCount = static_cast<Int>(lodNode.GetLodLevelCount());
    for (Int lodIndex = 0; lodIndex < maxLodCount; lodIndex++) {
        if (criterionValue <= lodNode.GetLodLevel(static_cast<UInt>(lodIndex)).upperBound) {
                // Unique LOD index has been found, as criterion value is greater equal lowerBound and less equal than upperBound.
                isLodIndexUnique = true;
                return lodIndex;
        }
        // If next LOD index exists, check if criterion value is greater than upperBound of current LOD and less than lowerBound of next LOD.
        else if (((lodIndex + 1) < maxLodCount) && (criterionValue < lodNode.GetLodLevel(static_cast<UInt>(lodIndex) + 1).lowerBound)) {
                // isLodIndexUnique = false;
                return lodIndex;
        }
        else {
            //do nothing
        }
    }
    // No LOD index matches given criterion value; return LOD level -1.
    return -1;
}

void BlendLodRenderStrategy::Update(LodNode& lodNode)
{
    if (GetLodCriterion() != 0) {
        Float criterionValue = GetLodCriterion()->GetCriterionValue(lodNode);
        bool isLodIndexUnique;
        Int lodIndex = GetLodIndex(lodNode, criterionValue, isLodIndexUnique);
        if (lodIndex >= 0) {
            if (isLodIndexUnique) {
                // Set unique LOD level if within one LOD's boundaries.
                ActivateLodLevel(lodNode, static_cast<UInt>(lodIndex));
            }
            else {
               // Blend adjacent LOD levels, if criterion value falls between two LOD boundaries.
               ActivateBlendTransition(lodNode, static_cast<UInt>(lodIndex), criterionValue);
            }
        }
    }
}

}
