//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_ThresholdCameraRenderStrategy_h
#define Candera_ThresholdCameraRenderStrategy_h

#include <Candera/Environment.h>
#include <Candera/Engine3D/Core/CameraRenderStrategy.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {

/** @addtogroup Core3D
 *  @{
 */
 
// Forward declaration.
class Node;

/**
 * @brief The class BenchmarkCameraRenderStrategy is an implementation of the abstract class CameraRenderStrategy.
 * The rendering of the nodes is done until the sum of the Node's render benchmark values (see Node::SetRenderBenchmark) exceeds the threshold.
 * At this point the rendering is paused or stopped, depending on the render pass action set for threshold (see SetRenderPassActionOnThreshold).
 */
class BenchmarkCameraRenderStrategy: public CameraRenderStrategy
{
    FEATSTD_TYPEDEF_BASE(CameraRenderStrategy);

    public:
        /**
         *  Constructor
         */
        BenchmarkCameraRenderStrategy();

        /**
         *  Descructor
         */
        virtual ~BenchmarkCameraRenderStrategy() override {}

        /**
         *  Sets the threshold for the accumulated Node's render benchmark values.
         *  When the sum of the nodes exceeds the threshold this render strategy prompts the class Renderer to
         *  pause or stop the camera render pass, according to the render pass action set for threshold.
         *  @param threshold Threshold for the accumulated Nodes render benchmark values. See method description.
         */
        void SetThreshold(Float threshold){ m_threshold = threshold; }

        /**
         *  Retrieves the threshold for the accumulated Node's render benchmark values.
         *  @return The threshold for the accumulated Node's render benchmark values.
         */
        Float GetThreshold() const { return m_threshold; }

        /**
         *  Set the render pass action for the strategy that will be returned by the function GetRenderPassAction
         *  when the threshold is exceeded. Reasonable values are PauseRenderPass or StopRenderPass.
         *  @param action Render Pass Action for strategy. This action will be applied when the threshold exceeds.
         */
        void SetRenderPassActionOnThreshold(CameraRenderStrategy::RenderPassAction action) { m_actionOnThreshold = action; }

        /**
         *  Recives the render pass action for the strategy. It will be returned by this function when the 
         *  threshold is exceeded.
         *  @return The current render pass action.
         */
        CameraRenderStrategy::RenderPassAction GetRenderPassActionOnThreshold() const { return m_actionOnThreshold; }

        FEATSTD_RTTI_DECLARATION();

    protected:
        /**
         *  Tells the Renderer if rendering of Nodes shall continue.
         *  According to the render pass action set for threshold, it returns PauseRenderPass or StopRenderPass when the threshold is exceeded.
         *  For further details see CameraRenderStrategy::GetRenderPassAction.
         *  @param nextNode specifies the next Node in the render order that shall be rendered.
         *  @return RenderPassAction to define how the Renderer shall proceed:
         *          ProceedRenderPass allows the Renderer to render the node given and to proceed with next node.
         *          PauseRenderPass pauses the Renderer before the given node is rendered. The Renderer resumes at the node given the next time the Camera is rendered.
         *          StopRenderPass stops the Renderer before the given node is rendered. The Renderer restarts at the first node in render order, the next time the Camera is rendered.
         */
        virtual CameraRenderStrategy::RenderPassAction GetRenderPassAction(const Node* nextNode) override;

        /**
         *  Called by Renderer at the beginning of each Camera's render pass and resets the threshold to zero.
         */
        virtual void OnRenderPassBegins() override;

        /**
         *  Called by Renderer each time the camera's render pass resumes from the previous node paused and resets the threshold to zero.
         */
        virtual void OnRenderPassResumes() override;

    private:
        Float m_threshold;      // The threshold benchmark value when the render pass action set shall be indicated. Default value is 0.0F.
        Float m_benchmarkSum;   // The accumulated benchmark values of Nodes since last camera render pass.
        CameraRenderStrategy::RenderPassAction m_actionOnThreshold; // The indicated render pass action if threshold has been exceeded. Default value is PauseRenderPass.
};
 
/** @} */ // end of Core3D
 
} // namespace Candera
#endif // Candera_BenchmarkCameraRenderStrategy_h
