//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Appearance.h"
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Shader.h>
#include <Candera/Engine3D/Core/Texture.h>
#include <Candera/Engine3D/Core/Material.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/RenderMode.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    FEATSTD_RTTI_DEFINITION(Appearance, Base)

    Appearance::Appearance()
    {
    }

    Appearance::Appearance(const Appearance& appearance) :
        Base(appearance),
        m_material(appearance.m_material),
        m_renderMode(appearance.m_renderMode),
        m_shader(appearance.m_shader),
        m_shaderParamSetter(appearance.m_shaderParamSetter)
    {
        for (Int i = 0; i < TextureUnitCount; i++) {
            m_textureUnits[i] = appearance.m_textureUnits[i];
        }
    }

    Appearance::SharedPointer Appearance::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        Appearance* ptr = FEATSTD_NEW(Appearance);
        if (ptr == 0) {
            FEATSTD_LOG_ERROR("Appearance create failed, out of memory.");
        }

        Appearance::SharedPointer sharedPointer(ptr);

        return sharedPointer;
    }

    Appearance::~Appearance()
    {
    }

    Appearance::SharedPointer Appearance::Clone() const
    {
        return Appearance::SharedPointer(FEATSTD_NEW(Appearance)(*this));
    }

    void Appearance::CopyProperties(const Appearance& sourceAppearance)
    {
        m_material = sourceAppearance.m_material;
        m_renderMode = sourceAppearance.m_renderMode;
        m_shader = sourceAppearance.m_shader;
        m_shaderParamSetter = sourceAppearance.m_shaderParamSetter;
        for (Int i = 0; i < TextureUnitCount; i++) {
            m_textureUnits[i] = sourceAppearance.m_textureUnits[i];
        }
    }

    bool Appearance::SetTexture(Texture::SharedPointer texture, UInt unit)
    {
        bool success = true;
        if (unit < CANDERA_MAX_TEXTURE_UNIT_COUNT) {
            m_textureUnits[static_cast<Int>(unit)] = texture;
        }
        else {
            FEATSTD_LOG_ERROR("Appearance::SetTexture: Texture unit %d exceeds max. number of textures supported by Candera.", unit);
            success = false;
        }

        return success;
    }

    Texture::SharedPointer Appearance::GetTexture(UInt unit) const {
        Texture::SharedPointer nullPointer(0);
        return (unit < CANDERA_MAX_TEXTURE_UNIT_COUNT) ?  m_textureUnits[static_cast<Int>(unit)]: nullPointer;
    }

    bool Appearance::Activate(const Node* /*node*/)
    {
        bool finalResult = true;

        if ( m_shader != 0) {
            // Activate shader.
            if (!m_shader->Activate()) {
                FEATSTD_LOG_ERROR("Shader activation failed.");
                finalResult = false;
            }
        }
        else {
            FEATSTD_LOG_ERROR("Shader is null.");
            finalResult= false;
        }

        // Activate render mode
        RenderMode::SharedPointer baseRenderMode;
        const Camera* camera = RenderDevice::GetActiveCamera();
        // The base RenderMode to inherit from is retrieved from the current active camera's RenderMode.
        if (camera != 0) {
            Appearance::SharedPointer appearance = camera->GetAppearance();
            if (appearance != 0) {
                baseRenderMode = appearance->GetRenderMode();
            }
        }
        // If the current active camera has no RenderMode set then take the default RenderMode as base RenderMode.
        if (baseRenderMode == 0) {
            baseRenderMode = Renderer::GetDefaultRenderMode();
        }

        // Activate either this Node's RenderMode or - if not set - the base RenderMode retrieved in previous step.
        // NOTE: If this behavior changes, RenderNode::AssignNodeToBin needs to be changed accordingly.
        const bool rmActOk = ((m_renderMode != 0) ? m_renderMode->Activate(baseRenderMode) : baseRenderMode->Activate());
        if (!rmActOk) {
            finalResult = false;
        }

        return finalResult;
    }

    bool Appearance::Upload()
    {
        bool finalResult = true;

        // Upload textures
        for(Int unit=0; unit < TextureUnitCount; unit++) {
            if ( m_textureUnits[unit] != 0 ) {
                const bool texUploadOk = m_textureUnits[unit]->Upload();
                if (!texUploadOk) {
                    FEATSTD_LOG_ERROR("Upload of m_textureUnits[%d] failed.", unit);
                    finalResult= false;
                }
            }
        }

        // Upload shader
        if ((m_shader == 0) || (!m_shader->Upload())) {
            FEATSTD_LOG_ERROR("Upload of shader failed.");
            finalResult = false;
        }

        return finalResult;
    }

    bool Appearance::Unload()
    {
        bool finalResult = true;

        // Unload textures.
        for(Int unit=0; unit < TextureUnitCount; unit++) {
            if ( m_textureUnits[unit] != 0 ) {
                const bool texUnloadRc = m_textureUnits[unit]->Unload();
                if (!texUnloadRc) {
                    FEATSTD_LOG_ERROR("Unload of m_textureUnits[%d] failed", unit);
                    finalResult = false;
                }
            }
        }

        // Unload shader.
        if ((m_shader == 0) || (!m_shader->Unload())) {
            FEATSTD_LOG_ERROR("Unload of shader failed");
            finalResult = false;
        }

        return finalResult;
    }

} // namespace Candera
