//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANVAS_TRANSFORMABLE_H)
#define CANDERA_CANVAS_TRANSFORMABLE_H

#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/Engine3D/Core/Node.h>

#if defined(CANDERA_LAYOUT_ENABLED)
//TODO: Layout functionality (Set/GetLayouter(), Layout(), LayoutingRectangle, PreferredSize,... ) has to be added to this Node.
//3D children will be excluded.
#endif

namespace Candera {

    class Node;
    class Canvas;
    class CanvasTransformable;

    class CanvasInterface {

    public: 

        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center of a bitmap,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector2& basePoint) const = 0;

        /**
        *  Gets the canvas axis-aligned bounding rectangle in local coordinate space which is computed by its
        *  actual content (e.g. boundaries of the image, text...).
        *  @param   boundingRectangle  Out parameter for the axis-aligned bounding rectangle in local coordinate space.
        */
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle) const = 0;

        /**
        *  Gets the CanvasTransformable of this Canvas node.
        *  @return  This nodes CanvasTransformable.
        */
        virtual CanvasTransformable& GetCanvasTransformable() = 0;

    };

    /**
      * @brief A CanvasTransformable represents an element that is transformable relative to the current Canvas plane.
      *       This means basically that transformations only affect X and Y components. This can be used to realize 2D movements.
      *       However the full 3D transformation remains available through the base class.
      *       Instances of this class cannot be used directly. Every 2D transformable canvas object needs to have one such CanvasTransformable.
      */
    class CanvasTransformable {

    public:

        friend class CanvasRenderable;
        friend class CanvasGroup;
        friend class CanvasText;
        friend class CompositeGroup;

        /**
         * Destructor.
         */
        virtual ~CanvasTransformable();


        /**
         * Concatenates the given 2D movement (translation) with current position component.
         * @param translation The translation that shall be applied.
         */
        void Translate2D(const ::Candera::Vector2& translation);


        /**
         * Multiplies the given 2D scale factors with current scale component.
         * @param scale The scale that shall be applied.
         */
        void Scale2D( const ::Candera::Vector2& scale);

        /**
         * Concatenates the given 2D rotation angles with current rotation component.
         * @param   rotation    Defines the rotation angle in degrees.
         */
        void Rotate2D( const ::Candera::Float rotation);

        //2D specific rectangles.

        /**
         *  Gets the axis-aligned bounding rectangle local canvas coordinate space.
         *  If no parent canvas is found,
         *  @param  boundingRectangle   Out parameter for the axis-aligned bounding rectangle in world coordinate space.
         *  @param  traverse            Determines whether to get the union of all child node's bounding rectangles (Deep) or
         *                              only the bounding rectangle of that node (Flat).
         *  @return  True whether bounding rectangle could be calculated. False otherwise.
         */
        bool GetCanvasAxisAlignedBoundingRectangle(Rectangle& boundingRectangle, Node::Traversing traverse = Node::Flat);

        /**
         *  Get oriented bounding rectangle in local canvas coordinate space.
         *  Only the current node is considered and will not include child nodes.
         *  @param   points  Out parameter containing the 4 points of the bounding rectangle in world space.
         *  @return  True whether bounding rectangle could be calculated. False otherwise.
         */
        bool GetCanvasOrientatedBoundingRectangle(Vector3 points[4]);


        /**
         *  Sets the bounding rectangle for this node to a fixed value.
         *  As long as the bounding rectangle for the node is set manually with this method, it is no longer calculated depending on its content dimensions.
         *  @param  boundingRectangle   The bounding rectangle to be set.
         */
        void SetBoundingRectangle(const Rectangle& boundingRectangle);

        /**
         *  Retrieves the manually set bounding rectangle.
         *  @return     The set bounding rectangle.
         */
        const Rectangle& GetBoundingRectangle() const;

        /**
         *  Check if there is a bounding rectangle set for this node.
         *  @return     True if the bounding rectangle is set.
         */
        bool IsBoundingRectangleSet() const;

        /**
         *  Clears the set bounding rectangle.
         *  From now on the bounding rectangle will be calculated depending on its content dimensions.
         *  @return     true if successful, false otherwise.
         */
        void ClearBoundingRectangle();

        /**
         *  Gets the canvas axis-aligned bounding rectangle in local coordinate space.
         *  If the bounding rectangle has been set with SetBoundingRectangle() then this rectangle is returned, otherwise
         *  the bounding rectangle is computed by its actual content (e.g. boundaries of the image, text...) by calling
         *  GetComputedBundingRectangle() internally.
         *  @param[out] boundingRectangle   Out parameter for the axis-aligned bounding rectangle in local coordinate space.
         */
        void GetEffectiveBoundingRectangle(Rectangle& boundingRectangle) const;


#if defined(CANDERA_LAYOUT_ENABLED)
        //Layout rectangles.

       /**
        *  3D objects have their position at center. So this method has a normally a recalculation to the upper left corner.
        *  For Canvas nodes this is not required. By default it is already positioned at 0,0.
        *  @return  layout start position
        */
        const Vector3 GetLayoutStartPosition() const;
#endif

        /**
         * Invalidates the current parent canvas. New parent canvas has to be looked for at the next call GetParentCanvas.
         */
        void InvalidateParentCanvas() { m_isParentCanvasValid = false; }

        /**
          * Used to query the parent Canvas. Returns 0 if no canvas was found.
          * 2D specific functions are only available if a parent Canvas exists.
          * @return The parent canvas, 0 if there isn't any.
          */
        Canvas* GetParentCanvas();

        /**
         * Returns whether a node is a canvas node or not.
         * @return Whether a node is a canvas node (true) or not (false).
         */
        static bool IsCanvasNode(const Node& node);

        /**
        * Returns whether a node has a parent canvas or not.
        * @return Whether a node has a parent canvas  (true) or not (false). True includes the node being a Canvas.
                  False includes the node not being a CanvasNode.
        */
        static bool HasParentCanvas(const Node& node);

        /**
         * Computes the current local XY aligned bounding rectangle.
         * @param transformation: The transformation to compute the rectangle from.
         * @param boundingRectangle The bounding rectangle to retrieve;
         */
        static void ComputeLocalXYAlignedBoundingRectangle(const Matrix4& transformation, Rectangle& boundingRectangle);

    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(CanvasTransformable);

        /**
        * Constructor.
        * @param node The node reference this class refers to.
        */
        CanvasTransformable(Node& node, CanvasInterface& canvasInterface);

        bool m_isBoundingRectangleSet;
        bool m_isParentCanvasValid;
        Rectangle m_boundingRectangle;
        Node& m_node;
        CanvasInterface& m_canvasInterfance;
        Canvas* m_parentCanvas;
    };

}


#endif  // CANDERA_CANVAS_TRANSFORMABLE_H
