//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CanvasTransformable.h"
#include <Candera/Engine3D/Canvas/Canvas.h>
#include <Candera/Engine3D/Canvas/CanvasRenderable.h>
#include <Candera/Engine3D/Canvas/CanvasText.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Core/CompositeGroup.h>

namespace Candera {


    class CanvasAxisAlignedBoundingRectangleTraverser : public ConstTreeTraverser {
    public:
        CanvasAxisAlignedBoundingRectangleTraverser() : m_isComputationValid(true) {}

        const Rectangle& GetBoundingRectangle() const { return m_boundingRect; }

        bool IsComputationValid() const { return m_isComputationValid; }

    protected:
        virtual TraverserAction ProcessNode(const Node& node) override{
            Rectangle boundingRect;
            CanvasTransformable* transformable = 0;
            if (node.IsTypeOf(CanvasRenderable::GetTypeId())) {
                transformable = &(static_cast<CanvasRenderable&>(const_cast<Node&>(node)).GetCanvasTransformable());
            }
            else if (node.IsTypeOf(CanvasGroup::GetTypeId())) {
                transformable = &(static_cast<CanvasGroup&>(const_cast<Node&>(node)).GetCanvasTransformable());
            }
            else if (node.IsTypeOf(CanvasText::GetTypeId())) {
                transformable = &(static_cast<CanvasText&>(const_cast<Node&>(node)).GetCanvasTransformable());
            }
            else if (node.IsTypeOf(CompositeGroup::GetTypeId())) {
                const CompositeGroup& compositeGroup = static_cast<const CompositeGroup&>(node);
                if (compositeGroup.GetCompositeGroupMode() == CompositeGroup::CanvasGroupMode) {
                    transformable = &(const_cast<CompositeGroup&>(compositeGroup).GetCanvasTransformable());
                }
            }
            else {
                //Do nothing.
            }
             
            if (transformable != 0) {
                if (!transformable->GetCanvasAxisAlignedBoundingRectangle(boundingRect)) {
                    m_isComputationValid = false;
                }
                m_boundingRect.Union(boundingRect);

                return ProceedTraversing;
            }
            else {
                return StopTraversingForDescendants;
            }
        }

    private:
        Rectangle m_boundingRect;
        bool m_isComputationValid;
    };


    CanvasTransformable::CanvasTransformable(Node& node, CanvasInterface& canvasInterface) :
        m_isBoundingRectangleSet(false),
        m_isParentCanvasValid(false),
        m_boundingRectangle(0.0F, 0.0F, -1.0F, -1.0F),
        m_node(node),
        m_canvasInterfance(canvasInterface),
        m_parentCanvas(0)
    {

    }

    CanvasTransformable::~CanvasTransformable()
    {

    }


    void CanvasTransformable::Translate2D(const ::Candera::Vector2& translation)
    {
        m_node.Translate(Vector3(translation.GetX(), translation.GetY(), 0.0F));
    }


    void CanvasTransformable::Scale2D(const ::Candera::Vector2& scale)
    {
        m_node.Scale(Vector3(scale.GetX(), scale.GetY(), 1.0F));
    }

    void CanvasTransformable::Rotate2D(const ::Candera::Float rotation)
    {
        m_node.Rotate(Vector3(0.0F, 0.0F, rotation));
    }



    bool CanvasTransformable::GetCanvasAxisAlignedBoundingRectangle(Rectangle& boundingRectangle, Node::Traversing traverse /*= Node::Flat*/)
    {
        bool result = false;

        if (traverse == Node::Flat) {
            GetEffectiveBoundingRectangle(boundingRectangle);

            Canvas* parentCanvas = GetParentCanvas();

            if (parentCanvas != 0) {
                Matrix4 inverseCanvasTransform = parentCanvas->GetWorldTransform();
                inverseCanvasTransform.Inverse();

                Matrix4 canvasLocalTransform = m_node.GetWorldTransform() * inverseCanvasTransform;

                ComputeLocalXYAlignedBoundingRectangle(canvasLocalTransform, boundingRectangle);
                result = true;
            }
        }
        else {
            CanvasAxisAlignedBoundingRectangleTraverser traverser;
            traverser.Traverse(m_node);
            boundingRectangle = traverser.GetBoundingRectangle();
            result = traverser.IsComputationValid();
        }


        return result;
    }

    bool CanvasTransformable::GetCanvasOrientatedBoundingRectangle(Vector3 points[4])
    {
        bool result = false;
        Rectangle boundingRectangle;
        GetEffectiveBoundingRectangle(boundingRectangle);

        const Float left = boundingRectangle.GetLeft();
        const Float top = boundingRectangle.GetTop();
        const Float width = boundingRectangle.GetWidth();
        const Float height = boundingRectangle.GetHeight();

        Canvas* parentCanvas = GetParentCanvas();

        if (parentCanvas != 0) {
            Matrix4 inverseCanvasTransform = parentCanvas->GetWorldTransform();
            inverseCanvasTransform.Inverse();

            Matrix4 canvasLocalTransform = m_node.GetWorldTransform() * inverseCanvasTransform;
            
            points[0] = Vector3(left, top, 0.0F);
            points[1] = Vector3(left +  width, top, 0.0F);
            points[2] = Vector3(left + width, top +  height, 0.0F);
            points[3] = Vector3(left, top + height, 0.0F);

            for (UInt8 i = 0; i < 4; i++) {
                points[i].TransformCoordinate(canvasLocalTransform);
            }

            result = true;

        }

        return result;
    }


    void CanvasTransformable::SetBoundingRectangle(const Rectangle& boundingRectangle)
    {
        m_boundingRectangle = boundingRectangle;
        m_isBoundingRectangleSet = true;
    }


    const Rectangle& CanvasTransformable::GetBoundingRectangle() const
    {
        return m_boundingRectangle;
    }
       

    bool CanvasTransformable::IsBoundingRectangleSet() const
    {
        return m_isBoundingRectangleSet;
    }

    void CanvasTransformable::ClearBoundingRectangle()
    {
        m_isBoundingRectangleSet = false;
    }

    void CanvasTransformable::GetEffectiveBoundingRectangle(Rectangle& boundingRectangle) const
    {
        if (m_isBoundingRectangleSet) {
            boundingRectangle = m_boundingRectangle;
        }
        else {
            m_canvasInterfance.GetComputedBoundingRectangle(boundingRectangle);
        }

        FEATSTD_UNUSED(boundingRectangle);
    }


#if defined(CANDERA_LAYOUT_ENABLED)
    const Candera::Vector3 CanvasTransformable::GetLayoutStartPosition() const
    {
        return Vector3();
    }
#endif

    Canvas* CanvasTransformable::GetParentCanvas()
    {
        
        if (!m_isParentCanvasValid) {
            Node* node = &m_node;
            Canvas* newParent = Dynamic_Cast<Canvas*>(node);

            while ((newParent == 0) && (node->GetParent() != 0)) {
                node = node->GetParent();
                newParent = Dynamic_Cast<Canvas*>(node);
            }
            m_parentCanvas = newParent;
            if (newParent != 0) {
                m_isParentCanvasValid = true;
            }

        }

        return m_parentCanvas;
    }


    bool CanvasTransformable::IsCanvasNode(const Node& node)
    {
        bool result = (node.IsTypeOf(CanvasGroup::GetTypeId()) ||
                       node.IsTypeOf(CanvasText::GetTypeId()) ||
                       node.IsTypeOf(CanvasRenderable::GetTypeId()));

        if (!result) {
            if (node.IsTypeOf(CompositeGroup::GetTypeId())) {
                result = static_cast<const CompositeGroup&>(node).GetCompositeGroupMode() == CompositeGroup::CanvasGroupMode;
            }
        }

        return result;
    }


    bool CanvasTransformable::HasParentCanvas(const Node& node)
    {
        bool result = false;

        if (node.IsTypeOf(CanvasGroup::GetTypeId())) {
            const CanvasGroup* canvasGroup = Dynamic_Cast<const CanvasGroup*>(&node);
            if (canvasGroup != 0) {
                if (const_cast<CanvasGroup*>(canvasGroup)->GetCanvasTransformable().GetParentCanvas() != 0) {
                    result = true;
                }
            }
        }
        else if (node.IsTypeOf(CanvasRenderable::GetTypeId())){
            const CanvasRenderable* canvasRenderable = Dynamic_Cast<const CanvasRenderable*>(&node);
            if (canvasRenderable != 0) {
                if (const_cast<CanvasRenderable*>(canvasRenderable)->GetCanvasTransformable().GetParentCanvas() != 0) {
                    result = true;
                }
            }
        }
        else if (node.IsTypeOf(CompositeGroup::GetTypeId())){
            const CompositeGroup* compositeGroup = Dynamic_Cast<const CompositeGroup*>(&node);
            if (compositeGroup != 0) {
                if (compositeGroup->GetCompositeGroupMode() == CompositeGroup::CanvasGroupMode) {
                    if (const_cast<CompositeGroup*>(compositeGroup)->GetCanvasTransformable().GetParentCanvas() != 0) {
                        result = true;
                    }
                }
            }
        }
        else if (node.IsTypeOf(CanvasText::GetTypeId())) {
            const CanvasText* canvasText = Dynamic_Cast<const CanvasText*>(&node);
            if (canvasText != 0) {
                if (const_cast<CanvasText*>(canvasText)->GetCanvasTransformable().GetParentCanvas() != 0) {
                    result = true;
                }
            }
        }
        else {
            //Do nothing.
        }

        return result;
    }


    void CanvasTransformable::ComputeLocalXYAlignedBoundingRectangle(const Matrix4& transformation, Rectangle& boundingRectangle)
    {
        
        const Float left = boundingRectangle.GetLeft();
        const Float top = boundingRectangle.GetTop();
        const Float width = boundingRectangle.GetWidth();
        const Float height = boundingRectangle.GetHeight();

        Vector3 points[4];
        points[0] = Vector3(left, top, 0.0F);
        points[1] = Vector3(left + width, top, 0.0F);
        points[2] = Vector3(left + width, top + height, 0.0F);
        points[3] = Vector3(left, top + height, 0.0F);

        Float maxX = 0.0F;
        Float minX = 0.0F;
        Float maxY = 0.0F;
        Float minY = 0.0F;

        for (UInt8 i = 0; i < 4; i++) {
            points[i].TransformCoordinate(transformation);

            if (i == 0) {
                maxX = points[i].GetX();
                minX = points[i].GetX();
                maxY = points[i].GetY();
                minY = points[i].GetY();
            }
            else {
                if (points[i].GetX() > maxX) {
                    maxX = points[i].GetX();
                }

                if (points[i].GetX() < minX) {
                    minX = points[i].GetX();
                }

                if (points[i].GetY() > maxY) {
                    maxY = points[i].GetY();
                }

                if (points[i].GetY() < minY) {
                    minY = points[i].GetY();
                }
            }
        }

        boundingRectangle = Rectangle(minX, minY, (maxX - minX), (maxY - minY));
    }

}

