//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANVAS_TEXT_H)
#define CANDERA_CANVAS_TEXT_H

#include <Candera/TextEngine/Internal/TextProcessProperties.h>
#include <Candera/TextEngine/Internal/TextProcessResult.h>

#include <Candera/Engine3D/Core/Node.h>
#include <Candera/Engine3D/Canvas/CanvasRenderable.h>
#include <Candera/Engine3D/Canvas/CanvasTransformable.h>
#include <Candera/EngineBase/Common/Color.h>

CANDERA_UNIT_TEST_TESTCASE_DECLARATION(CanvasCloneTextTest, TestRunCanvasText)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(CanvasTextTest, TestRun_Preprocessing)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(CanvasTextTest, TestRun_SimpleIntegration)
namespace Candera {
    /**
    *  @addtogroup Core3D
    *  @{
    */

    /**
    * @brief CanvasText represents a text node within a 3D Canvas object.
    */
    class CanvasText : public Node, public CanvasInterface, public Candera::TextRendering::Internal::ITextProcessUser {
        
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(CanvasCloneTextTest, TestRunCanvasText);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(CanvasTextTest, TestRun_Preprocessing);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(CanvasTextTest, TestRun_SimpleIntegration);
        FEATSTD_TYPEDEF_BASE(Node);
    public:
        FEATSTD_RTTI_DECLARATION();
        typedef  Candera::TextRendering::Internal::TruncationMode TruncationMode;
        typedef  Candera::TextRendering::Internal::HorizontalTextAlignment HorizontalTextAlignment;
   
        /**
        *  Gets the base point in object coordinate space. The base point is typically in the center of a bitmap,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in object coordinate space.
        */
        virtual void GetBasePoint(Vector2& basePoint) const override;

        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector3& basePoint) const;

        /**
        *  Gets the canvas axis-aligned bounding rectangle in object coordinate space which is computed by its
        *  actual content (e.g. boundaries of the image, text...).
        *  @param   boundingRectangle  Out parameter for the axis-aligned bounding rectangle in object coordinate space.
        */
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle) const override;
     
        /**
        *  Gets the CanvasTransformable of this Canvas node.
        *  @return  This nodes CanvasTransformable.
        */
        virtual CanvasTransformable& GetCanvasTransformable() override;

        /**
        * Get text which shall be shown.
        * @return text which shall be shown
        */
        FeatStd::String GetText() const;

        /**
        * Set text which shall be shown.
        * @param val text which shall be shown
        */
        void SetText(FeatStd::String const& val);

        /**
        * Get style used for rendering.
        * @return style of the text
        */
        Candera::TextRendering::SharedStyle::SharedPointer GetStyle() const;

        /**
        * Set style to be used for rendering.
        * @param val style of the text
        */
        void SetStyle(Candera::TextRendering::SharedStyle::SharedPointer val);

        /**
        * Get rendered truncation text.
        * @return If truncation mode is Ellipsis, an ellipsis is returned.
        * If truncation is set to none / without replacement text, an empty string is returned.
        * Otherwise the set truncation text is returned.
        */
        FeatStd::String GetSelectedTruncationText() const;

        /**
        * Get truncation text which is set by SetTruncationText. 
        * This is not the active truncation text which depends on the mode.
        * To get the active/selected truncation text use GetSelectedTruncationText()
        * @return truncation text which were set 
        */
        FeatStd::String GetTruncationText() const;

        /**
        * Sets truncation text.
        * The text will be ignored when truncation mode is set to none or ellipsis.
        * @param truncText truncation text which shall be set
        */
        void SetTruncationText(FeatStd::String const& truncText);

        /**
        * Get currently active truncation mode.
        * @return active truncation mode
        */
        TruncationMode::Enum GetTruncationMode() const;

        /**
        * Set active truncation mode.
        * @param truncMode truncation mode which shall be set
        */
        void SetTruncationMode(TruncationMode::Enum truncMode);

        /**
        * @return The word wrapping configuration for the given CanvasText.
        */
        bool IsWordWrapEnabled() const;

        /**
        * Sets the word wrapping configuration.
        * If word wrapping is enabled then the lines that exceed the client area will be split
        * during arrangement into multiple lines. The lines will be split at word boundaries.
        * @param wordWrapEnabled    The word wrapping enabled flag (disabled by default).
        */
        void SetWordWrapEnabled(bool wordWrapEnabled);

        /**
         * @return The multi line configuration for the given CanvasText.
         */
        bool IsMultiLineEnabled() const;

        /**
        * Sets the multi line configuration.
        * If multi line is disabled then the new line control characters will be ignored and text
        * will always be rendered in one line.
        * @param multiLineEnabled    The multi line enabled flag (enabled by default).
        */
        void SetMultiLineEnabled(bool multiLineEnabled);

        /**
        * @return The line spacing configured for the given CanvasText.
        */
        Candera::TextRendering::PixelSize GetLineSpacing() const;

        /**
        * Sets the line spacing value.
        * The line spacing is a metric usually defined by the Font configured for text rendering. To use
        * the default value of the Font, set 0 as the line spacing property. Any other positive value
        * will override the default value with an object space fixed height (independent on the font height).
        * @param lineSpacing  Line spacing value.
        */
        void SetLineSpacing(Candera::TextRendering::PixelSize lineSpacing);

        /**
        * @return The TextAlignment configured for the given CanvasText.
        */
        HorizontalTextAlignment::Enum GetHorizontalTextAlignment() const;

        /**
        * Sets the TextAlignment type.
        * The TextAlignment type, combined with the culture text direction, is used to
        * define how text will be horizontally aligned within its layout area.
        * @param textAlignment    The TextAlignment value.
        * @see TextAlignment
        */
        void SetHorizontalTextAlignment(HorizontalTextAlignment::Enum textAlignment);

        /**
        * @return The CanvasText dimension not considering layout restrictions.
        */
        FEATSTD_LINT_NEXT_EXPRESSION(1511, "False positive: it is not hiding SingleLinkedList<Candera::NodeListener*>::GetSize")
        Candera::TextRendering::TextSize GetSize() const;

        /**
        * Set the dimension of the CanvasText not considering layout restrictions.
        * Based on the size, the text will be aligned.
        * @param val  Size value.
        */
        void SetSize(Candera::TextRendering::TextSize val);
        
        /**
        * Gets raw color of the text.
        * @return the color of the text 
        */
        Candera::Color GetTextColor() const;

        /**
        * Sets raw color of the text.
        * @param color of the text
        */
        void SetTextColor(Candera::Color const& color);

        /**
        * The layout restriction is used to provide additional external influence on the possible dimension.
        * The CanvasText tries to align as much as possible with its internal size. However, the external influence
        * defines additional boundaries for the the size calculation.
        * @return external size restriction
        */
        Candera::TextRendering::TextSize GetLayoutRestriction() const;

        /**
        * The layout restriction is used to provide additional external influence on the possible dimension.
        * The CanvasText tries to align as much as possible with its internal size. However, the external influence
        * defines additional boundaries for the the size calculation.
        * @param val defines the external size restriction
        */
        void SetLayoutRestriction(Candera::TextRendering::TextSize val);

        /**
        * Text Clipping defines a text which will be clipped when it is outside of the maximum defined area.
        * This happens when the text is longer than the provided space. If the clip is not enabled, nothing will be clipped.
        * @return true if clipping is enabled otherwise false
        */
        bool IsClipEnabled() const;

        /**
        * Text Clipping defines a text which will be clipped when it is outside of the maximum defined area.
        * This happens when the text is longer than the provided space. If the clip is not enabled, nothing will be clipped.
        * @param val enables/disables clipping
        */
        void SetClipEnabled(bool val);
        /**
        * Allocates memory and creates a CanvasText clone.
        * @return a newly allocated Clone of this CanvasText
        */
        virtual CanvasText * Clone() const override;

        /**
        * Allocates memory and creates a CanvasText. 
        * @return a newly allocated CanvasText
        */
        static CanvasText * Create();


        /**
        * Destroys the Canvas text
        */
        virtual ~CanvasText();

        /**
        * Get the last valid result data containing the GlyphIterator, bounding boxes and offset.
        * It is advised to store the result in a variable as the pointer can be changed asynchronously and calculations can be triggered.
        * @return text processing result data
        */
        const Candera::TextRendering::Internal::TextProcessResult::SharedPointer GetActiveProcessResult() const;

        /**
        * Returns the currently used text cache in which the glyphs are stored.
        * CanvasText now supports GlyphAtlas,
        * so this cache will be returned.
        * @return glyph cache
        */
        virtual Candera::TextRendering::Internal::TextProcessProperties::TextCache * GetGlyphCache() override;

        /**
        * OnAncestorAdded adds the CanvasText to the attached canvas. 
        * If no canvas is available in the scene graph hierarchy, it cannot be attached and therefore it will not be rendered.
        * The canvas text will handle the required render information afterwards.
        * @param scene of the scene tree in which the CanvasText will be located
        */
        virtual void OnAncestorAdded(Scene* scene) override;

        /**
        * OnAncestorRemoved removes the CanvasText from an attached canvas.
        * If it is not attached to a canvas, it will not have to detach from a canvas as there is no canvas available.
        * The canvas text will handle the required render information afterwards.
        * @param scene of the scene tree in which the CanvasText was located
        */
        virtual void OnAncestorRemoved(Scene * scene) override;

        /**
        *  Upload the appearance of the CanvasText.
        *  @return True, if uploading of the appearance was successful, false otherwise.
        */
        virtual bool Upload() override;

        /**
        *  Unload the appearance of the CanvasText.
        *  @return True, if unloading of the appearance was successful, false otherwise.
        */
        virtual bool Unload() override;

        /**
        *  Called when one of the transforms (translation, rotation, scale, general transform) has changed.
        *  Useful for updating dependent properties. Implementing classes will probably want to propagate the call up the hierarchy.
        */
        virtual void OnCompositeTransformChanged() override;

        /**
        *  Tests if the bounding box of this mesh intersects with the given Line.
        *  @param line               The intersection Line.
        *  @param distance [out]     The distance from the nearest geometry to the starting point of the Line.
        *                            If no intersection is found then this param stays unchanged.
        *  @return                   True if an intersection is found with this mesh, otherwise false.
        */
        virtual bool IsLineIntersectingGeometry(const Line& line, Float& distance) const override;

    protected:
        typedef Candera::TextRendering::Internal::TextProcessInvalidation TextProcessInvalidation;

        /**
        * Default constructor initializes CanvasText
        */
        CanvasText();

        /**
        * Copy constructor.
        * It copies the properties of the text - but the results have to be generated newly.
        * @param elem to be copied
        */
        explicit CanvasText(CanvasText const& elem);

        /**
        * Sets the invalidation state of the CanvasText.
        * @param invalidationType represents the current state of the CanvasText
        */
        virtual void InvalidateProperties(TextProcessInvalidation::Enum invalidationType) override;

        /**
        * Helper function which produces the active process result.
        * This method checks whether a property has been changed and then updates the text. Otherwise it will not be updated.
        * To trigger the update invalidateProperties has to be called with TextProcessInvalidation::Enum != ValidAll. Based on the 
        * type only parts of the Text have to be updated.
        */
        virtual void UpdateText();

        /**
        * Empty render method - as the Canvas will handle the render part and integrates it into the render system
        */
        virtual void Render() override;

        /**
        * Disposal of the object
        */
        virtual void DisposeSelf() override;

        /**
        *  Computes axis-aligned bounding box (AABB) in object coordinate space.
        *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
        *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
        *  @return true if bounding box was computed successfully and false if not because of missing text.
        */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

        /**
        *  Tests if geometry of this mesh intersects with a pick in absolute Viewport coordinate in pixels.
        *  @remark                   This interface is usually used for object selection in scene editors.
        *  @param camera             The view camera. (i.e. the current camera)
        *  @param x                  The x value as an absolute Viewport coordinate in pixels starting from top-left.
        *  @param y                  The y value as an absolute Viewport coordinate in pixels starting from top-left.
        *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
        *                            If no intersection is found then this param stays unchanged.
        *  @return                   True if an intersection is found with this mesh, otherwise false.
        */
        virtual bool IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const override;
#ifdef CANDERA_LAYOUT_ENABLED
        /**
        *  Returns the "offset" coordinate which is required to position 3D objects correctly when using layout.
        *  Normally, 3D objects are centered but the layouter thinks its position is the "upper-left" corner.
        *  This will be replaced as soon as there is a more appropriate handling (e.g. own pivot point)
        *  @return  layout start position
        */
        virtual const Vector3 GetLayoutStartPosition() const override;

#endif

        CanvasTransformable m_canvasTransformable; ///< The transformable object of the CanvasText

    private:

        // Methods for Layout integration and further implementation details.
        // They are not fully functional and therefore not completely exposed into the API yet.

        /**
        * Returns the active culture of the CanvasText. If no culture is explicitly set by SetCulture,
        * the currently active culture will be returned as culture.
        * @return the custom culture if set else the active culture
        */
        Candera::Globalization::Culture::SharedPointer GetCulture() const;

        /**
        * Allows to explicitly set a culture to the CanvasText. If no culture is set (empty shared pointer),
        * the currently active culture will be used as default culture.
        * @param val sets the custom culture for this CanvasText
        */
        void SetCulture(Candera::Globalization::Culture::SharedPointer val);

        /**
        * The right to left direction has impact on the interpretation of the horizontal alignment.
        * If this direction is right to left, the horizontal alignment will be inverted.
        * @return true when right to left direction is set.
        */
        bool IsRightToLeftLayoutDirection() const;

        /**
        * The right to left direction has impact on the interpretation of the horizontal alignment.
        * If this direction is right to left, the horizontal alignment will be inverted.
        * @param val sets the right to left direction active when true. 
        */
        void SetRightToLeftLayoutDirection(bool val);

        /**
        * This flag switches between two line spacing modes.
        * First mode (true): The line spacing is an absolute pixel value (default).
        * Second mode (false): The line spacing is a ratio of line spacing to font height.
        * @return the current mode of the two modes described above.
        */
        bool IsPixelwiseLineSpacing() const;

        /**
        * This flag switches between two line spacing modes.
        * First mode (true): The line spacing is an absolute pixel value (default).
        * Second mode (false): The line spacing is a ratio of line spacing to font height.
        * @param val sets the current mode of the two modes described above.
        */
        void SetPixelwiseLineSpacing(bool val);

        /**
        * The glyph spacing defines the additional space given between each glyphs.
        * @return the current glyph spacing.
        */
        Candera::TextRendering::PixelSize GetGlyphSpacing() const;

        /**
        * The glyph spacing defines the additional space given between each glyphs.
        * @param val sets the current glyph spacing.
        */
        void SetGlyphSpacing(Candera::TextRendering::PixelSize val);

        void ResolveAutoHorizontalAlignment(HorizontalTextAlignment::Enum layoutAlignment);

        void SetResultOffset(Int16 x, Int16 y);

        Candera::TextRendering::Internal::TextProcessProperties m_activeProperties; ///< member which holds all the possible textengine properties
        Candera::TextRendering::Internal::TextProcessResult::SharedPointer m_activeResult;/// < member which holds the results after the textengine preprocessed the properties
        Candera::Color m_textColor; ///< color of the text additionally to the appearance settings
        TextProcessInvalidation::Enum m_isInvalid;///< invalidation state of the CanvasText
        HorizontalTextAlignment::Enum m_setAlignment;

        Canvas * m_parentalCanvas; ///< parent canvas which holds this canvas (is required as this canvas is needed after CanvasTransformable cannot track the parent anymore)

        friend class Canvas;
#ifdef CANDERA_LAYOUT_ENABLED
        friend class CanvasTextLayouter;
#endif
        Float m_previousEffectiveAlphaValue;
        bool m_previousIsRenderingEnabled;
    };

    /** @} */ // end of Core3D
} // namespace Candera

#endif  // CANDERA_CANVAS_TEXT_H
