//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CanvasText.h"

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <Candera/Engine3D/Canvas/Canvas.h>
#include <Candera/Engine3D/Core/GlyphAtlas3DGlyphCacheAccess.h>
#include <Candera/Engine3D/Mathematics/Math3D.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/System/Mathematics/Line.h>
#include <Candera/TextEngine/Internal/TextProcessHelper.h>
#include <Candera/TextEngine/Internal/TextProcessProperties.h>
#include <Candera/TextEngine/Internal/TextProcessResult.h>
#ifdef CANDERA_LAYOUT_ENABLED
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Layout/CanvasTextLayouter.h>
#endif

using namespace Candera::TextRendering;
using namespace Candera::TextRendering::Internal;

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);
    static void TextRectToRectangle(Rectangle& out, const TextRendering::TextRect& in)
    {
        if (in.GetTop() < in.GetBottom()) {
            out.SetTop(static_cast<Float>(in.GetTop()));
            out.SetHeight(static_cast<Float>(in.GetHeight()));
        }
        else {
            out.SetTop(0.0F);
            out.SetHeight(0.0F);
        }
        if (in.GetLeft() < in.GetRight()) {
            out.SetLeft(static_cast<Float>(in.GetLeft()));
            out.SetWidth(static_cast<Float>(in.GetWidth()));
        }
        else {
            out.SetLeft(0.0F);
            out.SetWidth(0.0F);
        }
    }
    void CanvasText::GetBasePoint(Vector2& basePoint) const
    {
        Rectangle boundingRectangle;
        GetComputedBoundingRectangle(boundingRectangle);
        basePoint.SetX(boundingRectangle.GetLeft() + (boundingRectangle.GetWidth() * 0.5F));
        if ((!m_activeResult.PointsToNull()) && (!m_activeResult->GetStyle().PointsToNull())) {
            const TextRendering::Metrics& metrics = m_activeResult->GetStyle()->GetMetrics();
            Float ascender = static_cast<Float>(metrics.ascender - 1);
            basePoint.SetY(ascender);
        }
    }


    void CanvasText::GetBasePoint(Vector3& basePoint) const
    {
        Vector2 basePoint2D;
        GetBasePoint(basePoint2D);

        basePoint.SetX(basePoint2D.GetX());
        basePoint.SetY(basePoint2D.GetY());
        basePoint.SetZ(GetPosition().GetZ());
    }

    Candera::CanvasTransformable& CanvasText::GetCanvasTransformable() { return m_canvasTransformable; }

    void CanvasText::GetComputedBoundingRectangle(Rectangle& boundingRectangle) const
    {
        CanvasText * canvasText = const_cast<CanvasText*>(this);
        FEATSTD_DEBUG_ASSERT(canvasText != 0);
        canvasText->UpdateText();
        if (!m_activeResult.PointsToNull()) {
            TextRectToRectangle(boundingRectangle, m_activeResult->GetResultLayoutRectangle());
        }
    }

    FeatStd::String CanvasText::GetText() const
    {
        return m_activeProperties.GetText();
    }


    void CanvasText::SetText(FeatStd::String const& val)
    {
        m_activeProperties.SetText(val);
    }


    Candera::TextRendering::SharedStyle::SharedPointer CanvasText::GetStyle() const
    {
        return m_activeProperties.GetStyle();
    }


    void CanvasText::SetStyle(Candera::TextRendering::SharedStyle::SharedPointer val)
    {
        m_activeProperties.SetStyle(val);
    }


    FeatStd::String CanvasText::GetSelectedTruncationText() const
    {
        return m_activeProperties.GetSelectedTruncationText();
    }


    FeatStd::String CanvasText::GetTruncationText() const
    {
        return m_activeProperties.GetTruncationText();
    }


    void CanvasText::SetTruncationText(FeatStd::String const& truncText)
    {
        m_activeProperties.SetTruncationText(truncText);
    }


    TruncationMode::Enum CanvasText::GetTruncationMode() const
    {
        return  m_activeProperties.GetTruncationMode();
    }


    void CanvasText::SetTruncationMode(TruncationMode::Enum truncMode)
    {
        m_activeProperties.SetTruncationMode(truncMode);
    }


    bool CanvasText::IsWordWrapEnabled() const
    {
        return m_activeProperties.IsWordWrapEnabled();
    }


    void CanvasText::SetWordWrapEnabled(bool wordWrapEnabled)
    {
        m_activeProperties.SetWordWrapEnabled(wordWrapEnabled);
    }


    bool CanvasText::IsMultiLineEnabled() const
    {
        return m_activeProperties.IsMultiLineEnabled();
    }


    void CanvasText::SetMultiLineEnabled(bool multiLineEnabled)
    {
        m_activeProperties.SetMultiLineEnabled(multiLineEnabled);
    }


    Candera::TextRendering::PixelSize CanvasText::GetLineSpacing() const
    {
        return m_activeProperties.GetLineSpacing();
    }


    void CanvasText::SetLineSpacing(Candera::TextRendering::PixelSize lineSpacing)
    {
        m_activeProperties.SetLineSpacing(lineSpacing);
    }


    HorizontalTextAlignment::Enum CanvasText::GetHorizontalTextAlignment() const
    {
        return m_setAlignment;
    }


    void CanvasText::SetHorizontalTextAlignment(HorizontalTextAlignment::Enum textAlignment)
    {
        m_activeProperties.SetHorizontalTextAlignment(textAlignment);
        if (m_setAlignment != textAlignment) {
            m_setAlignment = textAlignment;
            InvalidateProperties(TextProcessInvalidation::InvalidAll);
        }

    }


    Candera::TextRendering::TextSize CanvasText::GetSize() const
    {
        return m_activeProperties.GetInnerSizeRestriction();
    }


    void CanvasText::SetSize(Candera::TextRendering::TextSize val)
    {
        m_activeProperties.SetInnerSizeRestriction(val);
    }


    Candera::Color CanvasText::GetTextColor() const
    {
        return m_textColor;
    }

    Candera::TextRendering::PixelSize CanvasText::GetGlyphSpacing() const
    {
        return m_activeProperties.GetGlyphSpacing();
    }


    void CanvasText::SetGlyphSpacing(Candera::TextRendering::PixelSize val)
    {
        m_activeProperties.SetGlyphSpacing(val);
    }


    void CanvasText::ResolveAutoHorizontalAlignment(HorizontalTextAlignment::Enum layoutAlignment)
    {
        FEATSTD_DEBUG_ASSERT(layoutAlignment != HorizontalTextAlignment::Auto);
        m_activeProperties.SetHorizontalTextAlignment(layoutAlignment);
    }


    void CanvasText::SetResultOffset(Int16 x, Int16 y)
    {
        TextCoordinate coord(static_cast<PixelPosition>(x), static_cast<PixelPosition>(y));
        if ((m_activeResult->GetOffset().GetX() != x) || (m_activeResult->GetOffset().GetY() != y)) {
            m_activeResult->SetOffset(coord);
        }
    }

    void CanvasText::SetTextColor(Candera::Color const& color)
    {
        m_textColor = color;
    }

    Candera::CanvasText * CanvasText::Clone() const
    {
        return CANDERA_NEW(CanvasText)(*this);
    }

    Candera::CanvasText * CanvasText::Create()
    {
        return CANDERA_NEW(CanvasText)();
    }

    FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355,"Only reference is set, but not used inside constructor.")
    CanvasText::CanvasText() :
        Base(),
        m_canvasTransformable(*this, *this),
        m_activeProperties(),
        m_activeResult(),
        m_textColor(1.0F, 1.0F, 1.0F),
        m_isInvalid(TextProcessInvalidation::InvalidAll),
        m_setAlignment(HorizontalTextAlignment::Auto),
        m_parentalCanvas(0),
        m_previousEffectiveAlphaValue(1.0F),
        m_previousIsRenderingEnabled(true)
    {
#ifdef CANDERA_LAYOUT_ENABLED
        SetLayouter(&CanvasTextLayouter::GetDefault());
#endif
        m_activeProperties.SetTextProcessUser(this);
    }
    FEATSTD_SUPPRESS_MSC_WARNING_END()

    FEATSTD_SUPPRESS_MSC_WARNING_BEGIN(4355,"Only reference is set, but not used inside constructor.")
    CanvasText::CanvasText(CanvasText const& elem) :
        Base(elem),
        m_canvasTransformable(*this, *this),
        m_activeProperties(elem.m_activeProperties),
        m_activeResult(),
        m_textColor(elem.m_textColor),
        m_isInvalid(TextProcessInvalidation::InvalidAll),
        m_setAlignment(elem.m_setAlignment),
        m_parentalCanvas(0),
        m_previousEffectiveAlphaValue(elem.m_previousEffectiveAlphaValue),
        m_previousIsRenderingEnabled(elem.m_previousIsRenderingEnabled)
    {
        m_activeProperties.SetTextProcessUser(this);
    }
    FEATSTD_SUPPRESS_MSC_WARNING_END()

    CanvasText::~CanvasText()
    {
        m_activeProperties.SetTextProcessUser(0);
        m_parentalCanvas = 0;
    }

    const Candera::TextRendering::Internal::TextProcessResult::SharedPointer CanvasText::GetActiveProcessResult() const
    {
        CanvasText * canvasText = const_cast<CanvasText*>(this);
        FEATSTD_DEBUG_ASSERT(canvasText != 0);
        canvasText->UpdateText();
        return m_activeResult;
    }

    Candera::Globalization::Culture::SharedPointer CanvasText::GetCulture() const
    {
        return m_activeProperties.GetCulture();
    }


    void CanvasText::SetCulture(Candera::Globalization::Culture::SharedPointer val)
    {
        m_activeProperties.SetCulture(val);
    }


    Candera::TextRendering::TextSize CanvasText::GetLayoutRestriction() const
    {
        return m_activeProperties.GetOuterSizeRestriction();
    }


    void CanvasText::SetLayoutRestriction(Candera::TextRendering::TextSize val)
    {

        m_activeProperties.SetOuterSizeRestriction(val);
    }

    bool CanvasText::IsRightToLeftLayoutDirection() const
    {
        return m_activeProperties.IsRightToLeftLayoutDirection();
    }


    void CanvasText::SetRightToLeftLayoutDirection(bool val)
    {
        m_activeProperties.SetRightToLeftLayoutDirection(val);
    }


    bool CanvasText::IsPixelwiseLineSpacing() const
    {
        return m_activeProperties.IsPixelwiseLineSpacing();
    }


    void CanvasText::SetPixelwiseLineSpacing(bool val)
    {
        m_activeProperties.SetPixelwiseLineSpacing(val);
    }


    bool CanvasText::IsClipEnabled() const
    {
        return m_activeProperties.IsClipEnabled();
    }


    void CanvasText::SetClipEnabled(bool val)
    {
        m_activeProperties.SetClipEnabled(val);
    }


    void CanvasText::InvalidateProperties(TextProcessInvalidation::Enum invalidationType)
    {
        m_isInvalid = invalidationType;
#ifdef CANDERA_LAYOUT_ENABLED
        Scene * scene = GetScene();
        if (scene != 0) {
            scene->SetLayoutInvalid();
        }
#endif
    }



    void CanvasText::UpdateText()
    {
        m_activeProperties.CheckPassivePropertiesInvalidation();
        if ((m_activeResult.PointsToNull()) || (m_isInvalid != TextProcessInvalidation::ValidAll)) {
            if (!m_activeResult.PointsToNull()) {
                m_activeResult.Release();
            }
            m_activeResult = TextProcessResult::Create();

            if (TextProcessHelper::ProcessText(m_activeProperties, *m_activeResult)) {
                m_isInvalid = TextProcessInvalidation::ValidAll;
            }
            else {
                FEATSTD_LOG_WARN("Error processing text!");
            }
        }
    }

    void CanvasText::Render() {}


    void CanvasText::DisposeSelf()
    {
        CANDERA_DELETE(this);
    }

    Candera::TextRendering::Internal::TextProcessProperties::TextCache * CanvasText::GetGlyphCache()
    {
        return &GlyphAtlas3DGlyphCacheAccess::GetInstance();
    }

    void CanvasText::OnAncestorAdded(Scene* scene)
    {
        Base::OnAncestorAdded(scene);
        m_canvasTransformable.InvalidateParentCanvas();
        Canvas * parentalCanvas = m_canvasTransformable.GetParentCanvas();
        if ((m_parentalCanvas != parentalCanvas)) {
            if (m_parentalCanvas != 0) {// existing parental canvas has changed despite being attached to another canvas. 
                // it can also happen that the new parental canvas does not exist anymore. Even if ancestor removed should be called first,
                // this will handle this 'not possible' case, too. 
                FEATSTD_LOG_WARN("The parental canvas has changed without detaching the old one!");
                // This code should never be reached. But in case that it is reached, it has to unregister from the old canvas first.
                m_parentalCanvas->UnregisterCanvasText(this); // friend function
            }
            m_parentalCanvas = parentalCanvas;
            if (m_parentalCanvas != 0) {// if there is a new parental canvas, register on it
                if (!m_activeResult.PointsToNull()) {
                    m_activeResult->SetResultConsumed(false);
                }
                m_parentalCanvas->RegisterCanvasText(this); // friend function
            }
        }
    }


    void CanvasText::OnAncestorRemoved(Scene * scene)
    {
        Base::OnAncestorRemoved(scene);
        m_canvasTransformable.InvalidateParentCanvas();
        Canvas * parentalCanvas = m_canvasTransformable.GetParentCanvas(); 
        if (parentalCanvas == 0) { // only unregister the canvas if it is really removed from the canvas
            if (m_parentalCanvas != 0) { // a canvas were previously attached, unregister is required
                m_parentalCanvas->UnregisterCanvasText(this); // friend function
                if (!m_activeResult.PointsToNull()) {
                    // the canvas does not hold the transformed result anymore so it has to be newly transformed by the next attached canvas
                    m_activeResult->SetResultConsumed(false);
                }
                m_parentalCanvas = 0;
            }
        }
        else {
            // this is the normal case in which parental canvas keeps being the same
            // if this is not the case, one of the rules for ancestor add and remove has been broken.
            // This comes due to the fact that the nearest canvas ancestor becomes parent.
            // Besides this: Only one canvas can exist in the direct ancestors.
            FEATSTD_DEBUG_ASSERT(parentalCanvas == m_parentalCanvas);
        }
    }
    FEATSTD_RTTI_DEFINITION(CanvasText, Node)

    bool CanvasText::Upload()
    {
        bool success = true;

        // Upload appearance if not null.
        const MemoryManagement::SharedPointer<Appearance>& app = GetAppearance();
        if (!app.PointsToNull()) {
            if (!app->Upload()) {
                FEATSTD_LOG_ERROR("Upload Appearance failed for CanvasText:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        if (m_activeResult != 0) {
            m_activeResult->SetResultConsumed(false);
        }

        return success;
    }

    bool CanvasText::Unload()
    {
        bool success = true;

        // Unload appearance if not null.
        const MemoryManagement::SharedPointer<Appearance>& app = GetAppearance();
        if (!app.PointsToNull()) {
            if (!app->Unload()) {
                FEATSTD_LOG_ERROR("Unload Appearance failed for CanvasText:\"%s\".", (GetName() == 0) ? "" : GetName());
                success = false;
            }
        }

        if (m_activeResult != 0) {
            m_activeResult->SetResultConsumed(false);
        }

        return success;
    }

    void CanvasText::OnCompositeTransformChanged()
    {
        Base::OnCompositeTransformChanged();
        Canvas* parentalCanvas = m_canvasTransformable.GetParentCanvas();
        if (0 != parentalCanvas) {
            parentalCanvas->SetChanged();
        }
    }

    bool CanvasText::ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const
    {
        Rectangle boundingRectangle;
        GetComputedBoundingRectangle(boundingRectangle);
        if ((0.0F == boundingRectangle.GetWidth()) && (0.0F == boundingRectangle.GetHeight())) {
            return false;
        }

        minBounds = Vector3(boundingRectangle.GetLeft(), boundingRectangle.GetTop(), 0.0F);
        maxBounds = Vector3(boundingRectangle.GetLeft() + boundingRectangle.GetWidth(),
            boundingRectangle.GetTop() + boundingRectangle.GetHeight(), 0.0F);
        return true;
    }

    bool CanvasText::IsLineIntersectingGeometry(const Line& line, Float& distance) const
    {
        if (!IsIntersectionTestEnabled()) {
            return false;
        }

        if (!IsBoundingBoxValid()) {
            FEATSTD_LOG_ERROR("Is line intersecting geometry failed, invalid bounding box.");
            return false;
        }

        Vector3 minBounds;
        Vector3 maxBounds;
        GetBoundingBox(minBounds, maxBounds);
        FEATSTD_DEBUG_ASSERT((maxBounds.GetZ() - minBounds.GetZ()) == 0); // Assert that text is not a 3D model.

        Float min_distance = Math::MaxFloat();
        const Matrix4& worldTransform = GetWorldTransform();

        Vector3 triangleVertex1 = minBounds;
        Vector3 triangleVertex2(maxBounds.GetX(), minBounds.GetY(), minBounds.GetZ());
        Vector3 triangleVertex3 = maxBounds;

        triangleVertex1.TransformCoordinate(worldTransform);
        triangleVertex2.TransformCoordinate(worldTransform);
        triangleVertex3.TransformCoordinate(worldTransform);

        Vector3 hitPosition;
        // check intersection
        if (Math3D::TriangleLineIntersection(triangleVertex1,
            triangleVertex2,
            triangleVertex3,
            line,
            hitPosition)) {
            // store smallest distance
            Float temp_distance = line.GetStart().GetDistanceTo(hitPosition);
            if (temp_distance < min_distance) {
                min_distance = temp_distance;
                distance = min_distance;
            }

            return true;
        }

        Vector3 triangleVertex4(minBounds.GetX(), maxBounds.GetY(), minBounds.GetZ());
        triangleVertex4.TransformCoordinate(worldTransform);

        // check intersection
        if (Math3D::TriangleLineIntersection(triangleVertex1,
            triangleVertex3,
            triangleVertex4,
            line,
            hitPosition)) {
            // store smallest distance
            Float temp_distance = line.GetStart().GetDistanceTo(hitPosition);
            if (temp_distance < min_distance) {
                min_distance = temp_distance;
                distance = min_distance;
            }

            return true;
        }

        return false;
    }

    bool CanvasText::IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const
    {
        return Math3D::IsPickIntersectingGeometry(*this, camera, x, y, distance);
    }
#ifdef CANDERA_LAYOUT_ENABLED
    const Candera::Vector3 CanvasText::GetLayoutStartPosition() const
    {
        return m_canvasTransformable.GetLayoutStartPosition();
    }
#endif

}
