//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANVAS_IMAGE_H)
#define CANDERA_CANVAS_IMAGE_H

#include <Candera/Engine3D/Canvas/CanvasRenderable.h>
#include <Candera/Engine3D/Core/Renderable.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/System/Mathematics/Rectangle.h>

namespace Candera {

    /**
     * A CanvasSprite is a renderable canvas object. It consists of a normalized quad with start at 0,0 and dimensions of 1,1.
     * This quad is modified using the renderables vertex rectangle. The quad can either be configured to a fixed size
     * or it can be determined by a designated texture attachment.
     * As the vertex buffer itself is fixed, all CanvasSprites with the same or equivalent appearances are instanceable.
     * The CanvasSprite corresponds to the 2D Render Node. Various 2D effects 
     * and various custom effects can be implemented using the shader and rendermode. 
     */
    class CanvasSprite : public CanvasRenderable{


    public:
        FEATSTD_TYPEDEF_BASE(CanvasRenderable);

        /**
         * Creates a CanvasSprite. The first create initializes the instanceable vertex buffer.
         @ return A pointer to the new CanvasSprite instance.
         */
        static CanvasSprite* Create();

        /**
         * Destructor.
         */
        virtual ~CanvasSprite() override;


        /**
        *  Clones this CanvasSprite only.
        *  Attached Node resources  are not deep-copied but referenced. 
        *  @return  The pointer to the cloned Node if successful, otherwise NULL.
        */
        virtual CanvasSprite* Clone() const override;

        /**
         * Retrieves the vertex buffer of this CanvasSprite. The vertex buffer is only created once and then shared across
         * all CanvasSprite objects.
         * @return The vertex offer of this CanvasSprite.
         */
        VertexBuffer::SharedPointer GetVertexBuffer() const { return m_vertexBuffer; }

        /**
         * Retrieves the actual render size of this CanvasSprite. It is calculated with respect to the CanvasRenderables vertex rectangle and
         * considering the selected texture if one rectangle size component is smaller than 0.0F. The texture is evaluated for each 
         * appearance pass.
         * The returned dimension is the parameter that will be passed as "CanvasSize" to the shader.
         * @param appearance The appearance to consider textures from. Enables multi pass CanvasSprites.
         * @return The actual render size of this CanvasSprite.
         */
        virtual Vector2 GetActualDimension(const Appearance& appearance) const override;

        /**
        *  Tests if geometry of this CanvasSprite intersects with the given Line. Note: If a multi pass appearance is used, only the first appearance 
        *  is considered.
        *  @param line               The intersection Line.
        *  @param distance [out]     The distance from the nearest geometry to the starting point of the Line.
        *                           If no intersection is found then this param stays unchanged.
        *  @return                   True if an intersection is found with this canvas render node, otherwise false.
        */
        virtual bool IsLineIntersectingGeometry(const Line& line, Float& distance) const override;

        /**
        *  Upload asset data attached to this Node to video memory.
        *  @return False if Upload of vertex buffer or appearance failed. True otherwise.
        */
        virtual bool Upload() override;


        /**
        *  Unload asset data attached to this Node from video memory.
        *  @return False if Unload of vertex buffer or appearance failed. True otherwise.
        */
        virtual bool Unload() override;


        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center of a bitmap,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector2& basePoint) const override;

        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector3& basePoint) const;

        /**
        *  Gets the canvas axis-aligned bounding rectangle in local coordinate space which is computed by its
        *  actual content (e.g. boundaries of the image, text...).
        *  @param   boundingRectangle  Out parameter for the axis-aligned bounding rectangle in local coordinate space.
        */
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle) const override;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Set the nine patch bitmap. Setting a bitmap causes the CanvasSprite to act as a nine patch.
         *  Setting 0 causes the CanvasSprite to act as a sprite.
         *  @param bitmap  The bitmap, or 0, to determine the CanvasSprite's mode of operation.
         */
        void SetNinePatchBitmap(const Bitmap::SharedPointer& bitmap);

        /**
         *  Return the nine patch bitmap.
         *  @return  The nine patch bitmap.
         */
        const Bitmap::SharedPointer& GetNinePatchBitmap() const { return m_ninePatchBitmap; }

        /**
         * Extra functionality from base impl: Check if the bounding rectangle has updated, due to 
         * change in size or chage of texture size (for negative values)
         * @return true if bounding box is valid, false otherwise.
         */
        virtual bool IsBoundingBoxValid() const override;

    protected:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(CanvasSprite);
        CanvasSprite();

        /**
        *  Renders this CanvasSprite. This method is overridden from Node.
        */
        virtual void Render() override;


        /**
        *  Disposes the instance of this class.
        */
        virtual void DisposeSelf() override;


        /**
        *  Computes axis-aligned bounding box (AABB) in object coordinate space. The CanvasRenderables vertex rectangle
        *  is considered in the computation, as well as the result from CanvasSprite::GetActualDimension.
        *  @param minBounds is an out parameter describing the returned lower-left vertex of the bounding box.
        *  @param maxBounds is an out parameter describing the returned upper-right vertex of the bounding box.
        *  @return true if bounding box was computed successfully and false if not because of missing geometry.
        */
        virtual bool ComputeBoundingBoxImpl(Vector3& minBounds, Vector3& maxBounds) const override;

        /**
        *  Tests if geometry of this CanvasSprite intersects with a pick in absolute Viewport coordinate in pixels.
        *  @remark                   This interface is usually used for object selection in scene editors.
        *  @param camera             The view camera. (i.e. the current camera)
        *  @param x                  The x value as an absolute Viewport coordinate in pixels starting from top-left.
        *  @param y                  The y value as an absolute Viewport coordinate in pixels starting from top-left.
        *  @param distance [out]     The distance from the nearest geometry to the near plane of the camera.
        *                            If no intersection is found then this param stays unchanged.
        *  @return                   True if an intersection is found with this CanvasSprite, otherwise false.
        */
        virtual bool IsPickIntersectingGeometryInternal(const Camera& camera, Int x, Int y, Float& distance /*out*/) const override;

    private:
        void PrepareNinePatchVertexGeometry(const Float width, const Float height,
            const Bitmap::NinePatchProperties& ninePatchProperties, const Vector2& toScale, 
            const bool scaleX, const bool scaleY);

        //Initializes the vertex buffer.
        bool Initialize();

        /* 
         * Creates the internal static vertex buffer.
         * It is layouted as follows and builds a triangle strip.
         * C---D
         * | \ |
         * |  \|
         * A---B
         * The following vertex elements are available:
         * x y z u v
         *
         * Normals, binormals and tangents are not supported. 
         * Due to the simplicity of this vertex buffer (it is a single plane) these attributes can be realized in
         * adding a single vector3 uniform for each attribute. 
         */
        static VertexBuffer::SharedPointer CreateVertexBuffer();

        /**
         * Get the vertex buffer used for the sprite (not nine patch)
         */
        static VertexBuffer::SharedPointer GetSpriteVertexBuffer();

        /**
         * Overrides Renderable::IsRenderPrerequisiteFulfilled.
         * Returns true if precondition to be rendered is fulfilled, false otherwise.
         */
        virtual bool IsRenderPrerequisiteFulfilled() const override;

        /*
         * Internal helper method to retrieve the current vertex scale.
         */
        Vector3 GetVertexScale() const;

        mutable Vector3 m_vertexScaleCache; // cached value of vertex scale (updated for each compute of bounding box)

        Bitmap::SharedPointer m_ninePatchBitmap;

        /**
         *  Structure to cache the nine patch properties that define the contents of its vertex buffer.
         */
        struct NinePatchVertexBufferProperties
        {
            NinePatchVertexBufferProperties() :
                m_width(0.0F),
                m_height(0.0F),
                m_ninePatchProperties(),
                m_toScale(),
                m_scaleX(false),
                m_scaleY(false),
                m_isVertexBufferValid(false) {}

            inline bool IsVertexBufferValid() const { return m_isVertexBufferValid; }

            bool IsEqual(Float width, Float height, const Bitmap::NinePatchProperties& ninePatchProperties,
                const Vector2& toScale, bool scaleX, bool scaleY) const {
                FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
                return (m_isVertexBufferValid && (width == m_width) && (height == m_height) &&
                    (ninePatchProperties == m_ninePatchProperties) && (toScale.IsEqual(m_toScale)) &&
                    (scaleX == m_scaleX) && (scaleY == m_scaleY));
            }

            void Set(Float width, Float height, const Bitmap::NinePatchProperties& ninePatchProperties,
                const Vector2& toScale, bool scaleX, bool scaleY) {
                m_width = width;
                m_height = height;
                m_ninePatchProperties = ninePatchProperties;
                m_toScale = toScale;
                m_scaleX = scaleX;
                m_scaleY = scaleY;
                m_isVertexBufferValid = true;
            }

        private:
            Float m_width;
            Float m_height;
            Bitmap::NinePatchProperties m_ninePatchProperties;
            Vector2 m_toScale;
            bool m_scaleX;
            bool m_scaleY;

            bool m_isVertexBufferValid;
        } m_ninePatchVertexBufferProperties[CANDERA_MAX_CONTEXT_COUNT];

        CdaDynamicProperties(Candera::CanvasSprite, Candera::Node);
        CdaDynamicPropertiesEnd();
    };

}


#endif  // CANDERA_CANVAS_IMAGE_H
