//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANVAS_GROUP_H)
#define CANDERA_CANVAS_GROUP_H

#include <Candera/Engine3D/Canvas/CanvasTransformable.h>
#include <Candera/Engine3D/Core/Group.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>

namespace Candera {

    /**
     * A CanvasGroup is a Group with additional means to perform 2D transformations. 
     */
    class CanvasGroup : public Group, public CanvasInterface{

    public:
        FEATSTD_TYPEDEF_BASE(Group);

        /**
         * Creates the CanvasGroup.
         */
        static CanvasGroup* Create();

        /**
         * Destructor.
         */
        virtual ~CanvasGroup() override {}

        /**
        *  Clones this CanvasGroup only.
        *  Attached Node resources  are not deep-copied but referenced.
        *  @return  The pointer to the cloned Node if successful, otherwise NULL.
        */
        virtual CanvasGroup* Clone() const override;

        /**
        * Concatenates the given movement (translation) with current position component.
        * In addition to Translate, this also adjusts the pivot offset such that the object
        * preserves its appearance (global transformation matrix remains the same.)
        * @param translation The translation that shall be applied.
        */
        void TranslatePivot2D(const Vector2& translation);
        
        //Canvas Interface.

        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center of a bitmap,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector2& basePoint) const override;

#ifdef CANDERA_LAYOUT_ENABLED
        /**
        *  Gets the base point in local coordinate space. The base point is typically in the center,
        *  or on the base line of a text.
        *  @param  basePoint   Out parameter for the base point in local coordinate space.
        */
        virtual void GetBasePoint(Vector3& basePoint) const override;
#endif
        /**
        *  Gets the canvas axis-aligned bounding rectangle in local coordinate space which is computed by its
        *  actual content (e.g. boundaries of the image, text...).
        *  @param   boundingRectangle  Out parameter for the axis-aligned bounding rectangle in local coordinate space.
        */
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle) const override;

        /**
        *  Gets the CanvasTransformable of this Canvas node.
        *  @return  This nodes CanvasTransformable.
        */
        virtual CanvasTransformable& GetCanvasTransformable() override;


        /**
         * Override from Node.
         * Invalidates parent canvas.
         */
        virtual void OnAncestorAdded(Scene* scene) override { FEATSTD_UNUSED(scene); m_canvasTransformable.InvalidateParentCanvas(); }

        /**
         * Override from Node.
         * Invalidates parent canvas.
         */
        virtual void OnAncestorRemoved(Scene* scene) override { FEATSTD_UNUSED(scene); m_canvasTransformable.InvalidateParentCanvas(); }

        FEATSTD_RTTI_DECLARATION();

    protected:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(CanvasGroup);
        /**
         * Constructor.
         */
        CanvasGroup();

        /**
        *  Disposes the instance of this class.
        */
        virtual void DisposeSelf() override;

        /**
        *  Called when one of the transforms (translation, rotation, scale, general transform) has changed.
        *  Useful for updating dependent properties. Implementing classes will probably want to propagate the call up the hierarchy.
        */
        virtual void OnCompositeTransformChanged() override;

#ifdef CANDERA_LAYOUT_ENABLED
        /**
        *  Returns the "offset" coordinate which is required to position 3D objects correctly when using layout.
        *  Normally, 3D objects are centered but the layouter thinks its position is the "upper-left" corner.
        *  This will be replaced as soon as there is a more appropriate handling (e.g. own pivot point)
        *  @return  layout start position
        */
        virtual const Vector3 GetLayoutStartPosition() const override;

#endif
        CanvasTransformable m_canvasTransformable;

    private:
        CdaDynamicProperties(Candera::CanvasGroup, Candera::Group);
        CdaDynamicPropertiesEnd();
    };


    namespace Internal {
        class BoundingRectangleTraverser : public ConstTreeTraverser {
        public:

            BoundingRectangleTraverser(const Node& root);

            void GetBoundingRectangle(Rectangle& rect) const;

        protected:
            virtual TraverserAction ProcessNode(const Node& node) override;

        private:
            Rectangle m_boundingRectangle;
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
            const Node& m_root;
            Matrix4 m_inverseRootWorldTransform;

            //Forbid assignment, make it private
            BoundingRectangleTraverser& operator=(const BoundingRectangleTraverser&);
        };
    }

}




#endif  // CANDERA_CANVAS_GROUP_H
