//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANVAS_H)
#define CANDERA_CANVAS_H

#include <Candera/Engine3D/Canvas/CanvasGroup.h>
#include <Candera/Engine3D/Core/GlyphAtlasTextContainer.h>
#include <Candera/System/Container/Vector.h>


#if defined(CANDERA_LAYOUT_ENABLED)
/*
 * Layout root functionality has to be added to this class. (SetLayoutInvalid, ValidateLayout).
 * Nested canvases are not allowed. Layout stops at leafs or 3D children of Canvas.
 * List of canvases is to be kept in the scene. Scene calls ValidateLayout for all Canvases. Canvases in List need to be sorted from child to parent.
 * ValidateLayout is called on Scene, which calls it on all invalid canvases.
 * CanvasNodes need to have GetParentCanvas function. Canvases should behave as Layout roots like Scene2D.
 */
#endif

namespace Candera {

    /**
     * A Canvas represents the root node for functionality that is computed over whole canvas sub trees.
     * After support for Layout has been added, this will be the entry point for it.
     */
    class Canvas : public CanvasGroup
    {
        FEATSTD_TYPEDEF_BASE(CanvasGroup);

    public:
        /**
        * Creates an instance of this class.
        * Use Dispose() to delete the instance and possible children, if any.
        * @return Pointer to the created object
        */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::Canvas::Create, CANDERA_LINT_REASON_EXPLICITHIDING)
        static Canvas* Create(bool useUpdateSystem = false);

        /**
        * Destructor.
        */
        virtual ~Canvas() override;

        /**
        * Creates a clone of the Canvas.
        * @return A clone of the Canvas.
        */
        virtual Canvas* Clone() const override;

        /**
        * Indicate that the Canvas or elements contained in the Canvas have changed.
        */
        void SetChanged() { m_hasCanvasChanged = true; }

        /**
        * Override from CanvasGroup.
        * Notifies scene about adding the Canvas to it.
        * @param scene of the scene tree in which the Canvas is added.
        */
        virtual void OnAncestorAdded(Scene* scene) override;

        /**
        * Override from CanvasGroup.
        * Notifies scene about removing the Canvas from it.
        * @param scene of the scene tree in which the Canvas is removed from.
        */
        virtual void OnAncestorRemoved(Scene * scene) override;

        /**
        * Updates all Vertex-Buffer if necessary.
        * Traverses through the Canvas child nodes.
        */
        void PrepareRenderableText();

        /**
        * Query if this canvas uses the UpdateSystem.
        * @return  True, if the Canvas uses the UpdateSystem. False, otherwise.
        */
        bool UsesUpdateSystem() const { return m_useUpdateSystem; }

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Canvas(bool useUpdateSystem);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Canvas);


        /**
        * Disposes the instance of this class.
        */
        virtual void DisposeSelf() override;

    private:
        void RegisterCanvasText(CanvasText* canvasText);
        void UnregisterCanvasText(CanvasText const* canvasText);

        friend void Candera::CanvasText::OnAncestorAdded(Scene* scene);
        friend void Candera::CanvasText::OnAncestorRemoved(Scene* scene);
        Candera::Internal::Vector<Internal::GlyphAtlasTextContainer> m_canvasTexts;

        bool AttachDispatcher();

        struct BatchElement {
            Internal::GlyphAtlasTextContainer* textContainer;
            SizeType startIndex;
            SizeType count;
        };

        typedef Candera::Internal::Vector<BatchElement> Batch;
        typedef Candera::Internal::Vector<Batch> Batches;
        void BatchCanvasText();

        void CreateBatches(Batches& batches);
        bool IsEqual(const BatchElement& entryA, const BatchElement& entryB) const;
        VertexBuffer::SharedPointer CreateVertexBuffer(SizeType maxGlyphCount) const;
        void PerformBatching(const Batch& batch, SizeType batchElementIndex, SizeType batchElementSize,
            SizeType glyphCount, const GlyphAtlas3D& glyphAtlas);
        Mesh* SetupCanvasTextBatch(const Batch& batch, SizeType glyphCount, const GlyphAtlas3D& glyphAtlas);
        void CreateAndAssignProperties(const Batch& batch, const GlyphAtlas3D& glyphAtlas, Mesh* mesh) const;
        void CopyProperties(const Batch& sourceBatch, Mesh* destinationMesh, const GlyphAtlas3D& glyphAtlas) const;

        Group* m_canvasTextBatchesGroup;
        Matrix4 m_previousTransformation;
        bool m_hasCanvasChanged;
        bool m_useUpdateSystem;

        Candera::Internal::Vector<Mesh*> m_meshesInUse;
        Candera::Internal::Vector<Mesh*> m_meshesUnused;
    };
}

#endif  // CANDERA_CANVAS_H
