//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Canvas.h"

#include <Candera/Engine3D/Canvas/CanvasText.h>
#include <Candera/Engine3D/Canvas/CanvasTextBatch.h>
#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/TextMesh.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/EntityComponentSystem/EntityComponentSystem.h>
#include <Candera/System/UpdateSystem/UpdateSystem.h>

using namespace Candera::Internal;

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine3D);

    Canvas::Canvas(bool useUpdateSystem) :
        Base(),
        m_canvasTextBatchesGroup(Group::Create()),
        m_hasCanvasChanged(true),
        m_useUpdateSystem(useUpdateSystem)
    {
        this->Scale(Vector3(1.0F, -1.0F, -1.0F));
        m_canvasTextBatchesGroup->SetName("CanvasTextBatchedGroup");
        static_cast<void>(AttachDispatcher());
    }

    Canvas::~Canvas()
    {
        Node* const parentNode = m_canvasTextBatchesGroup->GetParent();
        if (parentNode != 0) {
            static_cast<void>(parentNode->RemoveChild(m_canvasTextBatchesGroup));
        }

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::Canvas::m_canvasTextBatchesGroup, freeing the text batches group is done in dispose)
        m_canvasTextBatchesGroup->Dispose();

        for (SizeType i = 0; i < m_meshesUnused.Size(); ++i) {
            m_meshesUnused[i]->Dispose();
        }
    }

    Canvas* Canvas::Create(bool useUpdateSystem)
    {
        return FEATSTD_NEW(Canvas)(useUpdateSystem);
    }

    Canvas::Canvas(const Candera::Canvas &other) :
        Base(other),
        m_canvasTextBatchesGroup(Group::Create()),
        m_hasCanvasChanged(true),
        m_useUpdateSystem(other.m_useUpdateSystem)
    {
        static_cast<void>(AttachDispatcher());
    }

    Canvas* Canvas::Clone() const
    {
        return FEATSTD_NEW(Canvas)(*this);
    }

    FEATSTD_RTTI_DEFINITION(Canvas, CanvasGroup)

    void Canvas::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }


    void Canvas::OnAncestorAdded(Scene* scene)
    {
        Base::OnAncestorAdded(scene);

        if (0 != scene) {
            scene->OnCanvasAdded(this);
        }
    }

    void Canvas::OnAncestorRemoved(Scene * scene)
    {
        Base::OnAncestorRemoved(scene);

        if (0 != scene) {
            scene->OnCanvasRemoved(this);
        }
    }

    struct SortCompare
    {
        bool operator()(const Mesh* a, const Mesh* b) const
        {
            UInt32 aVertexCount = a->GetVertexBuffer()->GetVertexGeometry()->GetVertexCount();
            UInt32 bVertexCount = b->GetVertexBuffer()->GetVertexGeometry()->GetVertexCount();
            return aVertexCount < bVertexCount;
        }
    };

    void Canvas::PrepareRenderableText()
    {
        bool hasTextChanged = false;

        for (SizeType i = 0; i < m_canvasTexts.Size(); ++i) {
            Internal::GlyphAtlasTextContainer& textContainer = m_canvasTexts[i];
            CanvasText* textNode = textContainer.GetTextNode();
            textContainer.UpdateContainer(textNode);

            if (!textContainer.IsContainerChangeConsumed()) {
                // Text, font, style, color, etc. has changed
                hasTextChanged = true;
            }

            FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
            if (textNode->GetEffectiveAlphaValue() != textNode->m_previousEffectiveAlphaValue) {
                hasTextChanged = true;
                textNode->m_previousEffectiveAlphaValue = textNode->GetEffectiveAlphaValue();
            }

            if (textNode->IsEffectiveRenderingEnabled() != textNode->m_previousIsRenderingEnabled) {
                hasTextChanged = true;
                textNode->m_previousIsRenderingEnabled = textNode->IsEffectiveRenderingEnabled();
            }

            textContainer.SetContainerChangeConsumed(true);
        }

        // Any CanvasText's local transformation change has already been pushed to
        // this Canvas's m_hasAnyCanvasTextChanged flag. Check if the world transformation of
        // the Canvas itself has changed to determine if (re-)creating Batches is necessary.
        if (!(hasTextChanged || m_hasCanvasChanged)) {
            if (GetWorldTransform() != m_previousTransformation) {
                m_hasCanvasChanged = true;
            }
        }

        if (hasTextChanged || m_hasCanvasChanged) {
            for (SizeType i = 0; i < m_meshesInUse.Size(); ++i) {
                if (!m_meshesUnused.Add(m_meshesInUse[i])) {
                    return;
                }
            }

            m_meshesInUse.Clear();

            SortCompare sortCompare;
            m_meshesUnused.Sort(sortCompare);

            m_canvasTextBatchesGroup->RemoveAllChildren();

            Scene* scene = GetScene();
            if (scene != m_canvasTextBatchesGroup->GetParent()) {
                if (!scene->AddChild(m_canvasTextBatchesGroup)) {
                    return;
                }
            }

            BatchCanvasText();
            m_previousTransformation = GetWorldTransform();
        }

        m_hasCanvasChanged = false;
    }

    void Canvas::RegisterCanvasText(CanvasText* canvasText)
    {
        if (canvasText != 0) {
            SizeType i = 0;
            while (i < m_canvasTexts.Size()) {
                if (m_canvasTexts[i].GetTextNode() == canvasText) {
                    FEATSTD_LOG_INFO("Tried to register a CanvasText in a Canvas twice!");
                    return;
                }
                ++i;
            }
            if (m_canvasTexts.Add(GlyphAtlasTextContainer())) {
                FEATSTD_DEBUG_ASSERT(m_canvasTexts.Size() > 0);
                FEATSTD_DEBUG_ASSERT(m_canvasTexts[m_canvasTexts.Size() - 1].GetTextNode() == 0);
                m_canvasTexts[m_canvasTexts.Size() - 1].SetTextNode(canvasText);
            }
            else {
                // This should not occur otherwise out of memory.
                FEATSTD_LOG_ERROR("Failed to register canvas text in Canvas update!");
            }
        }
    }

    void Canvas::UnregisterCanvasText(CanvasText const* canvasText)
    {
        if (canvasText != 0) {
            for (SizeType i = 0; i < m_canvasTexts.Size(); ++i) {
                if (m_canvasTexts[i].GetTextNode() == canvasText) {
                    static_cast<void>(m_canvasTexts.Remove(i));
                    break;
                }
            }
        }
    }

    bool Canvas::AttachDispatcher()
    {
        if (!m_useUpdateSystem) {
            return false;
        }

        UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
        if (updateSystem == 0) {
            FEATSTD_LOG_ERROR("UpdateSystem is not created yet! Binding Canvas update routine to UpdateSystem not possible!");
            return false;
        }

        UpdateSystem::Handle updateHandle = updateSystem->CreateComponent();
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::Update, "False positive because the template arguments match.")
        bool result = updateSystem->SetComponentLateUpdateDelegate(updateHandle, UpdateSystem::Delegate::Update<Canvas, &Canvas::PrepareRenderableText>());
        result = (result) && (updateSystem->AttachComponent(updateHandle, this));
        if (!result) {
            static_cast<void>(updateSystem->DestroyComponent(updateHandle));
        }

        return result;
    }

    bool Canvas::IsEqual(const BatchElement& entryA, const BatchElement& entryB) const
    {
        Internal::GlyphAtlasTextContainer* a = entryA.textContainer;
        Internal::GlyphAtlasTextContainer* b = entryB.textContainer;
        FEATSTD_DEBUG_ASSERT(a != 0);
        FEATSTD_DEBUG_ASSERT(b != 0);
        FEATSTD_DEBUG_ASSERT((a != b) || (entryA.startIndex != entryB.startIndex));
        FEATSTD_DEBUG_ASSERT(a->Size() > 0);
        FEATSTD_DEBUG_ASSERT(b->Size() > 0);
        FEATSTD_DEBUG_ASSERT(a->GetTextNode() != 0);
        FEATSTD_DEBUG_ASSERT(b->GetTextNode() != 0);
        FEATSTD_DEBUG_ASSERT((a->GetTextNode() != b->GetTextNode()) || (entryA.startIndex != entryB.startIndex));

        CanvasText* textA = a->GetTextNode();
        CanvasText* textB = b->GetTextNode();
        if (textA->GetAppearance() != textB->GetAppearance()) {
            return false;
        }

        if (textA->GetRenderOrderRank() != textB->GetRenderOrderRank()) {
            return false;
        }

        if (textA->GetRenderOrderBinAssignmentHash() != textB->GetRenderOrderBinAssignmentHash()) {
            return false;
        }

        if (textA->GetScopeMask() != textB->GetScopeMask()) {
            return false;
        }

        if (((*a)[entryA.startIndex]->BitmapImageIndex) != ((*b)[entryB.startIndex]->BitmapImageIndex)) {
            return false;
        }

        return true;
    }

    static const SizeType s_maxGlyphCountPerMesh = static_cast<SizeType>(FeatStd::Internal::Limits<UInt16>::Max() / 4);

    void Canvas::CreateBatches(Batches& batches)
    {
        // Divide CanvasTexts into batches by BitmapImageIndex usage. Also split CanvasTexts that exceed
        // the maximum glyph count per mesh.
        Batch canvasTexts;
        for (SizeType i = 0; i < m_canvasTexts.Size(); ++i) {
            Internal::GlyphAtlasTextContainer& textContainer = m_canvasTexts[i];

            if (!textContainer.GetTextNode()->IsEffectiveRenderingEnabled()) {
                continue; // Skip CanvasText when rendering is disabled.
            }

            const SizeType textContainerSize = textContainer.Size();
            if (textContainerSize == 0) {
                continue; // Skip CanvasText with no text.
            }

            FEATSTD_DEBUG_ASSERT(0 != textContainer.GetTextNode());
            if (textContainer.GetTextNode()->GetAppearance().PointsToNull()) {
                continue; // Skip CanvasText with no appearance.
            }

            SizeType currentStartBatchEntryIndex = canvasTexts.Size();
            BatchElement batchEntry;
            batchEntry.textContainer = &textContainer;
            batchEntry.startIndex = 0;
            batchEntry.count = textContainerSize;
            if (!canvasTexts.Add(batchEntry)) {
                return;
            }

            // Divide by BitmapImageIndex usage.
            SizeType previousBitmapImageIndex = textContainer[0]->BitmapImageIndex;
            for (SizeType j = 1; j < textContainerSize; ++j) {
                SizeType currentBitmapImageIndex = textContainer[j]->BitmapImageIndex;
                // Assert that the textContainer was sorted by BitmapImageIndex beforehand.
                FEATSTD_DEBUG_ASSERT(currentBitmapImageIndex >= previousBitmapImageIndex);
                if (textContainer[j]->BitmapImageIndex > previousBitmapImageIndex) {
                    BatchElement& previousEntry = canvasTexts[canvasTexts.Size() - 1];
                    previousEntry.count = (j - previousEntry.startIndex);

                    batchEntry.startIndex = j;
                    batchEntry.count = textContainerSize - j;
                    if (!canvasTexts.Add(batchEntry)) {
                        return;
                    }

                    previousBitmapImageIndex = currentBitmapImageIndex;
                }
            }

            // Divide resulting batches if max glyph count is exceeded
            const SizeType maxGlyphCountPerMesh = s_maxGlyphCountPerMesh;
            for (SizeType j = currentStartBatchEntryIndex; j < canvasTexts.Size(); ++j) {
                BatchElement& currentEntry = canvasTexts[j];
                if (currentEntry.count > maxGlyphCountPerMesh) {
                    batchEntry.startIndex = currentEntry.startIndex + maxGlyphCountPerMesh;
                    batchEntry.count = currentEntry.count - maxGlyphCountPerMesh;
                    currentEntry.count = maxGlyphCountPerMesh;
                    const SizeType newIndex = j + 1;
                    if (canvasTexts.Size() > newIndex) {
                        if (!canvasTexts.Insert(newIndex, batchEntry)) {
                            return;
                        }
                    }
                    else {
                        if (!canvasTexts.Add(batchEntry)) {
                            return;
                        }
                    }
                }
            }
        }

        // Put CanvasTexts with same atlas bitmap index, appearance, render order rank,
        // render order bin assignment, and scope mask in the same bin.
        FEATSTD_DEBUG_ASSERT(batches.Size() == 0);
        for (SizeType i = 0; i < canvasTexts.Size(); ++i) {
            const BatchElement& batchEntry = canvasTexts[i];
            bool isNewBatch = true;
            for (SizeType j = 0; j < batches.Size(); ++j) {
                Batch& batch = batches[j];
                FEATSTD_DEBUG_ASSERT(batch.Size() > 0);
                if (IsEqual(batch[0], batchEntry)) {
                    if (!batch.Add(batchEntry)) {
                        return;
                    }

                    isNewBatch = false;
                    break;
                }
            }

            if (isNewBatch) {
                Batch batch;
                if (!batch.Add(batchEntry)) {
                    return;
                }

                if (!batches.Add(batch)) {
                    return;
                }
            }
        }
    }

    void Canvas::BatchCanvasText()
    {
        Batches batches;
        CreateBatches(batches);

        // Get GlyphAtlas3D once, minimizing Mutex execution due to synced StaticObject usage.
        GlyphAtlas3D& glyphAtlas = GlyphAtlas3D::GetInstance();

        // Create batched meshes from batches.
        // Depending on the number of textures used by the GlyphAtlas, and the maximum size of a vertex buffer,
        // the number of batched meshes is equal or greater than the number of batches.
        for (SizeType i = 0; i < batches.Size(); ++i) {
            const Batch& batch = batches[i];

            SizeType currentGlyphCount = 0;
            SizeType lastBatch = 0;

            // Determine VertexBuffer size and batch text containers.
            for (SizeType j = 0; j < batch.Size(); ++j) {
                const BatchElement& element = batch[j];
                const SizeType elementGlyphCount = element.count;
                if ((currentGlyphCount + elementGlyphCount) <= s_maxGlyphCountPerMesh) {
                    currentGlyphCount += elementGlyphCount;
                    continue;
                }
                else {
                    PerformBatching(batch, lastBatch, j-lastBatch, currentGlyphCount, glyphAtlas);
                    currentGlyphCount = elementGlyphCount;
                    lastBatch = j;
                }
            }

            if (currentGlyphCount > 0) {
                PerformBatching(batch, lastBatch, batch.Size() - lastBatch, currentGlyphCount, glyphAtlas);
            }
        }
    }

    void Canvas::CreateAndAssignProperties(const Batch& batch, const GlyphAtlas3D& glyphAtlas, Mesh* mesh) const
    {
        FEATSTD_DEBUG_ASSERT(0 != mesh);
        Internal::GlyphAtlasTextContainer* textContainer = batch[0].textContainer;
        FEATSTD_DEBUG_ASSERT(0 != textContainer);
        CanvasText* canvasText = textContainer->GetTextNode();
        FEATSTD_DEBUG_ASSERT(0 != canvasText);
        const Appearance::SharedPointer& canvasTextAppearance = canvasText->GetAppearance();
        FEATSTD_DEBUG_ASSERT(canvasTextAppearance != 0);

        Texture::SharedPointer texture;
        Texture::SharedPointer userDummyTexture = canvasTextAppearance->GetTexture(0);
        if (userDummyTexture != 0) {
            texture = userDummyTexture->Clone();
        }
        else {
            texture = Texture::Create();
        }

        const GlyphAtlasTextContainer::GlyphInfo* glyphInfo = (*textContainer)[batch[0].startIndex];
        texture->SetTextureImage(glyphAtlas.GetBitmapImage(glyphInfo->BitmapImageIndex));
        Appearance::SharedPointer appearance = canvasTextAppearance->Clone();
        static_cast<void>(appearance->SetTexture(texture));
        static_cast<void>(appearance->Upload());

        // Returning the appearance and assigning it to another SharedPointer would trigger
        // another Mutex which we can avoid by passing the mesh and assigning it here.
        mesh->SetAppearance(appearance);
        mesh->SetName("CanvasTextBatched");

        mesh->SetScopeMask(canvasText->GetScopeMask());
        mesh->SetRenderOrderRank(canvasText->GetRenderOrderRank());
        mesh->SetRenderOrderBinAssignment(canvasText->GetRenderOrderBinAssignment());
    }

    void Canvas::CopyProperties(const Batch& sourceBatch, Mesh* destinationMesh, const GlyphAtlas3D& glyphAtlas) const
    {
        FEATSTD_DEBUG_ASSERT(0 != destinationMesh);
        FEATSTD_DEBUG_ASSERT(sourceBatch.Size() > 0);
        GlyphAtlasTextContainer* textContainer = sourceBatch[0].textContainer;
        FEATSTD_DEBUG_ASSERT(0 != textContainer);
        CanvasText* canvasText = textContainer->GetTextNode();
        FEATSTD_DEBUG_ASSERT(0 != canvasText);
        Appearance::SharedPointer canvasTextAppearance = canvasText->GetAppearance();
        FEATSTD_DEBUG_ASSERT(canvasTextAppearance != 0);

        Appearance::SharedPointer destinationAppearance = destinationMesh->GetAppearance();
        static_cast<void>(destinationAppearance->Unload());

        Texture::SharedPointer atlasTexture = destinationAppearance->GetTexture();
        FEATSTD_DEBUG_ASSERT(atlasTexture != 0);
        destinationAppearance->CopyProperties(canvasTextAppearance.GetSharedInstance());
        Texture::SharedPointer canvasTextTexture = canvasTextAppearance->GetTexture();
        if (canvasTextTexture != 0) {
            atlasTexture->CopyProperties(canvasTextTexture.GetSharedInstance());
        }
        else {
            // CanvasTexture's Appearance has no texture, so we use default values instead.
            atlasTexture->SetPropertiesToDefault();
        }

        static_cast<void>(destinationAppearance->SetTexture(atlasTexture));
        const GlyphAtlasTextContainer::GlyphInfo* glyphInfo = (*textContainer)[sourceBatch[0].startIndex];
        atlasTexture->SetTextureImage(glyphAtlas.GetBitmapImage(glyphInfo->BitmapImageIndex));
        static_cast<void>(destinationAppearance->Upload());

        destinationMesh->SetScopeMask(canvasText->GetScopeMask());
        destinationMesh->SetRenderOrderRank(canvasText->GetRenderOrderRank());
        destinationMesh->SetRenderOrderBinAssignment(canvasText->GetRenderOrderBinAssignment());
    }

    Mesh* Canvas::SetupCanvasTextBatch(const Batch& batch, SizeType glyphCount, const GlyphAtlas3D& glyphAtlas)
    {
        FEATSTD_DEBUG_ASSERT(glyphCount > 0);

        Mesh* mesh = 0;

        const SizeType batchVertexCount = glyphCount * 4;
        for (SizeType i = 0; i < m_meshesUnused.Size(); ++i) {
            mesh = m_meshesUnused[i];
            VertexBuffer::SharedPointer vertexBuffer = mesh->GetVertexBuffer();
            FEATSTD_DEBUG_ASSERT(vertexBuffer != 0);
            VertexGeometry* geometry = vertexBuffer->GetVertexGeometry();
            FEATSTD_DEBUG_ASSERT(0 != geometry);
            if (batchVertexCount <= geometry->GetVertexCount()) {
                CopyProperties(batch, mesh, glyphAtlas);
                static_cast<void>(m_meshesUnused.Remove(i));

                if (!m_meshesInUse.Add(mesh)) {
                    return 0;
                }

                FEATSTD_DEBUG_ASSERT(0 != m_canvasTextBatchesGroup);
                if (!m_canvasTextBatchesGroup->AddChild(mesh)) {
                    return 0;
                }

                return mesh;
            }
        }

        VertexBuffer::SharedPointer vertexBuffer = CreateVertexBuffer(glyphCount);
        if (vertexBuffer != 0) {
            mesh = CanvasTextBatch::Create();
            if (0 == mesh) {
                FEATSTD_LOG_ERROR("Canvas could not create CanvasTextBatch.");
                return 0;
            }

            Matrix4 identity;
            mesh->SetWorldTransform(identity);
            mesh->SetVertexBuffer(vertexBuffer);
            FEATSTD_DEBUG_ASSERT(0 != m_canvasTextBatchesGroup);
            if ((!m_canvasTextBatchesGroup->AddChild(mesh)) || (!m_meshesInUse.Add(mesh))) {
                mesh->Dispose();
                return 0;
            }

            CreateAndAssignProperties(batch, glyphAtlas, mesh);
        }

        return mesh;
    }

    struct VertexUvColor {
        // Position
        Float x;
        Float y;
        Float z;
        // UV
        Float u;
        Float v;
        // Color
        Float r;
        Float g;
        Float b;
        Float a;
    };

    static const VertexGeometry::VertexElementFormat VertexUvColorFormat[] = {
        { 0, VertexGeometry::Float32_3, VertexGeometry::Position, 0 },
        { 12, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 },
        { 20, VertexGeometry::Float32_4, VertexGeometry::Color, 0 }
    };

    void Canvas::PerformBatching(const Batch& batch, SizeType batchElementIndex, SizeType batchElementSize,
        SizeType glyphCount, const GlyphAtlas3D& glyphAtlas)
    {
        Mesh* canvasTextBatch = SetupCanvasTextBatch(batch, glyphCount, glyphAtlas);
        VertexBuffer::SharedPointer vertexBuffer = canvasTextBatch->GetVertexBuffer();

        const Float maxFloat = FeatStd::Internal::Limits<Float>::Max();
        const Float minFloat = -maxFloat;
        Vector3 minBounds(maxFloat, maxFloat, maxFloat);
        Vector3 maxBounds(minFloat, minFloat, minFloat);

        if (vertexBuffer != 0) {
            VertexGeometry::VertexArrayResource vertexResource(vertexBuffer->GetVertexGeometry()->GetVertexArrayResourceHandle());
            Float* vertices = const_cast<Float*>(FeatStd::Internal::PointerToPointer<const Float*>(vertexResource.GetData()));

            for (SizeType i = batchElementIndex; i < (batchElementIndex+batchElementSize); ++i) {
                const BatchElement& element = batch[i];
                const GlyphAtlasTextContainer* textContainer = element.textContainer;
                const CanvasText* textNode = textContainer->GetTextNode();
                Color color = textNode->GetTextColor();
                color.SetAlpha(color.GetAlpha() * textNode->GetEffectiveAlphaValue());
                const Matrix4& worldTransform = textNode->GetWorldTransform();

                for (SizeType j = element.startIndex; j < (element.startIndex + element.count); ++j) {
                    const GlyphAtlasTextContainer::GlyphInfo* glyphInfo = (*textContainer)[j];
                    FEATSTD_DEBUG_ASSERT(0 != glyphInfo);

                    const Float x0 = static_cast<Float>(glyphInfo->X);
                    const Float y0 = static_cast<Float>(glyphInfo->Y);
                    const Float x1 = x0 + static_cast<Float>(glyphInfo->Width);
                    const Float y1 = y0 + static_cast<Float>(glyphInfo->Height);
                    const Float z = 0.0F;

                    const GlyphAtlas3D::Entry::Rect& uvs = glyphInfo->UvRect;
                    const Vector2& topLeft = uvs.TopLeft;
                    const Vector2& bottomRight = uvs.BottomRight;
                    const Float u0 = topLeft.GetX();
                    const Float v0 = 1.0F - bottomRight.GetY();
                    const Float u1 = bottomRight.GetX();
                    const Float v1 = 1.0F - topLeft.GetY();

                    // 1st vertex
                    // position
                    Vector4 vtx0(x0, y1, z);
                    vtx0.TransformCoordinate(worldTransform);
                    *vertices++ = vtx0.GetX();
                    *vertices++ = vtx0.GetY();
                    *vertices++ = vtx0.GetZ();
                    if (minBounds.GetX() > vtx0.GetX()) {
                        minBounds.SetX(vtx0.GetX());
                    }
                    if (minBounds.GetY() > vtx0.GetY()) {
                        minBounds.SetY(vtx0.GetY());
                    }
                    if (minBounds.GetZ() > vtx0.GetZ()) {
                        minBounds.SetZ(vtx0.GetZ());
                    }
                    if (maxBounds.GetX() < vtx0.GetX()) {
                        maxBounds.SetX(vtx0.GetX());
                    }
                    if (maxBounds.GetY() < vtx0.GetY()) {
                        maxBounds.SetY(vtx0.GetY());
                    }
                    if (maxBounds.GetZ() < vtx0.GetZ()) {
                        maxBounds.SetZ(vtx0.GetZ());
                    }
                    // uv
                    *vertices++ = u0;
                    *vertices++ = v0;
                    // color
                    *vertices++ = color.GetRed();
                    *vertices++ = color.GetGreen();
                    *vertices++ = color.GetBlue();
                    *vertices++ = color.GetAlpha();

                    // 2nd vertex
                    // position
                    Vector4 vtx1(x1, y1, z);
                    vtx1.TransformCoordinate(worldTransform);
                    *vertices++ = vtx1.GetX();
                    *vertices++ = vtx1.GetY();
                    *vertices++ = vtx1.GetZ();
                    if (minBounds.GetX() > vtx1.GetX()) {
                        minBounds.SetX(vtx1.GetX());
                    }
                    if (minBounds.GetY() > vtx1.GetY()) {
                        minBounds.SetY(vtx1.GetY());
                    }
                    if (minBounds.GetZ() > vtx1.GetZ()) {
                        minBounds.SetZ(vtx1.GetZ());
                    }
                    if (maxBounds.GetX() < vtx1.GetX()) {
                        maxBounds.SetX(vtx1.GetX());
                    }
                    if (maxBounds.GetY() < vtx1.GetY()) {
                        maxBounds.SetY(vtx1.GetY());
                    }
                    if (maxBounds.GetZ() < vtx1.GetZ()) {
                        maxBounds.SetZ(vtx1.GetZ());
                    }
                    // uv
                    *vertices++ = u1;
                    *vertices++ = v0;
                    // color
                    *vertices++ = color.GetRed();
                    *vertices++ = color.GetGreen();
                    *vertices++ = color.GetBlue();
                    *vertices++ = color.GetAlpha();

                    // 3rd vertex
                    // position
                    Vector4 vtx2(x1, y0, z);
                    vtx2.TransformCoordinate(worldTransform);
                    *vertices++ = vtx2.GetX();
                    *vertices++ = vtx2.GetY();
                    *vertices++ = vtx2.GetZ();
                    if (minBounds.GetX() > vtx2.GetX()) {
                        minBounds.SetX(vtx2.GetX());
                    }
                    if (minBounds.GetY() > vtx2.GetY()) {
                        minBounds.SetY(vtx2.GetY());
                    }
                    if (minBounds.GetZ() > vtx2.GetZ()) {
                        minBounds.SetZ(vtx2.GetZ());
                    }
                    if (maxBounds.GetX() < vtx2.GetX()) {
                        maxBounds.SetX(vtx2.GetX());
                    }
                    if (maxBounds.GetY() < vtx2.GetY()) {
                        maxBounds.SetY(vtx2.GetY());
                    }
                    if (maxBounds.GetZ() < vtx2.GetZ()) {
                        maxBounds.SetZ(vtx2.GetZ());
                    }
                    // uv
                    *vertices++ = u1;
                    *vertices++ = v1;
                    // color
                    *vertices++ = color.GetRed();
                    *vertices++ = color.GetGreen();
                    *vertices++ = color.GetBlue();
                    *vertices++ = color.GetAlpha();

                    // 4th vertex
                    // position
                    Vector4 vtx3(x0, y0, z);
                    vtx3.TransformCoordinate(worldTransform);
                    *vertices++ = vtx3.GetX();
                    *vertices++ = vtx3.GetY();
                    *vertices++ = vtx3.GetZ();
                    if (minBounds.GetX() > vtx3.GetX()) {
                        minBounds.SetX(vtx3.GetX());
                    }
                    if (minBounds.GetY() > vtx3.GetY()) {
                        minBounds.SetY(vtx3.GetY());
                    }
                    if (minBounds.GetZ() > vtx3.GetZ()) {
                        minBounds.SetZ(vtx3.GetZ());
                    }
                    if (maxBounds.GetX() < vtx3.GetX()) {
                        maxBounds.SetX(vtx3.GetX());
                    }
                    if (maxBounds.GetY() < vtx3.GetY()) {
                        maxBounds.SetY(vtx3.GetY());
                    }
                    if (maxBounds.GetZ() < vtx3.GetZ()) {
                        maxBounds.SetZ(vtx3.GetZ());
                    }
                    // uv
                    *vertices++ = u0;
                    *vertices++ = v1;
                    // color
                    *vertices++ = color.GetRed();
                    *vertices++ = color.GetGreen();
                    *vertices++ = color.GetBlue();
                    *vertices++ = color.GetAlpha();
                }
            }

            canvasTextBatch->SetBoundingBox(minBounds, maxBounds);
            canvasTextBatch->SetCenter((minBounds + maxBounds) / 2.0F);
            canvasTextBatch->SetRadius(((maxBounds - minBounds) / 2.0F).GetLength());

            if (!vertexBuffer->IsUploaded()) {
                if (!vertexBuffer->Upload(DeviceObject::NoHint)) {
                    FEATSTD_LOG_ERROR("Could not upload vertex buffer.");
                }
            }
            else {
                if (vertexBuffer->Update(0, static_cast<UInt32>(glyphCount * 4))) {
                    vertexBuffer->SetElementCount(glyphCount * 6);
                }
                else {
                    FEATSTD_LOG_ERROR("Could not update vertex buffer.");
                }
            }
        }
    }

    VertexBuffer::SharedPointer Canvas::CreateVertexBuffer(SizeType maxGlyphCount) const
    {
        SizeType vertexCount = maxGlyphCount * 4;
        FEATSTD_DEBUG_ASSERT(vertexCount <= FeatStd::Internal::Limits<UInt16>::Max());

        SizeType vertexSize = sizeof(VertexUvColor);
        Float* vertices = CANDERA_NEW_ARRAY(Float, (vertexCount * vertexSize / sizeof(Float)));
        if (0 == vertices) {
            return VertexBuffer::SharedPointer(0);
        }

        SizeType indexCount = maxGlyphCount * 6;
        UInt16* indexBuffer = CANDERA_NEW_ARRAY(UInt16, indexCount);
        if (0 == indexBuffer) {
            CANDERA_DELETE_ARRAY(vertices);
            return VertexBuffer::SharedPointer(0);
        }

        for (SizeType i = 0; i < maxGlyphCount; ++i) {
            const UInt16 vertexOffset = FeatStd::Internal::NumericConversion<UInt16>(i * 4);
            const SizeType idx = i * 6;

            indexBuffer[idx] = 0 + vertexOffset;
            indexBuffer[idx + 1] = 1 + vertexOffset;
            indexBuffer[idx + 2] = 2 + vertexOffset;

            indexBuffer[idx + 3] = 0 + vertexOffset;
            indexBuffer[idx + 4] = 2 + vertexOffset;
            indexBuffer[idx + 5] = 3 + vertexOffset;
        }

        UInt16 vertexElementCount = static_cast<UInt16>(sizeof(VertexUvColorFormat) / sizeof(VertexGeometry::VertexElementFormat));
        VertexGeometry* vertexGeometry = CANDERA_NEW(VertexGeometry)(vertices,
            MemoryManagement::AdaptedArrayDisposer<const void*, const Float*>::Dispose,
            VertexUvColorFormat, 0,
            indexBuffer, MemoryManagement::AdaptedArrayDisposer<const void*, const UInt16*>::Dispose,
            static_cast<UInt32>(vertexCount), static_cast<UInt16>(vertexSize),
            vertexElementCount,
            static_cast<UInt32>(indexCount),
            VertexGeometry::VideoMemory,
            VertexGeometry::UInt16IndexedArrayBuffer,
            VertexGeometry::DynamicWrite);

        VertexBuffer::SharedPointer vertexBuffer = VertexBuffer::Create();
        // ignore return flag, because the vertex buffer was just created and cannot fail due to being uploaded already.
        if (vertexBuffer != 0) {
            static_cast<void>(vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose));
        }

        return vertexBuffer;
    }
}
