//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_UniformPropertySetter_H)
#define CANDERA_UniformPropertySetter_H

#include <Candera/EngineBase/Animation/AnimationPropertySetter.h>
#include <Candera/Engine3D/ShaderParamSetters/ShaderParamSetter.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Macros.h>

namespace Candera {

    namespace Animation {

        /** @addtogroup AnimationPropertySetters3D
         *  @{
         */

        /** @brief A PropertySetter that sets a uniform of a ShaderParamSetter. Base class.*/
        class UniformPropertySetter : public AnimationPropertySetter {

                FEATSTD_TYPEDEF_BASE(AnimationPropertySetter);

            public:
                FEATSTD_RTTI_DECLARATION();

                FEATSTD_TYPEDEF_SHARED_POINTER(UniformPropertySetter);

                /**
                 *  Constructor
                 */
                UniformPropertySetter();

                /**
                 *  Destructor
                 */
                ~UniformPropertySetter();

                /**
                 *  Set the ShaderParamSetter.
                 *  @param shaderParamSetter ShaderParamSetter to set uniforms.
                 */
                void SetShaderParamSetter(MemoryManagement::SharedPointer<ShaderParamSetter> shaderParamSetter) {
                    m_shaderParamSetter = shaderParamSetter;
                }

                /**
                 *  Retrieves the ShaderParamSetter.
                 *  @return The current ShaderParamSetter
                 */
                MemoryManagement::SharedPointer<ShaderParamSetter> GetShaderParamSetter() const {
                    return m_shaderParamSetter;
                }

                /**
                 *  Set the uniform name.
                 *  @param uniformName Name of shader uniform to set.
                 */
                void SetUniformName(const Char* uniformName) {
                    m_uniformName = uniformName;
                }

                /**
                 *  Retrieves the uniform name.
                 *  @return The current uniform name.
                 */
                const Char* GetUniformName() const {
                    return m_uniformName;
                }

                /**
                 *  Set the index of the uniform in an array.
                 *  An array uniform can not be animated all at once, only element by element. If more or all
                 *  elements in an array should be animated, a corresponding PropertySetter for each one
                 *  shall be created.
                 *
                 *  @param index     If the uniform is not an array (uniform Float u_myFloat;)
                 *                     then index must be always 0.
                 *                   If the uniform is an array (uniform Float u_myFloats[count];)
                 *                     then the index can take a value between 0 and count-1.
                 */
                void SetUniformIndex(Int32 index) {
                    m_index = index;
                }

                /**
                 *  Retrieves the index of the uniform in an array.
                 *  An array uniform can not be animated all at once, only element by element. If more or all
                 *  elements in an array should be animated, a corresponding PropertySetter for each one
                 *  shall be created.
                 *  If the uniform is not an array (uniform Float u_myFloat;)
                 *    then index must be always 0.
                 *  If the uniform is an array (uniform Float u_myFloats[count];)
                 *    then the index can take a value between 0 and count-1.
                 *  @return The current index of the uniform in an array.
                 */
                Int32 GetUniformIndex() const {
                    return m_index;
                }

            protected:
                void SetFloat(const Float* value, Int32 size);
                void SetInt(const Float* value, Int32 size);
                void SetBool(const Float* value, Int32 size);

            private:
                Int32 m_index;
                const Char* m_uniformName;
                MemoryManagement::SharedPointer<ShaderParamSetter> m_shaderParamSetter;

                bool IsValid();

                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1704, Candera::Animation::UniformPropertySetter::UniformPropertySetter, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                UniformPropertySetter(const UniformPropertySetter&);
                UniformPropertySetter& operator=(const UniformPropertySetter&);
        };

        /** @} */ // end of AnimationPropertySetters3D

    }

}
#endif
