//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATH2D_H)
#define CANDERA_MATH2D_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera {

    class Camera2D;
    class RenderTarget2D;
    class RenderNode;
    class Rectangle;
    class Node2D;

/** @addtogroup Mathematics2D
 *  @{
 */

    /**
    * @brief The Math2D class for 2D transformation and helper functions.
    */
class Math2D
{
public:

    /**
     *  Transform a point from scene space to camera viewport space.
     *  @param camera Camera used for the transformation
     *  @param pos A point in scene space.
     *  @return The corresponding point in camera viewport space.
     */
    static Vector2 TransformSceneToViewport(const Camera2D& camera, const Vector2& pos);

    /**
     *  Transform a point from camera viewport space to scene space.
     *  @param camera Camera used for the transformation.
     *  @param pos A point in camera viewport space.
     *  @return The corresponding point in scene space.
     */
    static Vector2 TransformViewportToScene(const Camera2D& camera, const Vector2& pos);

    /**
     *  Transform a point from camera viewport space to render target space.
     *  @param camera Camera used for the transformation.
     *  @param pos A point in camera viewport space.
     *  @return The corresponding point in render target space.
     */
    static Vector2 TransformViewportToRenderTarget(const Camera2D& camera, const Vector2& pos);

    /**
     *  Transform a point from render target space to camera viewport space.
     *  @param camera Camera used for the transformation.
     *  @param pos A point in render target space.
     *  @return The corresponding point in camera viewport space.
     */
    static Vector2 TransformRenderTargetToViewport(const Camera2D& camera, const Vector2& pos);

    /**
     *  Transform a point from render target space to screen space.
     *  @param renderTarget Render target used for the transformation.
     *  @param pos A point in render target space.
     *  @return The corresponding point in screen space.
     */
    static Vector2 TransformRenderTargetToScreen(const RenderTarget2D& renderTarget, const Vector2& pos);

    /**
     *  Transform a point from screen space to render target space.
     *  @param renderTarget Render target used for the transformation.
     *  @param pos A point in screen space.
     *  @return The corresponding point in render target space.
     */
    static Vector2 TransformScreenToRenderTarget(const RenderTarget2D& renderTarget, const Vector2& pos);

    /**
     * Transform a point from screen space to world space.
     * @param camera Camera used at a transformation to the screen space.
     * @param pos A point in the screen space.
     * @return The corresponding point in the world space.
     */
    static Vector2 TransformScreenToWorld(const Camera2D& camera, const Vector2& pos);

    /**
    * Transform a point from screen space to object space.
    * @param camera Camera used at a transformation to the screen space.
    * @param pos A point in the screen space.
    * @param node The node whose object space the returned point will be relative to.
    * @return The corresponding point in the object space.
    */
    static Vector2 TransformScreenToObject(const Camera2D& camera, const Vector2& pos, const Node2D& node);

    /**
     * Transforms a bounding box to surface aligned space considering perspective transformations of an attached perspective warping effect.
     * @param camera The camera that is linked to the surface.
     * @param node The node to consider for 2D transformation.
     * @param perspectivePosition Perspective 3D position.
     * @param perspectiveRotation  Perspective 3D rotation.
     * @param perspectiveScale Perspective 3D scale.
     * @param perspectivePivot
     * @param perspectiveNearPlane Perspective projection near plane.
     * @param perspectiveFarPlane Perspective projection far plane.
     * @param perspectiveFovY Perspective projection fovY.
     * @param perspectiveAspectRatio Perspective projection aspect ratio.
     * @param perspectiveViewport Perspective viewport.
     * @param boundingBox Out parameter: The bounding box to calculate.
     * @return Whether the bounding box could be calculated (True) or not (False).
     */
    static bool GetSurfaceAlignedPerspectiveBoundingRectangle(const Camera2D& camera, const RenderNode& node,
        const Vector3& perspectivePosition, const Vector3& perspectiveRotation, const Vector3& perspectiveScale, const Vector3& perspectivePivot,
        Float perspectiveNearPlane, Float perspectiveFarPlane, Float perspectiveFovY, 
        Float perspectiveAspectRatio, const Rectangle& perspectiveViewport,  Rectangle& boundingBox);
};

 /** @} */ // end of Mathematics3D
} // namespace Candera

#endif  // CANDERA_MATH3D_H
