//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Math2D.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/Window.h>
#include <Candera/System/Diagnostics/Log.h>

#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Rectangle.h>

namespace Candera {
FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine2D);

Vector2 Math2D::TransformViewportToScene(const Camera2D& camera, const Vector2& pos)
{
    return camera.GetWorldTransform().Multiply(pos);
}

Vector2 Math2D::TransformSceneToViewport(const Camera2D& camera, const Vector2& pos)
{
    return camera.GetViewMatrix().Multiply(pos);
}

Vector2 Math2D::TransformViewportToRenderTarget(const Camera2D& camera, const Vector2& pos)
{
    return pos + camera.GetViewport().GetPosition();
}

Vector2 Math2D::TransformRenderTargetToViewport(const Camera2D& camera, const Vector2& pos)
{
    return pos - camera.GetViewport().GetPosition();
}

static Vector2 ApplyRenderTargetTransformation(const RenderTarget2D& renderTarget, const Vector2& pos, 
    Vector2 (*transformation)(const Vector2&, const Vector2&, const Vector2&, const Vector2&))
{
    GraphicDeviceUnit* gdu = renderTarget.GetGraphicDeviceUnit();
    if (gdu != 0) {
        Window* window = gdu->ToWindow();
        bool isWindowValid = 
            (window != 0) && 
            (window->GetWidth() != 0) &&
            (window->GetHeight() != 0);
        if (isWindowValid) {
            Rectangle windowRectangle = Rectangle(
                static_cast<Float>(window->GetX()),
                static_cast<Float>(window->GetY()),
                static_cast<Float>(window->GetWidth()),
                static_cast<Float>(window->GetHeight()));
            Vector2 renderTargetSize = Vector2(
                static_cast<Float>(renderTarget.GetWidth()),
                static_cast<Float>(renderTarget.GetHeight()));
            return transformation(
                pos, 
                windowRectangle.GetPosition(),
                windowRectangle.GetSize(),
                renderTargetSize);
        }
    }
    return pos;
}

static Vector2 RenderTargetToScreenTransformation(
    const Vector2& pos, 
    const Vector2& windowPosition, 
    const Vector2& windowSize, 
    const Vector2& renderTargetSize)
{
    return ((pos * windowSize) / renderTargetSize) + windowPosition;
}

static Vector2 ScreenToRenderTargetTransformation(
    const Vector2& pos, 
    const Vector2& windowPosition, 
    const Vector2& windowSize, 
    const Vector2& renderTargetSize)
{
    return ((pos - windowPosition) * renderTargetSize) / windowSize;
}

Vector2 Math2D::TransformRenderTargetToScreen(const RenderTarget2D& renderTarget, const Vector2& pos)
{
    return ApplyRenderTargetTransformation(
        renderTarget,
        pos,
        &RenderTargetToScreenTransformation);
}

Vector2 Math2D::TransformScreenToRenderTarget(const RenderTarget2D& renderTarget, const Vector2& pos)
{
    return ApplyRenderTargetTransformation(
        renderTarget,
        pos,
        &ScreenToRenderTargetTransformation);
}

Vector2 Math2D::TransformScreenToWorld(const Camera2D& camera, const Vector2& pos)
{
    Vector2 pointInRenderTargetSpace = pos;
    if (!DevicePackageDescriptor::GetPlatformDescription().isWindowManagerUsed) {
        const RenderTarget2D* renderTarget = camera.GetRenderTarget();
        if (renderTarget != 0) {
            pointInRenderTargetSpace = Math2D::TransformScreenToRenderTarget(
                *renderTarget, pointInRenderTargetSpace);
        }
    }
    const Vector2 pointInWorldSpace =
        Math2D::TransformViewportToScene(camera,
        Math2D::TransformRenderTargetToViewport(camera,
        pointInRenderTargetSpace));

    return pointInWorldSpace;
}

Vector2 Math2D::TransformScreenToObject(const Camera2D& camera, const Vector2& pos, const Node2D& node)
{
    const Vector2 pointInWorldSpace = TransformScreenToWorld(camera, pos);
    Matrix3x2 matrix(node.GetWorldTransform());
    matrix.Inverse();
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    return matrix.Multiply(pointInWorldSpace) + node.GetPivotOffset();
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
}

bool CorrectViewport(const Rectangle& dstArea, const Rectangle& srcPerspectiveViewport,
    Float width, Float height,
    Rectangle &correction, Rectangle& perspectiveViewport)
{
    // Compute viewport
    Rectangle viewport;
    correction.SetLeft(0.F);
    correction.SetTop(0.F);

    viewport.SetLeft(Math::Floor(dstArea.GetLeft()));
    viewport.SetWidth(
        (dstArea.GetWidth() < 0.0F) ?
        (width - viewport.GetLeft()) :
        Math::Ceil((dstArea.GetWidth() + viewport.GetLeft()) - dstArea.GetLeft()));
    if (viewport.GetLeft() < 0.F) {
        correction.SetLeft(viewport.GetLeft());
        viewport.SetWidth(viewport.GetWidth() + viewport.GetLeft());
        viewport.SetLeft(0.F);
    }
    if ((viewport.GetWidth() + viewport.GetLeft()) > width) {
        viewport.SetWidth(width - viewport.GetLeft());
    }
    correction.SetWidth(viewport.GetWidth());
    if (viewport.GetWidth() <= 0.0F) {
        return false;
    }

    viewport.SetTop(Math::Floor(dstArea.GetTop()));
    viewport.SetHeight(
        (dstArea.GetHeight() < 0.0F) ?
        (height - viewport.GetTop()) :
        Math::Ceil((dstArea.GetHeight() + viewport.GetTop()) - dstArea.GetTop()));
    viewport.SetTop((height - viewport.GetTop()) - viewport.GetHeight());
    if (viewport.GetTop() < 0.F) {
        viewport.SetHeight(viewport.GetHeight() + viewport.GetTop());
        viewport.SetTop(0.F);
    }
    if ((viewport.GetHeight() + viewport.GetTop()) > height) {
        correction.SetTop(height - (viewport.GetHeight() + viewport.GetTop()));
        viewport.SetHeight(height - viewport.GetTop());
    }
    correction.SetHeight(viewport.GetHeight());
    if (viewport.GetHeight() <= 0.0F) {
        return false;
    }

    //Correction ~ 3D viewport, used later on for normalizing matrices.
    perspectiveViewport = srcPerspectiveViewport;

    perspectiveViewport.SetLeft(perspectiveViewport.GetLeft() + viewport.GetLeft());
    perspectiveViewport.SetTop(perspectiveViewport.GetTop() + dstArea.GetTop());
    if (perspectiveViewport.GetWidth() < 0.0F) {
        perspectiveViewport.SetWidth(width);
    }

    if (perspectiveViewport.GetHeight() < 0.0F) {
        perspectiveViewport.SetHeight(height);
    }
    perspectiveViewport.SetTop((height - perspectiveViewport.GetTop()) - perspectiveViewport.GetHeight()); //Flip Top.


    correction.SetLeft(0.0F);
    correction.SetTop(0.F);
    if (perspectiveViewport.GetLeft() < 0.F) {
        correction.SetLeft(perspectiveViewport.GetLeft());
    }
    if ((perspectiveViewport.GetHeight() + perspectiveViewport.GetTop()) > height) {
        correction.SetTop(height - (perspectiveViewport.GetHeight() + perspectiveViewport.GetTop()));
        perspectiveViewport.SetHeight(height - perspectiveViewport.GetTop());
    }

    correction.SetWidth(perspectiveViewport.GetWidth());
    correction.SetHeight(perspectiveViewport.GetHeight());


    return true;
}

void TransformSurfaceSpaceToOpenGLSpace(Matrix4& toTransform,
    const Rectangle& sourceRectangle,
    const Rectangle& correction) {

    Float rectangleWidth = sourceRectangle.GetWidth();
    Float rectangleHeight = sourceRectangle.GetHeight();
    Float correctionHalfWidth = 2.0F / correction.GetWidth();
    Float correctionHalfHeight = 2.0F / correction.GetHeight();


    //Bring matrix to viewport space.
    toTransform(0,0) *= rectangleWidth;
    toTransform(0,1) *= rectangleWidth;
    toTransform(0,2) *= rectangleWidth;
    toTransform(1,0) *= rectangleHeight;
    toTransform(1,1) *= rectangleHeight;
    toTransform(1,2) *= rectangleHeight;

    toTransform(3,0) += correction.GetLeft();
    toTransform(3,1) += correction.GetTop();

    toTransform(0,0) *= correctionHalfWidth;
    toTransform(0,1) *= -correctionHalfHeight;
    toTransform(0,2) *= correctionHalfHeight;
    toTransform(1,0) *= correctionHalfWidth;
    toTransform(1,1) *= -correctionHalfHeight;
    toTransform(1,2) *= correctionHalfHeight;
    toTransform(3,0) *= correctionHalfWidth;
    toTransform(3,1) *= -correctionHalfHeight;
    toTransform(3,2) *= correctionHalfHeight;

    toTransform(3,0) += (-1.0F);
    toTransform(3,1) += (1.0F);
}

bool Math2D::GetSurfaceAlignedPerspectiveBoundingRectangle(const Camera2D& camera, const RenderNode& node,
    const Vector3& perspectivePosition, const Vector3& perspectiveRotation, const Vector3& perspectiveScale, const Vector3& perspectivePivot,
    Float perspectiveNearPlane, Float perspectiveFarPlane, 
    Float perspectiveFovY, Float perspectiveAspectRatio, const Rectangle& perspectiveViewport,
    Rectangle& boundingBox)
{
    const RenderTarget2D* renderTarget = camera.GetRenderTarget();

    if (renderTarget == 0) {
        return false;
    }

    Rectangle correction;
    Rectangle correctedViewport;
    if (!CorrectViewport(camera.GetViewport(),
        perspectiveViewport,
        static_cast<Float>(renderTarget->GetWidth()), static_cast<Float>(renderTarget->GetHeight()),
        correction, correctedViewport)) {
        return false;
    }


    Rectangle sourceRectangle;
    Effect2D* nodeEffect = node.GetEffect(0);
    if (nodeEffect == 0) {
        return false;
    }

    nodeEffect->GetBoundingRectangle(sourceRectangle);

    Vector4 normalizedSourceRectangle[4];
    normalizedSourceRectangle[0] = Vector4(0.0F, 0.0F, 0.0F, 1.0F);
    normalizedSourceRectangle[1] = Vector4(1.0F, 0.0F, 0.0F, 1.0F);
    normalizedSourceRectangle[2] = Vector4(0.0F, 1.0F, 0.0F, 1.0F);
    normalizedSourceRectangle[3] = Vector4(1.0F, 1.0F, 0.0F, 1.0F);

    //Compute mvp matrix.
    const Matrix3x2 local2DTransform = node.GetWorldTransform() * camera.GetViewMatrix();
    Matrix4 srcPerspectiveTransformation;
    Matrix4 srcPerspectiveProjection;
    if (!RenderDevice2D::CalculatePerspectiveTransformationMatrix(srcPerspectiveTransformation, perspectivePosition, perspectiveRotation, perspectiveScale, perspectivePivot)) {
        return false;
    }
    if (!RenderDevice2D::CalculatePerspectiveProjectionMatrix(srcPerspectiveProjection, &camera, 
        perspectiveViewport, perspectiveNearPlane, perspectiveFarPlane, perspectiveFovY, perspectiveAspectRatio)) {
            return false;
    }

    //"Cast" src 2D matrix to 3D matrix.
    Matrix4 modelMatrix(
        local2DTransform(0, 0), local2DTransform(0, 1), 0.0F, 0.0F,
        local2DTransform(1, 0), local2DTransform(1, 1), 0.0F, 0.0F,
        0.0F, 0.0F, 1.0F, 0.0F,
        local2DTransform(2, 0), local2DTransform(2, 1), 0.0F, 1.0F);
    //3D matrix transformation is child of node 2D transformation.
    modelMatrix = srcPerspectiveTransformation * modelMatrix;

    TransformSurfaceSpaceToOpenGLSpace(modelMatrix, sourceRectangle, correction);

    //View is fixed for Perspective warping effects.
    Matrix4 mvpMatrix = modelMatrix  /* * view.SetIdentity()*/ * srcPerspectiveProjection;

    Vector2 minBound(Math::MaxFloat(), Math::MaxFloat());
    Vector2 maxBound(Math::MinFloat(), Math::MinFloat());

    for (UInt32 i = 0; i < 4; i++) {
        normalizedSourceRectangle[i].TransformCoordinate(mvpMatrix);
        if (normalizedSourceRectangle[i].GetW() == 0.0F) {
            FEATSTD_LOG_ERROR("GetSurfaceAlignedPerspectiveBoundingRectangle failed, vec4.GetW() == 0.");
            return false;
        }
 

        normalizedSourceRectangle[i] /= Math::Absolute(normalizedSourceRectangle[i].GetW());
        normalizedSourceRectangle[i].SetX(((normalizedSourceRectangle[i].GetX() + 1.0F) / 2.0F) * correctedViewport.GetWidth());
        normalizedSourceRectangle[i].SetY(((normalizedSourceRectangle[i].GetY() + 1.0F) / 2.0F) *  correctedViewport.GetHeight());
        normalizedSourceRectangle[i].SetY(correctedViewport.GetHeight() - normalizedSourceRectangle[i].GetY());

        maxBound.SetX(Math::Maximum<Float>(normalizedSourceRectangle[i].GetX(), maxBound.GetX()));
        minBound.SetX(Math::Minimum<Float>(normalizedSourceRectangle[i].GetX(), minBound.GetX()));
        maxBound.SetY(Math::Maximum<Float>(normalizedSourceRectangle[i].GetY(), maxBound.GetY()));
        minBound.SetY(Math::Minimum<Float>(normalizedSourceRectangle[i].GetY(), minBound.GetY()));
    }

    boundingBox = Rectangle(minBound.GetX(), minBound.GetY(), maxBound.GetX() - minBound.GetX(), maxBound.GetY() - minBound.GetY());
    boundingBox.SetPosition(boundingBox.GetPosition() + perspectiveViewport.GetPosition() + camera.GetViewport().GetPosition());

    return true;
}

} // namespace Candera

