//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Effect2D_H)
    #define CANDERA_Effect2D_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <CanderaPlatform/Device/Common/Base/Effect2DMetaInformation.h>

namespace Candera {
/** @addtogroup Effects2D
 *  @{
 */

    class Matrix3x2;

    class BrushEffect2D;
    class InPlaceEffect2D;
    class BlendEffect2D;

    class Rectangle;
    class Vector2;

    /**
     *  @brief Base class for all 2D effects which can be added to a 2D RenderNode.
     */
    class Effect2D : public CanderaObject {
        FEATSTD_TYPEDEF_BASE(CanderaObject);

        public:

            FEATSTD_TYPEDEF_SHARED_POINTER(Effect2D);

            FEATSTD_RTTI_DECLARATION();

            /**
             *  Destructor
             */
            virtual ~Effect2D() override {}

            /**
             *  Gets a valid BrushEffect2D pointer if the effect is a brush effect or
             *  if it is a combined effect and has a brush effect, otherwise 0.
             *  @return     A valid BrushEffect2D pointer if the effect is a brush effect or
             *              if it is a combined effect and has a brush effect, otherwise 0.
             */
            virtual BrushEffect2D* GetBrushEffect2D() { return 0; }

            /**
             *  Retrieve the number of in-place effects in case a combined effect implements
             *  more than one in-place effects.
             *  @return     The number of implemented in-place effects.
             */
            virtual UInt8 GetInPlaceEffect2DCount() const { return 0; }

            /**
             *  Gets a valid InPlaceEffect2D pointer if the effect is an in-place effect or
             *  if it is a combined effect and has an in-place effect, otherwise 0.
             *  @param      index   The index of the in-place effect.
             *  @return             A valid InPlaceEffect2D pointer if the effect is an in-place effect or
             *                      if it is a combined effect and has an in-place effect, otherwise 0.
             */
            virtual InPlaceEffect2D* GetInPlaceEffect2D(UInt8 index);

            /**
             *  Gets a valid BlendEffect2D pointer if the effect is a blend effect or
             *  if it is a combined effect and has a blend effect, otherwise 0.
             *  @return     A valid BlendEffect2D pointer if the effect is a blend effect or
             *              if it is a combined effect and has a blend effect, otherwise 0.
             */
            virtual BlendEffect2D* GetBlendEffect2D() { return 0; }

            /**
             *  Retrieves an axis-aligned bounding rectangle in local space coordinates.
             *  For non-brush effects, the bounding rectangle is adjusted to reflect the change applied by the effect.
             *  @param   boundingRectangle Is an out parameter describing the bounding rectangle of this Effect2D object.
             *                            For non-brush effect is also an input parameter describing the bound of the input
             *                            image.
             */
            virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const;

            /**
             *  Retrieve the rectangle that is used for layout this effect.
             *  This can be different from the bounding box for effects like
             *  text or shadow, when layout should be done by a logical
             *  bound and not by the rectangle bounding the pixels updated by
             *  the effect.
             *  For non-brush effects, the size is adjusted to reflect the
             *  change applied by the effect.
             *  @param rectangle Is an output parameter describing the size of
             *                   this Effect2D object. For non-brush effect is
             *                   also an input parameter describing the size of
             *                   the input image.
             */
            virtual void GetLayoutingRectangle(Rectangle& rectangle) const;

            /**
             *  Get the point in local space coordinates that represents the base of the object
             *  rendered by the effect. This is typically in the center of a bitmap, or on the base line of a text.
             *  For non-brush effects, the base point is adjusted to reflect the change applied by the effect.
             *  @param   basePoint Is an output parameter describing the base point of this Effect2D object.
             *                    For non-brush effect is also an input parameter describing the base point of the input
             *                    image.
             */
            virtual void GetBasePoint(Vector2& basePoint) const;

            /**
             *  Retrieves the meta-information associated with an effect.
             *  @return Pointer to a meta-information object.
             */
            virtual MetaInfo::Effect2DMetaInfo* GetMetaInfo() const { return 0; }

            /**
             *  Uploads all associated resources.
             *  @return true if successful.
             */
            virtual bool Upload() { return true; }

            /**
             *  Unloads all associated resources.
             *  @return true if successful.
             */
            virtual bool Unload() { return true; }

            /**
             *  Uploads all resources that changed since last Upload/Update.
             *  Unloads resources removed from the effect, that were previously uploaded
             *  by an Upload/Update.
             *  This should only be called on Uploaded effects. (otherwise, call Upload.)
             *  @return true if successful.
             */
            virtual bool Update() { return true; }

            /**
             *   Check whether all the resources are uploaded.
             *   @return true if all the resources are uploaded, false otherwise.
             */
            virtual bool IsUploaded() const  { return true; }

            /**
             * Returns a copy of this effect. No behavior here, but derived class introduce cloning behavior.
             * @return Pure virtual function that must be overridden in each object derived from Effect2D.
             *         Derived classes return a copy of this effect.
             */
            virtual Effect2D::SharedPointer Clone() const = 0;

            /**
             * Mesh2D cannot be supported by all effects, as certain effects would create 2D vertex buffers, which would be overriden by the Mesh2D.
             * This flag helps to distinguish between effects that can be used with Mesh2D and effects that are not available.
             * @return Whether Mesh2D can be used with this effect (true) or not (false).
             */
            virtual bool IsMesh2DSupported() const { return false; }

        protected:
            // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of all derived classes.
            Effect2D();
            explicit Effect2D(const Effect2D& rhs);
            Effect2D& operator = (const Effect2D& rhs);

            /**
             *  Triggers the rendering for the effect.
             *  @param  input       Defines the input for the effect. The first effect inside the effect list NO input. The input for the second
             *                      effect in the list must be set to the output of the first effect and so on.
             *  @param  inputArea   Area of the input that should be used as input. This should by supplied through the ouputArea parameter of the last effect.
             *  @param  transform   Defines the transformation matrix for the effect.
             *  @param  node        Node this effect is referenced to, containing transformation matrix for the effect.
             *  @param  output      Defines the output for the effect. The last effect inside the effect list must be the context of the render target.
             *  @param  outputArea  Output parameter through which the effect supplies the area of the output that was modified.
             */
            virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea) = 0;

        private:
            friend class RenderNode;

            CANDERA_SHARED_POINTER_DECLARATION();
    };

    inline void Effect2D::GetBoundingRectangle(Rectangle& /*boundingRectangle*/) const {}
    inline void Effect2D::GetLayoutingRectangle(Rectangle& /*rectangle*/) const {}
    inline void Effect2D::GetBasePoint(Vector2& /*basePoint*/) const {}
    inline InPlaceEffect2D* Effect2D::GetInPlaceEffect2D(UInt8 /*index*/) { return 0; }

 /** @} */ // end of Effects2D
}   // namespace Candera
#endif  // CANDERA_Effect2D_H
