//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VERTEX_BUFFER_2D_H)
#define CANDERA_VERTEX_BUFFER_2D_H

#include <Candera/Engine2D/Core/DeviceObject2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <FeatStd/Diagnostics/Measurable.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <Candera/System/Mathematics/Rectangle.h>



namespace Candera {

    /** @addtogroup Core2D
    *  @{
    */
    class VertexGeometry2D;

    /**
    *  @brief   The VertexBuffer2D encapsulates the attributes of an uploaded VertexGeometry2D.
    *           It holds the actual handles to the allocated and filled VRAM memory.
    *           Before is called the first time a VertexGeometry2D must be set, this will be upload to VRAM.
    *           The VertexBuffer implements manual reference counting to prevent multiple uploads into VRAM,
    *           the actual unload only takes place if the reference count gets 0.
    *           This enables the VertexBuffer to be shareable between different Mesh2D objects without
    *           multiple uploads into VRAM.
    *
    */
    class VertexBuffer2D : public DeviceObject2D {
        FEATSTD_TYPEDEF_BASE(DeviceObject2D);
        friend class RenderDevice2D;

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(VertexBuffer2D);

        /**
        *  Creates a VertexBuffer2D object with default values.
        *  @return Shared Pointer to the created VertexBuffer2D object.
        */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        typedef MemoryManagement::Disposer<const VertexGeometry2D*> VertexGeometry2DDisposer;
        typedef VertexGeometry2DDisposer::DisposerFunction VertexGeometry2DDisposerFn;

        /**
        *  The primitive types
        */
        enum PrimitiveType {
            Triangles,      ///< Triangle primitive, vertex geometry represents triplets of vertices forming triangles:   123456 -> 123, 456.
            TriangleStrip,  ///< Triangle primitive, vertex geometry represents triplets of vertices forming triangles:   123456 -> 123, 234, 345, 456.
            TriangleFan,    ///< Triangle primitive, vertex geometry represents triplets of vertices forming triangles:   123456 -> 123, 134, 145, 156.
            Lines,          ///< Line primitive, vertex geometry represents pairs of vertices forming lines:              123456 -> 12, 34, 56.
            LineStrip,      ///< Line primitive, vertex geometry represents list of connected vertices forming a line:    123456 -> 12, 23, 34, 45, 56.
            LineLoop,       ///< Line primitive, vertex geometry represents a circular list of connected vertices:        123456 -> 12, 23, 34, 45, 56, 61.
            Points          ///< Point primitive, vertex geometry represents single dots:                                 123456 -> 1, 2, 3, 4, 5, 6.
        };


        /**
        *  Destructor
        */
        virtual ~VertexBuffer2D() override;

        /**
        *  Updates data within the VertexArray of an already uploaded VertexBuffer2D directly in VRAM.
        *  The data is taken from the VertexArray of the attached VertexGeometry2D in the VertexBuffer.
        *  Note: Only existing vertices in a buffer can be updated.The buffer size cannot be changed, indices are also not to be updated.
        *  @param index     The starting index (beginning from 0) of the VertexArray.
        *  @param count     The count of vertices to update.
        *  @return          Whether data in VRAM has been updated(true) or not(false).
        *                   Also fails if vertices are not in VRAM.
        */
        bool Update(UInt32 index, UInt32 count);

        /**
        *  Updates data within the VertexArray of an already uploaded VertexBuffer2D directly in VRAM.
        *  Note: Only existing vertices in a buffer can be updated.The buffer size cannot be changed, indices are also not to be updated.
        *  @param offset    The offset at which the update should start.
        *  @param size      The size in bytes of the data.
        *  @param data      The source data to update in the already uploaded VRAM VertexBuffer2D.
        *  @return          Whether data in VRAM has been updated(true) or not(false).
        *                   Also fails if vertices are not in VRAM.
        */
        bool Update(Int offset, Int size, const UInt8* data);

        /**
         * Retrieves GeometryHandle that is associated with this vertex buffer 2D in the RenderDevice2D implementation.
         * @return Retrieves GeometryHandle that is associated with this vertex buffer 2D in the RenderDevice2D implementation.
         */
        GeometryHandle Get2DGeometryHandle() const { return m_geometryHandle; }


        /**
        *  Sets the 2D vertex geometry which will be uploaded to VRAM and the corresponding Disposer for the vertex geometry.
        *  This needs to be set before Upload can be called.
        *  @param vertexGeometry     The vertex geometry2D.
        *  @param disposerFn         The function which defines the dispose method for the 2D vertex geometry.
        *                            If the Disposer is set to null the vertex geometry will not be disposed.
        *  @return                   True if vertex geometry was set. False if it's already uploaded.
        */
        bool SetVertexGeometry2D(VertexGeometry2D* vertexGeometry2D, VertexGeometry2DDisposerFn disposerFn = 0);

        /**
        *  Retrieves the 2D vertex geometry.
        *  @return A pointer to the 2D vertex geometry.
        */
        const VertexGeometry2D* GetVertexGeometry2D() const {
            return m_vertexGeometry2D;
        }
        VertexGeometry2D* GetVertexGeometry2D()
        {
            return m_vertexGeometry2D;
        }

        /**
        *  Sets the vertex primitive type. @see PrimitiveType.
        *  This value defines how the vertices stored in VertexGeometry are connected and therefore how the primitive is assembled.
        *  @param primitiveType specifies the primitive type of this VertexBuffer.
        */
        void SetPrimitiveType(PrimitiveType primitiveType) {
            m_primitiveType = primitiveType;
        }

        /**
        *  Retrieves the vertex primitive type. @see PrimitiveType.
        *  @return The vertex primitive type.
        */
        PrimitiveType GetPrimitiveType() const {
            return m_primitiveType;
        }

        /**
        * Clones a VertexBuffer2D. Note:
        * Ownership of the cloned VertexGeometry2D member will be in the cloned VertexBuffer2D instance.
        * Note: The clone is not Uploaded, it needs to be Uploaded separatedly.
        * @return The cloned VertexBuffer2D or a SharedPointer to 0 if it couldn't be cloned.
        */
        VertexBuffer2D::SharedPointer Clone() const;

        /**
         * Queries whether the underlying RenderDevice2D supports the chosen primitive type.
         * @param type The PrimitiveType to check if it is supported.
         * @return true if the given primitive type is supported by the underyling RenderDevice2D.
         */
        bool IsPrimitiveTypeSupported(PrimitiveType type) const;

        /**
        * Queries whether the underlying RenderDevice2D supports index buffers or not.
        * @return true if the index buffers are supported by the RenderDevice2D.
        */
        bool IsIndexBufferSupported() const;

        /**
         * Invalidates the bounding box. This needs to be set if e.g. a 2D vertex buffer gets updated.
         * Otherwise the new bounding box calculation will not be triggered.
         * In most cases this is set automatically except 2D Vertices in SystemMemory get updated. Here the user needs to call this function.
         */
        void SetBoundingBoxInvalid() { m_isBoundingRectangleValid = false; }

        /**
         * Calculates (if invalidated) and retrieves the bounding rectangle of this vertex buffer.
         */
        const Rectangle& GetBoundingRectangle();

    protected:

        /**
        *  overridden from DeviceObject2D
        *  @return Whether uploading 2D vertex geometry to VRAM succeeded (true), stays in system memory(true) or failed(false).
        */
        virtual bool UploadInternal(LoadingHint loadingHint) override;

        /**
        *  overridden from DeviceObject2D
        *  @return Whether unloading 2D vertex geometry from VRAM succeeded (true), is in system memory and doesn't have to be unloaded(true)
        *  or failed(false).
        */
        virtual bool UnloadInternal(LoadingHint loadingHint) override;

        // override DeviceObject::DisposeInternal
        virtual void DisposeInternal() override;


    private:

        friend class RenderDevice2D;

        VertexGeometry2D* m_vertexGeometry2D;
        VertexGeometry2DDisposerFn m_vertexGeometry2DDisposerFn;
        GeometryHandle m_geometryHandle;
        PrimitiveType m_primitiveType;
        bool m_isBoundingRectangleValid;
        Rectangle m_boundingRectangle;

        //FEATSTD_MAKE_CLASS_UNCOPYABLE(VertexBuffer2D); = operator needed for cloning.
        VertexBuffer2D& operator=(const VertexBuffer2D&);
        VertexBuffer2D(const VertexBuffer2D& rhs);
        VertexBuffer2D();

        void SetGeometryHandle(GeometryHandle handle) {
            m_geometryHandle = handle;
        }
        void DisposeVertexGeometry2D();

        CANDERA_SHARED_POINTER_DECLARATION();


    };

}

/** @} */ // end of Core2D
#endif
