//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VertexBuffer2D.h"
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/Engine2D/Core/VertexGeometry2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

namespace Candera {
    using namespace Diagnostics;
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine2D);


    VertexBuffer2D::VertexBuffer2D():
        Base(),
        m_vertexGeometry2D(0),
        m_vertexGeometry2DDisposerFn(0),
        m_geometryHandle(0),
        m_primitiveType(Triangles),
        m_boundingRectangle(0.0F, 0.0F, 0.0F, 0.0F),
        m_isBoundingRectangleValid(false)
    {

    }


    VertexBuffer2D::~VertexBuffer2D() {
        static_cast<void>(Unload(Force));
        DisposeVertexGeometry2D();
        m_geometryHandle = 0;
    }

    VertexBuffer2D::SharedPointer VertexBuffer2D::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)
        VertexBuffer2D* ptr = FEATSTD_NEW(VertexBuffer2D)();
        if (ptr == 0) {
            FEATSTD_LOG_ERROR("VertexBuffer2D create failed, out of memory.");
        }
        SharedPointer sharedPointer(ptr);
        return sharedPointer;
    }

    bool VertexBuffer2D::Update(UInt32 index, UInt32 count)
    {
        bool result = false;
        //if m_vertexGeometry is in SystemMemory no update is needed, the values can directly be changed.
        SetBoundingBoxInvalid();
        if ((m_vertexGeometry2D != 0) && (m_vertexGeometry2D->GetMemoryPool() == VertexGeometry2D::VideoMemory) && IsUploaded()) {
            VertexGeometry2D::VertexArray2DResource vertexArrayResource2DObject(m_vertexGeometry2D->GetVertexArray2DResourceHandle());
            const UInt8* data = FeatStd::Internal::PointerToPointer<const UInt8*>(vertexArrayResource2DObject.GetData());
            if (data != 0) {
                Int offset = static_cast<Int>(index * sizeof(VertexGeometry2D::Vertex2D));
                Int size = static_cast<Int>(count * sizeof(VertexGeometry2D::Vertex2D));
                result = RenderDevice2D::UpdateVertexBuffer2D(*this, offset, size, data);
            }
        }

        return result;
    }

    bool VertexBuffer2D::Update(Int offset, Int size, const UInt8* data)
    {
        bool result = false;
        SetBoundingBoxInvalid();
        if ((m_vertexGeometry2D != 0) && (m_vertexGeometry2D->GetMemoryPool() == VertexGeometry2D::VideoMemory)
            && IsUploaded() && (data != 0)) {
            result = RenderDevice2D::UpdateVertexBuffer2D(*this, offset, size, data);
        }
        return result;
    }


    bool VertexBuffer2D::SetVertexGeometry2D(VertexGeometry2D* vertexGeometry2D, VertexGeometry2DDisposerFn disposerFn)
    {
        if (IsUploaded()) {
            return false; // vertexBuffer data is already uploaded
        }
        SetBoundingBoxInvalid();
        DisposeVertexGeometry2D();
        m_vertexGeometry2D = vertexGeometry2D;
        m_vertexGeometry2DDisposerFn = disposerFn;
        return true;
    }

    VertexBuffer2D::SharedPointer VertexBuffer2D::Clone() const
    {
        VertexBuffer2D::SharedPointer clone = VertexBuffer2D::Create();

        if (!clone.PointsToNull()) {
            clone->SetName(this->GetName());
            clone->SetPrimitiveType(this->GetPrimitiveType());
            const VertexGeometry2D* geometry = this->GetVertexGeometry2D();
            if (0 != geometry) {
                VertexGeometry2D* clonedGeometry = geometry->Clone();
                static_cast<void>(clone->SetVertexGeometry2D(clonedGeometry, VertexGeometry2DDisposer::Dispose));
            }
            //Force new calculation of bounding box for clone.
            clone->m_boundingRectangle = Rectangle(0.0F, 0.0F, 0.0F, 0.0F);
            clone->SetBoundingBoxInvalid();
        }

        return clone;
    }

    bool VertexBuffer2D::IsPrimitiveTypeSupported(PrimitiveType type) const
    {
        return RenderDevice2D::IsPrimitiveTypeSupported(type);
    }

    bool VertexBuffer2D::IsIndexBufferSupported() const
    {
        return RenderDevice2D::IsIndexBufferSupported();
    }

    bool VertexBuffer2D::UploadInternal(LoadingHint loadingHint)
    {
        //When implementing VideoMemoryStatistic, lock VertexBuffer3D statistics here.
        bool success = Renderer2D::UploadVertexBuffer2D(*this, loadingHint);
        //When implementing VideoMemoryStatistic, unlock VertexBuffer3D statistics here.

        if (success) {
            //Count VideoMemoryStartics here.
        }

        return success;
    }

    bool VertexBuffer2D::UnloadInternal(LoadingHint loadingHint)
    {
        bool success = Renderer2D::UnloadVertexBuffer2D(*this, loadingHint);
        //Count VideoMemoryStatistic here.
        return success;
    }

    void VertexBuffer2D::DisposeInternal()
    {
        if ((m_vertexGeometry2D != 0) && (m_vertexGeometry2D->GetMemoryPool() == VertexGeometry2D::VideoMemory)) {
            m_vertexGeometry2D->DisposeVertex2DData();
        }
    }


    void VertexBuffer2D::DisposeVertexGeometry2D()
    {
        if ((m_vertexGeometry2DDisposerFn != 0) && (m_vertexGeometry2D != 0)) {
            m_vertexGeometry2DDisposerFn(m_vertexGeometry2D);
        }
        m_vertexGeometry2D = 0;
        m_vertexGeometry2DDisposerFn = 0;
    }

    const Rectangle& VertexBuffer2D::GetBoundingRectangle()
    {
        if (!m_isBoundingRectangleValid) {
            m_boundingRectangle = Rectangle(0.0F, 0.0F, 0.0F, 0.0F);

                const VertexGeometry2D* vg2D = GetVertexGeometry2D();
                if (0 != vg2D) {
                    const ResourceDataHandle vertexArray2DResourceHandle = vg2D->GetVertexArray2DResourceHandle();
                    VertexGeometry2D::VertexArray2DResource vertexArray2DResource(vertexArray2DResourceHandle);
                    const VertexGeometry2D::Vertex2D* data = vertexArray2DResource.GetData();

                    if ((0 != data) && (vg2D->GetVertexCount() > 0)) {
                        Float left = data[0].Position.GetX();
                        Float right = data[0].Position.GetX();
                        Float top = data[0].Position.GetY();
                        Float bottom = data[0].Position.GetY();

                        for (UInt32 i = 0; i < vg2D->GetVertexCount(); ++i) {
                            if (data[i].Position.GetX() < left) { left = data[i].Position.GetX(); }
                            if (data[i].Position.GetX() > right) { right = data[i].Position.GetX(); }
                            if (data[i].Position.GetY() < top) { top = data[i].Position.GetY(); }
                            if (data[i].Position.GetY() > bottom) { bottom = data[i].Position.GetY(); }
                        }

                        m_boundingRectangle = Rectangle(left, top, (right - left), (bottom - top));
                    }
                }
            m_isBoundingRectangleValid = true;
        }

        return m_boundingRectangle;
    }
}
