//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Transformable2D.h"
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(Transformable2D, CanderaObject)

    class Transformable2DDynamicProperties
    {
    private:
        friend class Transformable2D;

        static bool SetRotation(Transformable2D& transformable, Float rotation)
        {
            return transformable.SetValue(CdaDynamicPropertyInstance(Rotation), rotation);
        }

        static Float GetRotation(const Transformable2D& transformable)
        {
            return transformable.GetValue(CdaDynamicPropertyInstance(Rotation));
        }

        static bool SetScale(Transformable2D& transformable, const Vector2& scale)
        {
            return transformable.SetValue(CdaDynamicPropertyInstance(Scale), scale);
        }

        static const Vector2& GetScale(const Transformable2D& transformable)
        {
            return transformable.GetValue(CdaDynamicPropertyInstance(Scale));
        }

        static bool SetPivotPoint(Transformable2D& transformable, const Vector2& pivot)
        {
            return transformable.SetValue(CdaDynamicPropertyInstance(PivotPoint), pivot);
        }

        static const Vector2& GetPivotPoint(const Transformable2D& transformable)
        {
            return transformable.GetValue(CdaDynamicPropertyInstance(PivotPoint));
        }

        static bool SetPivotOffset(Transformable2D& transformable, const Vector2& pivot)
        {
            return transformable.SetValue(CdaDynamicPropertyInstance(PivotOffset), pivot);
        }

        static const Vector2& GetPivotOffset(const Transformable2D& transformable)
        {
            return transformable.GetValue(CdaDynamicPropertyInstance(PivotOffset));
        }

        static Vector2 DefaultScaleValue()
        {
            return Vector2(1.0F, 1.0F);
        }

        static Vector2& GetDefaultScale()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(Vector2, s_defaultScale, DefaultScaleValue());
            return s_defaultScale;
        }

        static Vector2& GetDefaultPivotPoint()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(Vector2, s_defaultPivotPoint);
            return s_defaultPivotPoint;
        }

        static Vector2& GetDefaultPivotOffset()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(Vector2, s_defaultPivotOffset);
            return s_defaultPivotOffset;
        }

        CdaDynamicPropertiesDefinition(Candera::Transformable2D, Candera::Transformable2D::Base);

            CdaDynamicProperty(Rotation, Float);
                CdaDynamicPropertyDefaultValue(0.0F);
                CdaDynamicPropertyDescription("The local rotation of this node.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(Scale, Vector2);
                CdaDynamicPropertyDefaultValue(GetDefaultScale());
                CdaDynamicPropertyDescription("The local scale factor of this node.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(PivotPoint, Vector2);
                CdaDynamicPropertyDefaultValue(GetDefaultPivotPoint());
                CdaDynamicPropertyDescription("The pivot point of this Node."
                                              "The pivot point is the point that is used for rotation and scale transformation of this node.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(PivotOffset, Vector2);
                CdaDynamicPropertyDefaultValue(GetDefaultPivotOffset());
                CdaDynamicPropertyDescription("The pivot offset of this node. The pivot offset is the local "
                                              "space translation applied to the object before applying the other transformations: "
                                              "scale, rotation and translation.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();

        CdaDynamicPropertiesEnd();
    };

    const Candera::DynamicProperties::PropertyHierarchyNode&  Transformable2D::GetPropertyHierarchy() const
    {
        return Transformable2DDynamicProperties::GetPropertyHierarchy();
    }

    Float Transformable2D::GetRotation() const
    {
        return Transformable2DDynamicProperties::GetRotation(*this);
    }

    const Vector2& Transformable2D::GetScale() const
    {
        return Transformable2DDynamicProperties::GetScale(*this);
    }

    const Vector2& Transformable2D::GetPivotPoint() const
    {
        return Transformable2DDynamicProperties::GetPivotPoint(*this);
    }

    const Vector2& Transformable2D::GetPivotOffset() const
    {
        return Transformable2DDynamicProperties::GetPivotOffset(*this);
    }

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    Transformable2D::Transformable2D() :
        Base(),
        m_isCompositeTransformCacheValid(false),
        m_position(0.0F, 0.0F)
    {
    }

    Transformable2D::Transformable2D(const Transformable2D& transformable) :
        Base(transformable),
        m_isCompositeTransformCacheValid(false),
        m_position(transformable.m_position)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    Transformable2D::~Transformable2D()
    {
    }

    /******************************************************************************
     *  SetPosition
     ******************************************************************************/
    void Transformable2D::SetPosition(Float x, Float y)
    {
        m_position.SetX(x);
        m_position.SetY(y);
        InvalidateCompositeCacheAndNotifyDerivedClasses();
    }

    void Transformable2D::SetPosition(const Vector2& position)
    {
        m_position = position;
        InvalidateCompositeCacheAndNotifyDerivedClasses();
    }

    /******************************************************************************
     *  SetRotation
     ******************************************************************************/
    void Transformable2D::SetRotation(Float rotation)
    {
        if (Transformable2DDynamicProperties::SetRotation(*this, rotation)) {
            InvalidateCompositeCacheAndNotifyDerivedClasses();
        }
    }

    /******************************************************************************
     *  SetScale
     ******************************************************************************/
    void Transformable2D::SetScale(const Vector2& scale)
    {
        if (Transformable2DDynamicProperties::SetScale(*this, scale)) {
            InvalidateCompositeCacheAndNotifyDerivedClasses();
        }
    }


    /******************************************************************************
    *  SetPivotPoint
    ******************************************************************************/
    void Transformable2D::SetPivotPoint(const Vector2& pivot)
    {
        if (Transformable2DDynamicProperties::SetPivotPoint(*this, pivot)) {
            InvalidateCompositeCacheAndNotifyDerivedClasses();
        }
    }

    /******************************************************************************
     *  SetPivotOffset
     ******************************************************************************/
    void Transformable2D::SetPivotOffset(const Vector2& pivot)
    {
        if (Transformable2DDynamicProperties::SetPivotOffset(*this, pivot)) {
            InvalidateCompositeCacheAndNotifyDerivedClasses();
        }
    }

    void Transformable2D::SetPivotOffset(Float xPivot, Float yPivot)
    {
        FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
        SetPivotOffset(Vector2(xPivot, yPivot));
        FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
    }

    /******************************************************************************
     *  Translate
     ******************************************************************************/
    void Transformable2D::Translate(const Vector2& translation)
    {
        m_position += translation;
        InvalidateCompositeCacheAndNotifyDerivedClasses();
    }

    /******************************************************************************
     *  Scale
     ******************************************************************************/
    void Transformable2D::Scale(const Vector2& scale)
    {
        SetScale(GetScale() * scale);
    }

    /******************************************************************************
     *  Rotate
     ******************************************************************************/
    void Transformable2D::Rotate(const Float rotation)
    {
        SetRotation(GetRotation() + rotation);
    }

    /******************************************************************************
     *  TranslatePivot
     ******************************************************************************/
    void Transformable2D::TranslatePivot(const Vector2& translation)
    {
        const Matrix3x2& composite = GetCompositeTransform();
        Float d = (composite(0, 0) * composite(1, 1)) - (composite(0, 1) * composite(1, 0));

        Vector2 pivotOffset(
            (- composite(1, 1) * translation.GetX()) + (composite(1, 0) * translation.GetY()),
            (+ composite(0, 1) * translation.GetX()) - (composite(0, 0) * translation.GetY()));

        pivotOffset /= d;

        m_position += translation;

        FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
        SetPivotOffset(GetPivotOffset() + pivotOffset);
        FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
    }

    /******************************************************************************
    *  TranslatePivotPoint
    ******************************************************************************/
    void Transformable2D::TranslatePivotPoint(const Vector2& translation)
    {
        SetPivotPoint(GetPivotPoint() + translation);
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(534, "Ignoring return value of function.")
        GetCompositeTransform();
    }

    /******************************************************************************
     *  GetCompositeTransform
     ******************************************************************************/
    const Matrix3x2& Transformable2D::GetCompositeTransform() const
    {
        // Compute composite transform and update composite transformation cache only, if not up-to-date.
        if (!m_isCompositeTransformCacheValid) {
            // Calculate composite transformation matrix, which is:
            // PO * invPP * S * R * PP * T (PivotOffset * inversePivotPoint * Scale * Rotation * PivotPoint * Translation).
            // NOTE: pure translations like PivotOffset and PivotPoint are commutative until scale/rotation comes in, therefore
            // the formula has several valid forms (e.g. invPP*PO*S*R*T*RR is the same as the one above).
            // Instead of creating and multiplying according matrices with PO*-PP*S*R*PP*T formula with performance expensive 3x3 matrix multiplications,
            // this is done by setting the transformation values in a composite matrix, directly. See below a simple sketch in which dedicated
            // rows and columns the transform values are represented in matrices:
            //   PivotOffset (PO):   inversePivotPoint:    Scale (S):        Rotation (R):     PivotPoint (PP):    Translation (T):
            //   ---------------     -----------------     -------------     -------------     ---------------     -------------
            //   |  1 |  0 | 0 |     |   1 |   0 | 0 |     | S | 0 | 0 |     | R | R | 0 |     |  1 |  0 | 0 |     | 1 | 0 | 0 |
            //   |  0 |  1 | 0 |  *  |   0 |   1 | 0 |  *  | 0 | S | 0 |  *  | R | R | 0 |  *  |  0 |  1 | 0 |  *  | 0 | 1 | 0 |  =  CompositeTransform
            //   | PO | PO | 1 |     | -PP | -PP | 1 |     | 0 | 0 | 1 |     | 0 | 0 | 1 |     | PP | PP | 1 |     | T | T | 1 |
            //   ---------------     -----------------     -------------     -------------     ---------------     -------------
            //    in fact the last column is not used!
            //

            // First step: Init the composite matrix with the rotation matrix.
            m_compositeTransformCache.SetRotation(GetRotation());

            // Second step: Multiply rotation values with scaling values.
            const Vector2& scale = GetScale();
            m_compositeTransformCache.Scale(scale.GetX(), scale.GetY());

            // Third step: Set translation in composite transform matrix.
            const Vector2& pivotPoint = GetPivotPoint();
            m_compositeTransformCache(2, 0) = m_position.GetX() + pivotPoint.GetX();
            m_compositeTransformCache(2, 1) = m_position.GetY() + pivotPoint.GetY();

            // Fourth step: Apply pivot.
            FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
            const Vector2& pivotOffset = GetPivotOffset();
            FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
            if ((pivotOffset.GetX() != 0.0F) || (pivotPoint.GetX() != 0.0F)) {
                m_compositeTransformCache(2, 0) += (pivotOffset.GetX() - pivotPoint.GetX()) * m_compositeTransformCache(0, 0);
                m_compositeTransformCache(2, 1) += (pivotOffset.GetX() - pivotPoint.GetX()) * m_compositeTransformCache(0, 1);
            }

            if ((pivotOffset.GetY() != 0.0F) || (pivotPoint.GetY() != 0.0F)) {
                m_compositeTransformCache(2, 0) += (pivotOffset.GetY() - pivotPoint.GetY()) * m_compositeTransformCache(1, 0);
                m_compositeTransformCache(2, 1) += (pivotOffset.GetY() - pivotPoint.GetY()) * m_compositeTransformCache(1, 1);
            }

            // Reset composite transform cache to valid state.
            m_isCompositeTransformCacheValid = true;
        }
        return m_compositeTransformCache;

    }

    /******************************************************************************
     *  OnCompositeTransformChanged
     ******************************************************************************/
    void Transformable2D::OnCompositeTransformChanged()
    {
        // nothing to do
    }

    /******************************************************************************
     *  InvalidateCompositeCacheAndNotifyDerivedClasses
     ******************************************************************************/
    void Transformable2D::InvalidateCompositeCacheAndNotifyDerivedClasses()
    {
        // There is no need to traverse the whole sub tree if the cache is invalid.
        // Which means that the tree has already been traversed. New nodes that are added
        // to the tree are invalidated automatically.
        if (m_isCompositeTransformCacheValid) {
            m_isCompositeTransformCacheValid = false;
            OnCompositeTransformChanged();
        }
    }

}   // namespace Candera
