//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextNode2DLayouter.h"

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/TextEngine/LayoutingOptions.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/DefaultTextNode2DLayouter.h>
#include <Candera/Engine2D/Core/TextNode2D.h>

#if defined(CANDERA_GLOBALIZATION_ENABLED)
#include <CanderaGlobalization/CultureManager.h>
#endif

namespace Candera {
    using namespace Candera::Internal;
    using namespace TextRendering;
#if defined(CANDERA_GLOBALIZATION_ENABLED)
    using namespace Globalization;
#endif

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);

#if defined(CANDERA_LAYOUT_ENABLED)
    static TextSize Vector2ToTextSize(const Vector2& size)
    {
        return TextSize(static_cast<Int16>(Math::Minimum(size.GetX(), static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()))),
                        static_cast<Int16>(Math::Minimum(size.GetY(), static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()))));
    }

    static TextCoordinate Vector2ToTextCoordinate(const Vector2& coordinate)
    {
        return TextCoordinate(static_cast<PixelPosition>(Math::Minimum(coordinate.GetX(), static_cast<Float>(FeatStd::Internal::Limits<PixelPosition>::Max()))),
                              static_cast<PixelPosition>(Math::Minimum(coordinate.GetY(), static_cast<Float>(FeatStd::Internal::Limits<PixelPosition>::Max()))));
    }

    static Vector2 TextSizeToVector2(const TextSize& size)
    {
        return Vector2(static_cast<Float>(size.GetWidth()), static_cast<Float>(size.GetHeight()));
    }

    static Vector2 TextCoordinateToVector2(const TextCoordinate& coordinate)
    {
        return Vector2(static_cast<Float>(coordinate.GetX()), static_cast<Float>(coordinate.GetY()));
    }

    static TextRect RectangleToTextRect(const Rectangle& rectangle)
    {
        return TextRect(Vector2ToTextCoordinate(rectangle.GetPosition()), Vector2ToTextSize(rectangle.GetSize()));
    }

    static Rectangle TextRectToRectangle(const TextRect& rectangle)
    {
        return Rectangle(TextCoordinateToVector2(rectangle.GetPosition()), TextSizeToVector2(rectangle.GetSize()));
    }

    bool TextNode2DLayouter::PreprocessTextInternal(TextRendering::TextRenderArgs::SharedPointer renderArgs)
    {
        return TextNode2D::PreprocessTextInternal(renderArgs);
    }

    bool TextNode2DLayouter::IsLayoutValid(TextNode2D& node) const
    {
        node.CheckPropertiesValidity();
        return (node.GetPreparationState() == TextNode2DRenderState::Invalid);
    }

    void TextNode2DLayouter::ValidateLayout(TextNode2D& node) const
    {
        node.SetPreparationFinished();
    }

    TextNode2DLayouter& TextNode2DLayouter::GetDefault()
    {
        return DefaultTextNode2DLayouter::GetInstance();
    }

    Vector2 TextNode2DLayouter::OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea)
    {
        if (node.IsValid()) {
            TextNode2D* textNode = Dynamic_Cast<TextNode2D*>(node.ToNode2D());
            if (textNode == 0) {
                return Vector2(-1.0F, -1.0F);
            }

            const Vector2& setSize = Layouter::GetSize(*node.ToCanderaObject());

            Vector2 measuredSize = TextSizeToVector2(OnMeasureText(*textNode, Vector2ToTextSize(clientArea)));
            if (setSize.GetX() > -1.F) {
                measuredSize.SetX(setSize.GetX());
            }

            if (setSize.GetY() > -1.F) {
                measuredSize.SetY(setSize.GetY());
            }
            return measuredSize;
        }
        return Vector2();
    }

    void TextNode2DLayouter::OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea)
    {
        if (node.IsValid()) {
            TextNode2D* textNode = Dynamic_Cast<TextNode2D*>(node.ToNode2D());
            if (textNode == 0) {
                SetNodePosition(node, clientArea.GetPosition());
            }
            else {
                static_cast <void> (OnArrangeText(*textNode, RectangleToTextRect(clientArea)));

                Rectangle boundingRect;
                GetParentSpaceAxisAlignedBoundingRect(node, boundingRect);

                Rectangle rect(0.0F, 0.0F, boundingRect.GetWidth(), boundingRect.GetHeight());
                AlignRectangle(clientArea, rect, node);

                SetNodePosition(node, rect.GetPosition());
            }

            Vector2 actualSize = GetPreferredSize(node);
            if (GetHorizontalAlignment(*node.ToCanderaObject()) == HStretch) {
                actualSize.SetX(clientArea.GetWidth());
            }
            SetArrangeActualSize(actualSize);
        }
    }

    Rectangle TextNode2DLayouter::GetComputedLayoutRectangle(const Node2D& node) const
    {
        const TextNode2D* textNode = Dynamic_Cast<const TextNode2D*>(&node);
        Rectangle rect;
        if (textNode == 0) {
            node.GetComputedLayoutRectangle(rect);
        }
        else {
            rect = TextRectToRectangle(textNode->GetLayoutTextRectangle());
        }
        return rect;
    }

    LayoutingOptions TextNode2DLayouter::GetLayoutingOptions(const TextNode2D& textNode)
    {
        return LayoutingOptions(GetLanguageSensitiveTextAlignment(textNode),
                                VTop,
                                TextCoordinate(),
                                Vector2ToTextSize(GetSize(static_cast<const CanderaObject&>(textNode))),
                                IsMultiLineEnabled(textNode),
                                IsWordWrapEnabled(textNode),
                                GetLineSpacing(textNode));
    }

    void TextNode2DLayouter::SetWordWrapEnabled(TextNode2D& node, bool enabled)
    {
        node.InvalidateText();
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(WordWrapEnabled), enabled));
    }

    void TextNode2DLayouter::SetMultiLineEnabled(TextNode2D& node, bool enabled)
    {
        node.InvalidateText();
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(MultiLineEnabled), enabled));
    }

    TextNode2DLayouter::TextAlignment TextNode2DLayouter::GetTextAlignment(const TextNode2D& node)
    {
        return node.GetValue(CdaDynamicPropertyInstance(TextAlignment));
    }

    void TextNode2DLayouter::SetTextAlignment(TextNode2D& node, TextAlignment textAlignment)
    {
        node.InvalidateText();
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(TextAlignment), textAlignment));
    }

    TextNode2DLayouter::Trimming TextNode2DLayouter::GetTrimming(const TextNode2D& node)
    {
        return node.GetValue(CdaDynamicPropertyInstance(Trimming));
    }

    void TextNode2DLayouter::SetTrimming(TextNode2D& node, Trimming trimming)
    {
        node.InvalidateText();
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(Trimming), trimming));
    }

    TextRendering::PixelSize TextNode2DLayouter::GetLineSpacing(const TextNode2D& node)
    {
        return node.GetValue(CdaDynamicPropertyInstance(LineSpacing));
    }

    void TextNode2DLayouter::SetLineSpacing(TextNode2D& node, TextRendering::PixelSize lineSpacing)
    {
        node.InvalidateText();
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(LineSpacing), lineSpacing));
    }

    void TextNode2DLayouter::SetTrimmingText(TextNode2D& node, FeatStd::String const& text)
    {
        node.InvalidateText();
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(TrimmingText), text));
    }

    FeatStd::String TextNode2DLayouter::GetTrimmingText(const TextNode2D& textNode)
    {
        return textNode.GetValue(CdaDynamicPropertyInstance(TrimmingText));
    }

    HorizontalAlignment TextNode2DLayouter::GetLanguageSensitiveTextAlignment(const TextNode2D& textNode)
    {
        TextAlignment hTextAlign = GetTextAlignment(textNode);
        HorizontalAlignment result = HCenter;
        switch (hTextAlign) {
        case Auto:
            result = GetHorizontalAlignment(static_cast<const CanderaObject&>(textNode)); break;
        case Left:
            result = HLeft; break;
        case Center:
            result = HCenter; break;
        case Right:
            result = HRight; break;
        case Justified:
            result = HStretch; break;
        default:
            break;
        }
        AbstractNodePointer abstractNode(const_cast<TextNode2D*>(&textNode));
        if (GetLanguageSensitiveDirection(abstractNode) == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
            if (result == HLeft) {
                result = HRight;
            }
            else {
                if (result == HRight) {
                    result = HLeft;
                }
            }
        }

        return result;
    }

#endif
}   // namespace Candera
