//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SURFACE_TEXT_RENDER_CONTEXT_H)
    #define CANDERA_SURFACE_TEXT_RENDER_CONTEXT_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <Candera/TextEngine/GlyphCacheAccess.h>
#include <Candera/TextEngine/TextRenderContext.h>
#include <Candera/TextEngine/TextRect.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Core/ProxyImage2D.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRendererTools.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/GlyphImageSource2D.h>

namespace Candera {

    class TextNode2D;

    /**
    *  @brief  SurfaceGlyphCacheAccess class
    *          @see Candera::TextRendering::GlyphCacheAccess
    */
    FEATSTD_SUPPRESS_MULTI_WARNING_BEGIN(611, "not overriden because it's deprecated")
    class SurfaceGlyphCacheAccess2 : public TextRendering::GlyphCacheAccess
    {
        FEATSTD_TYPEDEF_BASE(TextRendering::GlyphCacheAccess);
    public:
        static SurfaceGlyphCacheAccess2& Instance();

        virtual ~SurfaceGlyphCacheAccess2() override;

        // overrides TextRendering::GlyphCacheAccess::Create
        virtual UInt8* Create(const TextRendering::GlyphBitmap &bitmap, UInt8* cacheItem) override final;

        // overrides TextRendering::GlyphCacheAccess::DestroyCacheItem
        virtual void DestroyCacheItem(TextRendering::PixelBuffer buffer) override final;

        // overrides TextRendering::GlyphCacheAccess::CheckPixelBuffer
        virtual bool CheckCacheItem(TextRendering::PixelBuffer buffer) override final;

    private:
        FEATSTD_MAKE_CLASS_STATIC(SurfaceGlyphCacheAccess2);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(SurfaceGlyphCacheAccess2);
    };
    FEATSTD_SUPPRESS_MULTI_WARNING_END()

    inline SurfaceGlyphCacheAccess2::SurfaceGlyphCacheAccess2() {}
    inline SurfaceGlyphCacheAccess2::~SurfaceGlyphCacheAccess2() { ReleaseCache(); }
    inline SurfaceGlyphCacheAccess2& SurfaceGlyphCacheAccess2::Instance()
    {
        static SurfaceGlyphCacheAccess2 instance;
        return instance;
    }

    class TextNodeTextRenderContext : public TextRendering::TextRenderContext
    {
    public:
        virtual bool IsCachingEnabled() const override final { return false; }
        virtual TextRendering::GlyphCacheAccess* GetGlyphCacheAccess() const override final { return 0; }
        virtual void Blit(Int16 x, Int16 y, const TextRendering::GlyphBitmap &glyph) override final;
    private:
        virtual void BlitImage(Int16 x, Int16 y, ImageSource2D& glyph) = 0;
    };

    class GlyphCacheTextNodeTextRenderContext : public TextRendering::TextRenderContext
    {
    public:
        virtual bool IsCachingEnabled() const override final { return true; }
        virtual TextRendering::GlyphCacheAccess* GetGlyphCacheAccess() const override final { return &GetSurfaceGlyphCacheAccess(); }
        virtual void Blit(Int16 x, Int16 y, const TextRendering::GlyphBitmap &glyph) override final;
    private:
        static SurfaceGlyphCacheAccess2& GetSurfaceGlyphCacheAccess();
        virtual void BlitImage(Int16 x, Int16 y, ImageSource2D& glyph) = 0;
    };

    /**
     *  @brief  SurfaceTextNodeTextRenderContext class
     *          @see Candera::TextRendering::TextRenderContext
     */
    template <typename Base = TextNodeTextRenderContext>
    class GlyphByGlyphTextNodeTextRenderContext : public Base
    {
    public:
        /**
         *  Constructor
         */
        GlyphByGlyphTextNodeTextRenderContext(::Candera::TextNode2D& textNode, ::Candera::RenderTarget2D* renderTarget, const ::Candera::Matrix3x2& localTransform) :
            m_image2D(ProxyImage2D::Create()),
            m_textNode(textNode),
            m_renderTarget(renderTarget),
            m_localTransform(localTransform)
        {
            m_image2D->SetSyncEnabled(false);
        }
        const ::Candera::TextRendering::TextRect &GetClipRect(void) const override final
        {
            static ::Candera::TextRendering::TextRect m_clip;
            if (m_renderTarget->Get2DContextHandle() != 0) {
                Float left;
                Float top;
                Float width;
                Float height;
                if (RenderDevice2D::GetActiveArea(m_renderTarget->Get2DContextHandle(), RenderDevice2D::DestinationSurface, &left, &top, &width, &height)) {

                    Matrix3x2 trans(m_localTransform);
                    trans.Inverse();

                    Rectangle viewport(0.0F, 0.0F, width, height);
                    viewport.Transform(trans);

                    m_clip = ::Candera::TextRendering::TextRect(
                        TextRendering::TextCoordinate(
                        static_cast<Int16>(Math::Floor(viewport.GetLeft())),
                        static_cast<Int16>(Math::Floor(viewport.GetTop()))),
                        TextRendering::TextSize(
                        static_cast<Int16>(Math::Ceil(viewport.GetWidth())),
                        static_cast<Int16>(Math::Ceil(viewport.GetHeight()))));
                }
            }
            return m_clip;
        }
    private:
        ::FeatStd::MemoryManagement::SharedPointer< ::Candera::ProxyImage2D> m_image2D;
        ::Candera::TextNode2D& m_textNode;
        ::Candera::RenderTarget2D* m_renderTarget;
        const ::Candera::Matrix3x2& m_localTransform;

        virtual void BlitImage(::Candera::Int16 x, ::Candera::Int16 y, ::Candera::ImageSource2D &glyph) override final
        {
            CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "Blit"));

            if (m_image2D == 0) {
                return;
            }

            if (glyph.GetSurfaceHandle() == 0) {
                return;
            }

            m_image2D->SetImageSource(&glyph);
            Candera::Internal::TextNodeRendererTools::RenderImage(m_image2D.GetSharedInstance(), m_textNode, m_renderTarget, m_localTransform, Vector2(x, y));
            m_image2D->SetImageSource(0);
        }

        FEATSTD_MAKE_CLASS_UNCOPYABLE(GlyphByGlyphTextNodeTextRenderContext);
    };

    /**
    *  @brief  SurfaceTextNodeTextRenderContext class
    *          @see Candera::TextRendering::TextRenderContext
    */
    template <typename Base = GlyphCacheTextNodeTextRenderContext>
    class SurfaceTextNodeTextRenderContext : public Base
    {
    public:
        /**
        *  Constructor
        */
        SurfaceTextNodeTextRenderContext():m_context(0) {}

        /**
        *  Sets the context2D handle.
        *  @param context2D The context2D handle that is set.
        */
        void Set2DContext(::Candera::Handle context2D) { m_context = context2D; }

        /**
        *  Retrieves the current clip rectangle.
        *  @return The current text rectangle.
        */
        virtual const ::Candera::TextRendering::TextRect& GetClipRect() const { return m_clip; }

        /** Define clipping rectangle for text rendering
        *  @param rect new clipping rectangle
        */
        void SetClipRect(const ::Candera::TextRendering::TextRect &rect) { m_clip = rect; }

        /**
        *  Resets the updated area.
        */
        void ResetUpdatedArea() { m_updatedArea = Rectangle(); }

        /**
        *  Retrieves the updated area, this returned property
        *  can be changed. The performed changes will affect this class.
        *  @return The updated area as a rectangle.
        */
        const ::Candera::Rectangle& GetUpdatedArea() const { return m_updatedArea; }
    private:
        ::Candera::Handle m_context;
        ::Candera::Rectangle m_updatedArea;

        mutable ::Candera::TextRendering::TextRect m_clip;

        virtual void BlitImage(::Candera::Int16 x, ::Candera::Int16 y, ::Candera::ImageSource2D &glyph) override final
        {
            CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "Blit"));
            if (m_context == 0) {
                return;
            }

            if (glyph.GetSurfaceHandle() == 0) {
                return;
            }

            Matrix3x2 pos;
            pos.Translate(static_cast<Float>(x), static_cast<Float>(y));

            bool result = Renderer2D::SetTransformationMatrix(m_context, RenderDevice2D::SourceSurface, pos);
            result = result && RenderDevice2D::SetSurface(m_context, RenderDevice2D::SourceSurface, glyph.GetSurfaceHandle());
            result = result && Renderer2D::Blit(m_context, true);

            if (result) {
                Float left;
                Float top;
                Float width;
                Float height;
                if (RenderDevice2D::GetUpdatedArea(m_context, &left, &top, &width, &height)) {
                    m_updatedArea.Union(Rectangle(left, top, width, height));
                }
            }
        }

    };

}   // namespace Candera

#endif  // CANDERA_SURFACE_TEXT_RENDER_CONTEXT_H
