//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DefaultTextNode2DLayouter.h"

#include <FeatStd/Util/StaticObject.h>
#include <Candera/TextEngine/AlignTextMeasureContext.h>
#include <Candera/TextEngine/Async/TextRenderDispatcher.h>
#include <Candera/TextEngine/Async/TextRenderArgs.h>
#include <Candera/TextEngine/CursorTextMeasureContext.h>
#include <Candera/TextEngine/Internal/StyleTools.h>
#include <Candera/TextEngine/LayoutingOptions.h>
#include <Candera/TextEngine/ShapingOptions.h>
#include <Candera/TextEngine/TextProperties.h>
#include <Candera/TextEngine/TextRenderContexts/LineBreakLayoutStrategy.h>
#include <Candera/TextEngine/TextRenderContexts/NestedTextLayoutStrategy.h>
#include <Candera/TextEngine/TextRenderContexts/TextToGlyphIteratorContext.h>
#include <Candera/TextEngine/TextRenderContexts/TruncationToGlyphIteratorContext.h>
#include <Candera/TextEngine/TextRenderer.h>
#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRenderer.h>
#include <Candera/System/Diagnostics/Log.h>


namespace Candera {
#if defined(CANDERA_LAYOUT_ENABLED)
    using namespace TextRendering;
    using namespace TextRendering::Internal;

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine2D);

    static const Char* c_ellipsisText = "...";

    const TextSize& DefaultTextNode2DLayouter::GetDefaultTextSize()
    {
        static TextSize size(-1, -1);
        return size;
    }
    static TextSize Vector2ToTextSize(const Vector2& size)
    {
        return TextSize(static_cast<Int16>(Math::Minimum(size.GetX(), static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()))),
                        static_cast<Int16>(Math::Minimum(size.GetY(), static_cast<Float>(FeatStd::Internal::Limits<Int16>::Max()))));
    }

    static TextSize GetMinArea(const TextSize& clientArea, const TextSize& textMaxArea)
    {
        TextSize result(0, 0);
        //set width
        if (textMaxArea.GetWidth() <= 0) {
            result.SetWidth(clientArea.GetWidth());
        }
        else {
            result.SetWidth(static_cast<Int16>(Math::Minimum(clientArea.GetWidth(), textMaxArea.GetWidth())));
        }

        //set height
        if (textMaxArea.GetHeight() <= 0) {
            result.SetHeight(clientArea.GetHeight());
        }
        else {
            result.SetHeight(static_cast<Int16>(Math::Minimum(clientArea.GetHeight(), textMaxArea.GetHeight())));
        }

        return result;
    }

    bool DefaultTextNode2DLayouter::MeasureText(TextNode2D& textNode, const TextSize& clientArea, TextSize& layoutSize) const
    {
        bool returnValue = true;
        FEATSTD_UNUSED(layoutSize);
        const TextRenderContext* referenceContext = (textNode.GetTextNodeRenderer() != 0)?textNode.GetTextNodeRenderer()->GetMeasureReferenceContext():0;
        TextSize maxClientArea = GetMinArea(clientArea, Vector2ToTextSize(TextNode2DLayouter::GetSize(static_cast<CanderaObject&>(textNode))));
        SharedStyle::SharedPointer sharedStyle = textNode.GetStyle();
        if (sharedStyle.PointsToNull()) {
            FEATSTD_LOG_WARN("Tried to render TextNode2D without attached style");
            // State is set to idle. As long as no style has been set, a text cannot be rendered anyway.
            // However, setting a style invalidates the state - and then the next render attempt will be triggered.
            textNode.SetPreparationState(TextNode2DRenderState::Idle);
            return false;
        }
        TruncationToGlyphIteratorContext::SharedPointer truncationContext;
        bool isTrimmingEnabled = (GetTrimming(textNode) == Ellipsis) || (GetTrimming(textNode) == CustomText);

        Style& style = *sharedStyle;
        UInt16 styleVersion = TextRendering::Internal::StyleTools::GetStyleVersion(style);
        if (isTrimmingEnabled) {

            truncationContext = GetTruncateContext(textNode);
            if (styleVersion != GetStyleVersion(style)) {
                if (truncationContext.PointsToNull()) {
                    truncationContext = TruncationToGlyphIteratorContext::Create();
                }
                TextProperties truncationText((GetTrimming(textNode) == Ellipsis)?(c_ellipsisText):(GetTrimmingText(textNode).GetCString()));
                truncationContext->Reinit(textNode.GetStyle(), referenceContext);
                if (TextRenderer().Render(*truncationContext, LayoutingOptions(), ShapingOptions(textNode.GetStyle()), truncationText)) {
                    static_cast<void>(SetTruncateContext(textNode, truncationContext));
                }
                else {
                    // Should not occur. However, there are unknown occurrences of this else.
                    // when this happens - there will not be a truncation rendered - and therefore there will not be a truncation available
                    // or it will be continued to use the old truncation.
                    // FEATSTD_DEBUG_ASSERT(false);
                }
            }
        }

        TextRenderArgs::SharedPointer args;

        UInt8 freeIdx = textNode.GetRenderController().GetFreeArgs(args);
        // should never be reached - there should always be a free slot
        FEATSTD_DEBUG_ASSERT(freeIdx != textNode.GetRenderController().InvalidIndex());
        if (freeIdx == textNode.GetRenderController().InvalidIndex()) {
            return false;
        }
        // retrieve args...
        FEATSTD_DEBUG_ASSERT(!args.PointsToNull());
        static_cast<void>(args->MeasureInit(textNode, maxClientArea, GetLayoutingOptions(textNode), truncationContext));

        //generate result
        TextSize ts = args->GetLayoutingOptionsRef().GetSize();
        if (maxClientArea.GetWidth() > 0) {
            ts.SetWidth(maxClientArea.GetWidth());
        }
        if (maxClientArea.GetHeight() > 0) {
            ts.SetHeight(maxClientArea.GetHeight());
        }
        args->GetLayoutingOptionsRef().SetSize(ts);

        if (GetLanguageSensitiveTextAlignment(textNode) == HStretch) {
            args->GetLayoutingOptionsRef().SetHorizontalAlignment(HStretch);
        }
        // enables croping on correct side - and shifting text in a direction when trimming/truncation is enabled.
        args->GetLayoutingOptionsRef().SetOrientationRealignmentEnabled(isTrimmingEnabled);
        args->GetLayoutingOptionsRef().SetCultureHandlingRequired(false);
        AbstractNodePointer abstractNode(&textNode);
        args->GetLayoutingOptionsRef().SetRightToLeftDirection(LayoutAlignment::LayoutDirection::RightToLeftDirection == Layouter::GetLanguageSensitiveDirection(abstractNode));
        textNode.SetPreparationState(TextNode2DRenderState::Processing);
        if (!textNode.IsAsyncPreRenderEnabled()) {
            returnValue = PreprocessTextInternal(args);
            textNode.GetRenderController().SetFinishedHandle(freeIdx);
        }
        else {
            textNode.GetRenderController().SetHandle(freeIdx, TextRenderDispatcher::RenderAsync(TextNode2DLayouter::PreprocessTextInternal, args, 1));
        }
        return returnValue;
    }


    TextSize DefaultTextNode2DLayouter::OnMeasureText(TextNode2D& node, const TextSize& clientArea)
    {
        TextSize textSize(-1, -1);
        bool renderResult;
        if (!node.IsCustomGlyphInformationUsed()) {

            node.CheckPropertiesValidity(false);
            TextSize prevClientArea = GetPreviousInputClientArea(node);
            if ((clientArea.GetWidth() != prevClientArea.GetWidth()) ||
                (clientArea.GetHeight() != prevClientArea.GetHeight())) {
                node.InvalidateText(false);
                static_cast<void>(SetPreviousInputClientArea(node, clientArea));
            }

            LayoutAlignment::LayoutDirection::Enum prevLayoutDirection = GetPreviousLayoutDirection(node);
            AbstractNodePointer abstractNode(&node);
            LayoutAlignment::LayoutDirection::Enum currLayoutDirection = GetLanguageSensitiveDirection(abstractNode);
            if (currLayoutDirection != prevLayoutDirection) {
                node.InvalidateText(false);
                static_cast<void>(SetPreviousLayoutDirection(node, currLayoutDirection));
            }


            if (node.GetPreparationState() == TextNode2DRenderState::Invalid) {
                renderResult = MeasureText(node, clientArea, textSize);
                if (renderResult) {
                    FEATSTD_DEBUG_ASSERT(node.IsAsyncPreRenderEnabled() || (node.GetPreparationState() == TextNode2DRenderState::Finished));
                }
            }

            //retrieved result -> so its in idle phase again
            if (node.GetPreparationState() == TextNode2DRenderState::Finished) {
                TextRenderArgs::SharedPointer ptr;
                UInt8 freeIdx;
                bool retrieveResult =
                    (!node.IsAsyncPreRenderEnabled()) ||
                    (node.GetRenderController().GetValidationHelper() == 0) ||
                    (node.GetRenderController().GetValidationHelper()->IsValid());

                if (retrieveResult) {
                    freeIdx = node.GetRenderController().GetRenderCompletedIndex(ptr, renderResult);
                    if (freeIdx != node.GetRenderController().InvalidIndex()) {
                        FEATSTD_DEBUG_ASSERT(!ptr.PointsToNull());
                        if (renderResult) {
                            // node specific call
                            node.SetPreprocessedText(ptr);
                            // layout specific calls
                            textSize = ptr->GetLayoutRectangle().GetSize();

                            static_cast<void>(SetClientArea(node, clientArea));
                            static_cast<void>(SetLayoutingSize(node, textSize));
                        }
                        // async specific calls
                        node.GetRenderController().FreeIndex(freeIdx);
                        ValidateLayout(node);
                        textSize = GetLayoutingSize(node);

                        TextCoordinate startPosition(node.GetLayoutTextRectangle().GetLeft(), node.GetLayoutTextRectangle().GetTop());
                        bool shiftContainer = false;
                        if (static_cast<Int>(TextNode2DLayouter::GetSize(static_cast<CanderaObject&>(node)).GetX()) <= 0) {
                            startPosition.SetX(0);
                            shiftContainer = true;
                        }
                        if (static_cast<Int>(TextNode2DLayouter::GetSize(static_cast<CanderaObject&>(node)).GetY()) <= 0) {
                            startPosition.SetY(0);
                            shiftContainer = true;
                        }
                        if (shiftContainer) {
                            node.ArrangePreprocessedTextPosition(TextRect(startPosition, textSize));
                        }
                        if ((node.IsAsyncPreRenderEnabled()) && (node.GetRenderController().GetValidationHelper() != 0)) {
                            node.GetRenderController().GetValidationHelper()->ConfirmValidHandled();
                        }
                    }
                    else {
                        // A validator which brings no result or a sync call that has also no result is useless
                        // The only possibility is a not existing validator
                        //    FEATSTD_DEBUG_ASSERT(node.GetRenderController().GetValidationHelper() == 0);
                    }
                }
            }
        }

        textSize = GetLayoutingSize(node);
        if ((clientArea.GetWidth() >= 0) && (clientArea.GetWidth() < textSize.GetWidth())) {
            textSize.SetWidth(clientArea.GetWidth());
        }

        if ((clientArea.GetHeight() >= 0) && (clientArea.GetHeight() < textSize.GetHeight())) {
            textSize.SetHeight(clientArea.GetHeight());
        }

        return textSize;
    }

    TextRendering::TextRect DefaultTextNode2DLayouter::OnArrangeText(TextNode2D& node, const TextRect& clientArea)
    {
        FEATSTD_UNUSED(node);
        return clientArea;
    }

    DefaultTextNode2DLayouter& DefaultTextNode2DLayouter::GetInstance()
    {
        FEATSTD_SYNCED_STATIC_OBJECT(DefaultTextNode2DLayouter, s_defaultTextNode2DLayouter);
        return s_defaultTextNode2DLayouter;
    }

#endif

}   // namespace Candera
