//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TextNode2D_H)
#define CANDERA_TextNode2D_H

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSection.h>
#endif
#include <FeatStd/Util/String.h>
#include <Candera/TextEngine/Async/TextRenderArgs.h>
#include <Candera/TextEngine/Async/TextRenderHandleController.h>
#include <Candera/TextEngine/Async/TextValidator.h>
#include <Candera/TextEngine/MinimalPreprocessingContext.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/TextRenderContexts/TextToGlyphIteratorContext.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <FeatStd/Event/EventListener.h>


CANDERA_UNIT_TEST_TESTCASE_DECLARATION(TextNode2DTest, AsyncToSyncWithoutLayout)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(TextNode2DTest, AsyncToSyncWithLayout)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(TextNode2DTest, CultureChangeInvalidationCheck)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(TextNode2DTest, StylePropertyChanged)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(TextLayoutTest, Run)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(Globalization, CultureChangeTextNode2DRecognition)

namespace Candera {

    namespace Globalization {
        class CultureChangeListener;
    }

    class TextNode2DLayouter;
    class TextNodeRenderer;
    class Text2DCultureChangeListener;
    namespace Internal {
        class TextNodeRendererTools;
    }

    namespace TextRendering {
        class LayoutingOptions;
        class TextProperties;
        class TruncationToGlyphIteratorContext;
    }

    namespace TextNode2DRenderState {
    /// @addtogroup Core2D
    /// @{
        /**
        *  @brief TextNode2DRenderState enumeration defines the states a text can have
        *  in view of rendering the text.
        */
        enum Enum {
            Idle,       ///< Text is rendered or no text is pending.
            Invalid,    ///< Text (or properties) changed.
            Processing, ///< Text is queued for preprocessing/rendering.
            Finished    ///< Text has been preprocessed/rendered and is waiting to activate the text.
        };
    /// @}
    }

    /// @addtogroup Core2D
    /// @{
    class CustomPreprocessingResult;
    /**
    *  @brief TextNode2DRenderStates has all render states in it.
    * Currently there are layouting (Measurement and preprocessing the text)
    * and render (prerender) as the two different render steps.
    */
    class TextNode2DRenderStates {
    public:
        //layout is invalid because no measurement is done - prerender is idle because its waiting for layout
        TextNode2DRenderStates() :m_layoutState(TextNode2DRenderState::Invalid), m_prerenderState(TextNode2DRenderState::Idle) {}

        void SetLayoutState(TextNode2DRenderState::Enum state)
        {
            m_layoutState = static_cast<UInt8>(state);
        }
        TextNode2DRenderState::Enum GetLayoutState()const
        {
            return static_cast<TextNode2DRenderState::Enum>(m_layoutState);
        }

        void SetPrerenderState(TextNode2DRenderState::Enum state)
        {
            m_prerenderState = static_cast<UInt8>(state);
        }
        TextNode2DRenderState::Enum GetPrerenderState()const
        {
            return static_cast<TextNode2DRenderState::Enum>(m_prerenderState);
        }
    private:
        UInt m_layoutState:4;
        UInt m_prerenderState:4;
    };

    /**
    *  @brief TextNode2D is a RenderNode that renders text using the attached BitmapBrush effect.
    *
    *  To render text a FeatStd::String and a TextRendering::Style needs to be provided, as well as a
    *  TextNodeRenderer object which implements the strategy for providing Image2D objects, which will be
    *  rendered by the associated BitmapBrush.
    *
    * For text layout a TextNode2DLayouter should be attached to the TextNode2D.
    */
    class TextNode2D : public RenderNode, public FeatStd::EventListener, public Candera::TextRendering::ITextValidationUser {
        friend class Candera::Internal::TextNodeRendererTools;
        friend class TextNode2DLayouter;
        friend class TextNodeRenderer;
        friend class DefaultTextNode2DLayouter;
        friend class Candera::TextRendering::TextRenderArgs;
        friend class Candera::Text2DCultureChangeListener;
        FEATSTD_TYPEDEF_BASE(RenderNode);
    public:
        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to the created object
         */
        static TextNode2D* Create();

        /**
         *  Destructor
         */
        virtual ~TextNode2D();

        /**
        *  Shallow clone of this TextNode - creates a new instance with the
        *  same parameters as this instance.
        *  Obeys the rules from Node2D::Clone
        *  @remark    m_renderer is set to 0, use DeepTextNode2DCloneStrategy to clone it.
        *  @return    The cloned TextNodeRenderer if successful, 0 otherwise.
        */
        virtual TextNode2D* Clone() const override;

        /**
         * Set text to be rendered.
         */
        void SetText(const FeatStd::String& text);

        /**
         * Get rendered text.
         */
        const FeatStd::String& GetText() const { return m_text; }

        /**
         * Set style to be used for rendering.
         */
        void SetStyle(const TextRendering::SharedStyle::SharedPointer& style);

        /**
         * Get style used for rendering.
         */
        const TextRendering::SharedStyle::SharedPointer& GetStyle() const { return m_style; }

        /**
        * Get the line height in pixels for the current style.
        * @return line height in pixels for the current style or 0 if no style is set.
        */
        TextRendering::PixelSize GetLineHeight() const {
            if (!m_style.PointsToNull())
            {
                return static_cast<TextRendering::PixelSize>(m_style.GetPointerToSharedInstance()->GetMetrics().lineHeight);
            }
            return 0;
        }

        /**
         * Set TextNodeRenderer, which will generate the images to be rendered.
         * If a TextNodeRenderer has been set before (GetTextNodeRenderer() != 0),
         * the renderer needs to be disposed by the SetTextnodeRenderer-Caller.
         * The destructor calls dispose on the last set renderer.
         * @param renderer The new render type which shall handling the text rendering.
         */
        void SetTextNodeRenderer(TextNodeRenderer* renderer);

        /**
         * Get TextNodeRenderer.
         */
        const TextNodeRenderer* GetTextNodeRenderer() const { return m_renderer; }
        /**
        * Get TextNodeRenderer.
        */
        TextNodeRenderer* GetTextNodeRenderer() { return m_renderer; }

        /**
         *  @return The axis-aligned layout rectangle in local coordinate space.
         */
        TextRendering::TextRect GetLayoutTextRectangle() const;

        /**
         *  @return The axis-aligned bounding rectangle in local coordinate space.
         */
        TextRendering::TextRect GetBoundingTextRectangle() const;


        // overrides Node2D::GetComputedBoundingRectangle
        FEATSTD_LINT_NEXT_EXPRESSION(1735, "Virtual function uses same default parameter as abstract base class.")
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible = false) const override;

        // overrides Node2D::GetBasePoint
        virtual void GetBasePoint(Vector2& basePoint) const override;

#if defined(CANDERA_LAYOUT_ENABLED)
        /**
        * @param[out] rectangle is the rectangle which is needed to show the text.
        * NOTE: If this method actually calculates the rectangle, then there is no TextNode2D - Layouter attached.
        *       and only the DefaultLayouter in general is used.\n
        * If there is one attached this method does not calculate the rectangle. It needs a layout call for this node.\n
        * This method is not able to retrieve layout information itself.
        */
        virtual void GetComputedLayoutRectangle(Rectangle& rectangle) const override;
#endif

        /**
        * Enables or disables whether the text node handles preprocessing/prerendering asynchronous.
        */
        void SetAsyncPreRenderEnabled(bool enabled);

        /**
        * @return text node is rendered asynchronously.
        */
        virtual bool IsAsyncPreRenderEnabled() const override { return m_isAsyncPreRenderEnabled; }

        /**
        * Attaches a validation helper to this object.
        * Default attachment is TextValidator (by TextValidationGroup)
        * Attaching a new one needs detaching first.
        */
        virtual bool AttachValidationHelper(FeatStd::ValidationHelperBase* validationHelper) override;
        /**
        * Detaches the validation helper.
        * Default attachment is TextValidator (by TextValidationGroup)
        * and has to be removed by calling the detach method of TextValidationGroup.
        */
        virtual bool DetachValidationHelper(FeatStd::ValidationHelperBase* validationHelper) override;

        /**
        * Triggers an update of layouting when its validator/validation group is true.
        */
        virtual void TriggerValidUpdate() override;

        /**
        * Sets an active glyph container and the dimensions which will be shown.
        */
        void SetPreprocessedText(TextRendering::GlyphDataContainer::SharedPointer activeGlyphDataContainer,
                                 const TextRendering::TextRect& layoutRectangle,
                                 const TextRendering::TextRect& boundingRectangle);

        /**
        * Sets a preprocessed text based on the render args calculated by the preprocessing/measurement step.
        */
        void SetPreprocessedText(TextRendering::TextRenderArgs::SharedPointer& resultArgs);

        /**
        * @return precaluclated text iterator.
        */
        const TextRendering::PreprocessingContext::Iterator GetTextIterator() const;

        /**
        * @return the validation user which overrides the behavior of TextNode2D
        */
        Candera::TextRendering::ITextValidationUser * GetOverrideValidationUser() const { return m_overrideValidationUser; }

        /**
        * Override default ValidationUser functionality of TextNode2D by an own validation user.
        * It is advised to also handle attach/detach as a minimum content equal to TextNode2D.
        * That refers to Attaching/Detaching the validator to the GetRenderController() - object.
        */
        void SetOverrideValidationUser(Candera::TextRendering::ITextValidationUser * val) { m_overrideValidationUser = val; }

        /**
        * Textnode2D is normally set to false unless behavior is overridden.
        * So the trigger update only occurs when the textnode actually changed its text.
        */
        virtual bool IsTriggerUpdateForced() override;


        /**
        * TextNode2D supports a custom generated glyph iterator.
        * After one is set, TextNode2D will not render text by itself anymore.
        * This flag checks if a custom glyph iterator were set or not.
        * @return true when a custom glyph iterator is set
        */
        bool IsCustomGlyphInformationUsed() const;

        /**
        * TextNode2D supports a custom generated glyph iterator.
        * This function is used to set the custom glyph iterator.
        * The lifetime of the custom glyph iterator is not handled by TextNode2D.
        * The user has to make sure that the iterator exists long enough to render the text.
        * It is possible to initialize the TextNode2D with an invalid iterator to avoid triggering 
        * a superfluous render process. 
        * Additionally, the bounding rectangles have to be set to handle layout/transformation operations correctly.
        * They are not required to be correct if these operations are handled externally.
        * @param customIterator defines a custom glyph iterator which shall be rendered instead of the internal one.
        * @param layoutRectangle defines the layout rectangle in which the text should be placed.
        * @param boundingRectangle defines the actual rectangle of the text.
        *
        */
        void SetCustomGlyphInformation(Candera::TextRendering::PreprocessingContext::Iterator customIterator,
                                    const TextRendering::TextRect& layoutRectangle,
                                    const TextRendering::TextRect& boundingRectangle);

        /**
        * TextNode2D supports a custom generated glyph iterator.
        * As soon as a custom glyph iterator is set, the internal render process becomes disabled.
        * To reenable the internal render process, this method has to be called.
        * It frees the internal information for the customized renderer - it does not free the iterator itself.
        * To reenable the custom render process, a new iterator has to be set.
        */
        void DisableCustomGlyphInformation();

        /**
        * Handles events which are delegated to TextNode2D.
        * It is currently used to invalidate the TextNode2D as soon as the style has changed.
        * @param event the event which shall be handled by TextNode2D.
        * @return FeatStd::EventResult::Proceed
        */
        virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& event) override;

    protected:
        TextNode2D();
        explicit TextNode2D(const TextNode2D& textNode2D);
        virtual void DisposeSelf();
        virtual bool UploadSelf() override;
        virtual bool UnloadSelf() override;

        //overrides RenderNode::Render
        virtual void Render(RenderTarget2D* renderTarget, const Matrix3x2& localTransform);
        /**
        * @return if the TextNode2D has an layouter attached (so its measurement can be (and therefore will be) done
        * in the layout process.
        * If not, the prerender currently preprocesses the text. Using the text (without own layouter) within the layouting process
        * is not advised. Because the Prerender is done afterwards and has to retrigger the Layouting behavior.
        */
        virtual bool IsLayouterAttached() const;
        /**
        * @return precaluclated text
        */
        TextRendering::GlyphDataContainer& GetActiveGlyphContainer();
        /**
        * @return precaluclated text (const)
        */
        const TextRendering::GlyphDataContainer& GetActiveGlyphContainer() const;

#ifdef FEATSTD_THREADSAFETY_ENABLED
        FEATSTD_EVENTLISTENER_STANDARD_THREADSAFETY;
#endif

    private:
        TextRendering::TextRect const& GetBoundingRectangle() const;
        TextRendering::TextRect const& GetLayoutRectangle() const;
        /**
        * Handler which handles all preprocessing and measurement calls.
        */
        Candera::TextRendering::Internal::TextRenderHandleController& GetRenderController() { return m_renderController; }

        /**
        * Sets current preprocessing/measurement state.
        */
        void SetPreparationState(TextNode2DRenderState::Enum state);

        /**
        * @return current preprocessing/measurement state.
        */
        TextNode2DRenderState::Enum  GetPreparationState() const;

        /**
        * Sets current preprocessing/measurement state to finished (based on texts in queue the right state will be selected).
        */
        void SetPreparationFinished();

        /**
        * Transforms text again to a new specified layouting rectangle.
        * Useful when there is e.g. no given dimension and the text is centered. Then the text is
        * calculated to the center of 0 - indefinite (max PixelPosition). In this case around x = 16k
        * The text has to be rearranged afterwards (the offset has to vanish)
        * The text renderer itself does not care about this fact (offset subtraction) but the layouter
        * needs this information.
        */
        void ArrangePreprocessedTextPosition(const TextRendering::TextRect& layoutingRectangle);

        virtual void PreRender() override;

#if defined(CANDERA_LAYOUT_ENABLED)
        virtual void OnBeforeLayouterSet(Layouter* newLayouter) override;
#endif

        /**
         * Renders a given Image2D, provided by the associated TextNodeRenderer.
         */
        void RenderImage(Image2D& image, RenderTarget2D* renderTarget, const Matrix3x2& localTransform);
        /**
        * Rechecks if a complex property has changed.
        * E.g. Style. When the style has changed internally - so not the property itself,
        * TextNode2D would not notice this. So it checks with this call whether the style has changed.
        */
        void CheckPropertiesValidity(bool invalidateScene = true);

        /**
        * Preprocesses a text; it is static and the name has changed compared to the old way of preprocessing text
        * because the async handler uses a function pointer to call this function in an asynchronous manner.
        */
        static bool PreprocessTextInternal(TextRendering::TextRenderArgs::SharedPointer renderArgs);
        /**
        * Invalidates the text - marks the renderstate to invalid.
        * When a Layouter or prerender (depends on whether a TextNode2D-Layouter is used)
        * call happens, and the state is invalid, then a new preprocessing step is triggered.
        */
        void InvalidateText(bool invalidateScene = true);

        /**
        * Current way to handle a TextNode2D when not Layouter is attached
        * @param forceSyncPreProcessOnly: don't allow ASync preprocessing, even if availlable (this ensures
        *        that the node info is updated after this call)
        */
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1735, "has default parameter [MISRA C++ Rule 8-3-1])")
        virtual void DefaultTextPlacement(bool forceSyncPreProcessOnly = false);
        /**
        * @return true when a text is successfully preprocessed and therefore waiting for rendering or
        * when the text is asynchronously prerendered and the result is done.
        * NOTE: Prerender asynchronously is not available in this version, so the second case will not happen.
        */
        bool IsPrerenderRequired() const;

        void ValidateBoundingInfo() const;

        FeatStd::String m_text; ///< Used for first step (update/layout)

        TextRendering::Internal::TextRenderHandleController m_renderController;

        CustomPreprocessingResult * m_customPreprocessingResult;

#ifdef FEATSTD_THREADSAFETY_ENABLED
        mutable FeatStd::Internal::CriticalSection m_textAsyncLock;
#endif
        TextRendering::TextRect m_boundingRectangle;
        TextRendering::TextRect m_layoutRectangle;

        TextRendering::GlyphDataContainer::SharedPointer m_activeContainer;
        TextRendering::SharedStyle::SharedPointer m_style;

        TextNodeRenderer* m_renderer;
        Candera::TextRendering::ITextValidationUser * m_overrideValidationUser; ///< 
        Candera::Globalization::CultureChangeListener * m_cultureChangeListener;

        UInt16 m_styleVersion; ///< Version of style - needed to recognize style changes
        TextNode2DRenderStates m_renderStates; ///< contains the render states of all render steps needed for the TextNode2D

        bool m_isAsyncPreRenderEnabled:1; ///< flag determines asynchronous or not
        bool m_isAsyncPreRenderChanged:1; ///< flag is set when async rendering changes to sync (so queued texts can be handled correctly)
        bool m_isLayouterUsesTextNodeNoLayout:1; ///< flag to guess if TextNode2D is used by another Layouter or not
        bool m_isPreviousRightToLeft:1; ///< flag to check whether the previous version were right to left culture. If it is not equal to current culture then the text has to be updated.

        CANDERA_UNIT_TEST_TESTCASE_FRIEND(TextNode2DTest, AsyncToSyncWithoutLayout);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(TextNode2DTest, AsyncToSyncWithLayout);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(TextNode2DTest, CultureChangeInvalidationCheck);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(TextNode2DTest, StylePropertyChanged);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(Globalization, CultureChangeTextNode2DRecognition);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(TextLayoutTest, Run);
    };
    /// @}  // end of Core2D

}   // namespace Candera

#endif  // CANDERA_TextNode2D_H
