//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SCENE2D_H)
    #define CANDERA_SCENE2D_H

#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/Engine2D/Core/RenderOrder2D.h>
#include <Candera/Engine2D/Core/TreeTraverser2D.h>
#include <Candera/System/Rtti/Rtti.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

CANDERA_UNIT_TEST_TESTCASE_DECLARATION(RenderOrder2DTest, Sorting)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(RenderOrder2DTest, RenderOrderRank)
CANDERA_UNIT_TEST_TESTCASE_DECLARATION(RenderOrder2DTest, RenderOrderRank2)

namespace Candera {
/** @addtogroup Core2D
    *  @{
    */

    /**
     * @brief   The class Scene2D represents a top level scene graph node. Multiple scenes are allowed.
     *          A scene can not be part of any other scene.
     */
    class Scene2D: public Group2D {
        FEATSTD_TYPEDEF_BASE(Group2D);

    public:
        /**
        * Creates an instance of this class.
        * Use Dispose() to delete the instance and possible children, if any.
        * @return Pointer to the created object
        */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::Group2D::Create, CANDERA_LINT_REASON_EXPLICITHIDING)
        static Scene2D* Create();

        /**
         * Destructor.
         */
        virtual ~Scene2D() override;

        virtual Scene2D* Clone() const override;

        FEATSTD_RTTI_DECLARATION();

#if defined(CANDERA_LAYOUT_ENABLED)
        /**
         *  \brief  Indicates that the scene layout is invalid.
         *  
         *  To trigger a scene to be laid out only once per Update/Render loop, all children of the scene (also widgets) can invalidate the scene layout by calling
         *  Scene::SetLayoutInvalid(). Before rendering is triggered by the application, Scene::ValidataLayout() can be called to trigger the layout if needed.
         */
        void SetLayoutInvalid() { SetLayoutValid(false); }

        /**
         *  Triggers a layout update in case the scene layout is invalid. See Scene::SetLayoutInvalid().
         */
        void ValidateLayout();
#endif

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Scene2D();
        Scene2D(const Scene2D& src);
        Scene2D& operator = (const Scene2D& rhs);

        /**
         * Disposes the instance of this class.
         */
        virtual void DisposeSelf() override;

    private:
        bool GetLayoutValid() const 
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            return m_isLayoutValid; 
        }

        void SetLayoutValid(bool val) 
        {
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&m_lock);
#endif
            m_isLayoutValid = val;
        }

        friend class Node2D;
        friend class RenderNode;
        friend class AddNodesToRenderOrderTraverser;
        friend class Renderer2D;

        CANDERA_UNIT_TEST_TESTCASE_FRIEND(RenderOrder2DTest, Sorting);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(RenderOrder2DTest, RenderOrderRank);
        CANDERA_UNIT_TEST_TESTCASE_FRIEND(RenderOrder2DTest, RenderOrderRank2);

        bool m_isLayoutValid : 1;
        RenderOrder2D m_renderOrder;
#ifdef FEATSTD_THREADSAFETY_ENABLED
        mutable FeatStd::Internal::CriticalSection m_lock;
#endif


        RenderOrder2D& GetRenderOrder() { return m_renderOrder; }
    };
 /** @} */ // end of Core2D


} // namespace Candera
#endif  // CANDERA_SCENE2D_H
